import java.awt.*;
import java.awt.image.*;
import java.lang.*;		// includes math libraries
import java.lang.String.*;
import java.net.*;
import java.util.*;
import javax.swing.*;

/**
  * Class <CODE>BLKMComponent</CODE>.
  *
  * @author Robert Marlowe
  * @author Mike Cammarano
  * @author Charlie So
  *
  * BLKM implementation of the <CODE>CircuitComponent</CODE> 
  * interface.
  *
  * Class for 16-bit Table to Table Function. This is utilized by 
  * <CODE>BLKMComponent</CODE>.
  *
  *  The BLKM (block move) instruction copies the entire contents of a
  *	source table to a destination table in one scan.
  *
  *	Inputs - 
  *		BLKM has one control input (to the top node).  This 
  *		input initiates the DX move operation.
  *	Outputs- 
  *		BLKM produces one output (from the top node), which 
  *		echoes the state of the top input.
  *	Top Node Content -
  *		The top node specifies the source table that will have
  *		its contents copied in the block move.
  *	Middle Node Content -
  *		The middle node specifies the destination table where 
  *		the contents of the source table will be copied in the 
  *		block move.
  *	Botton Node Content -
  *		The integer value entered in the bottom node specifies 
  *		the table size - i.e., the number of registers or 
  *		16-bit words - for both the source and destination
  *		tables; they are of equal length.  The table length 
  *		can range from 1 ... 1024.
  *
  */
public class BLKMComponent implements CircuitComponent {
  /**
    * The register used to hold this component's top node value.
    */
  private int sourceLine;
  /**
    * The register used to hold this component's middle node value.
    */
  private int destinationLine;
  /**
    * The register used to hold this component's bottom node value.
    */
  private int lengthLine;
  /**
    * Flag indicating whether component should be highlighted. 
    */
  private boolean powered;
  /**
    * Image used to draw this component in a network when it is not 
    * powered.
    */
  private static Image pic;
  /**
    * Image used to draw this component in a network when it is  
    * powered.
    */
  private static Image picLit;
  /**
    * Cursor image used for this component during drag and drop.
    */
  private static Image cursorImage;
  /**
    * Icon used for this component in the component bin.
    */
  private static ImageIcon icon;
  /**
    * ToolTip text used for this component.
    */
  private static String tip = "BLKM";
  /**
    * The simulations internal name for this component.
    */
  private static String name = "BLKM";

  static {
      // This static initializer loads all the image files needed
      URL myURL;
      ImageIcon myIcon;
      myURL = ClassLoader.getSystemResource("BigBLKM.gif");
    	myIcon = new ImageIcon(myURL);
    	pic = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("BigBLKMOn.gif");
        myIcon = new ImageIcon(myURL);
        picLit = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("CursorBLKM.gif");
        myIcon = new ImageIcon(myURL);
        cursorImage = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("ToolBLKM.gif");
        icon = new ImageIcon(myURL);
  }

  /**
    * Construct a 3-rung BLKM function using default source, 
    * destination, and length values.
    */
  public BLKMComponent() {
    sourceLine      = 0;  // initial registers used
    destinationLine = 1;
    lengthLine      = 2;  // SimMemory auto initializes regs
  }

  /**
    * @return 3, the number of ladder logic rungs occupied by this 
    * component.
    */
  public int size() {
    return 3;
  }

  /**
    * Performs one evaluation of this component using the specified
    * input logic levels. If the 1st rung of the BLKM is logic-low,
    * then the BLKM resets. Otherwise, the operation commences and 
    * the current state of the first rung is passed to the output.  
    *
    * The top rung of the input (b[0]) is read and the top rung of 
    *	the output (boolean[0]) is affected by the operation.
    *
    * @param b - array containing input logic levels of the rungs
    *            occupied by this component. 
    *
    * @return the output logic levels from this component.
    */
  public boolean[] step(boolean[] b) {
     powered = b[0];
     boolean tempArr[];
     tempArr = new boolean[] {false, false, false};
     int length = SimMemory.GetReg(lengthLine); // table length
     if(b[0]){			// if top rung is ON
	tempArr[0] = true;	// pass input state to out top rung
	for(int i=0; i < length; i++) {
	    int source 	    = SimMemory.GetReg(sourceLine+i);
	    int destination = SimMemory.GetReg(destinationLine+i);
	    destination     = source;
	    SimMemory.SetReg(destinationLine+i, destination);
	}
     }
     return tempArr;			
  }					

  /**
    * Displays a modal dialog box to allow the user to edit the
    *  	parameters of this component, such as which registers are 
    * 	used. This method will initialize the state of the 
    * 	existing instance of <CODE>Node3Dialog</CODE> in 
    * 	<CODE>Master</CODE>, show it, and then update source and 
    *   destination registers to be used in the operation of the 
    *	AND based on the user's selections from the dialog.
    */
  public void edit() {
    Node3Dialog a = Master.DialogNode3;
    a.setTitle("BLKM Register Setup");
    for(int i=destinationLine; i < (SimMemory.GetReg(lengthLine)); i++){
    	SimMemory.deleteUser(i, this);
    }
    a.setState(sourceLine,destinationLine, lengthLine, this);
    a.setVisible(true);
    sourceLine 		= a.getValue1Line();
    destinationLine 	= a.getValue2Line();
    lengthLine 		= a.getResultLine();
    for(int i=destinationLine; i < (SimMemory.GetReg(lengthLine)); i++){
    	SimMemory.addUser(i, this);
    }
  }

  /**
    * Clears any registers marked as used by this component prior to
    * its deletion.
    */
  public void clear() {
    for(int i=destinationLine; i < (SimMemory.GetReg(lengthLine)); i++){
    	SimMemory.deleteUser(i, this);
    }
  }

  /**
    * Resets this component, and initializes values
    * in <CODE>SimMemory</CODE>.
    */
  public void reset() {
	// Not needed in this component
  }

  /**
    * Returns the text to be used as a tool tip when this component
    * is placed in the component bin.
    *
    * @return the string to display.
    */
  public String getTip() {
    return tip;
  }

  /**
    * Returns the name of this component to be used internally in 
    * saved files and in drag and drop operations.
    *
    * @return the string to use.
    */
  public String getName() {
    return name;
  }

  /**
    * Returns the Icon to be used for this component in the component 
    * bin.
    *
    * @return the icon.
    */
  public ImageIcon getToolIcon() {
    return icon;
  }

  /**
    * Returns a one line string representing the state of this 
    * component for use in a save file. The string will be of the 
    * form: <BR>
    * BLKM 
    * <I>source-register destination-register length-register
    * </I>.
    *
    * @return the parameter string.
    */
  public String getSaveData() {
    return name + " " + sourceLine + " " + destinationLine 
		+ " " + lengthLine;
  }

  /**
    * Returns whether this component must be the rightmost component 
    * on any rung.  For example, an output coil terminates a rung; 
    * a switch does not.
    *
    * @return <CODE>False</CODE> since BLKMs don't terminate rungs.
    */
  public boolean terminatesRung() {
    return false;
  }

  /**
    * Returns the image to use as a mouse cursor for this component 
    * during a drag and drop operation.
    *
    * @return the cursor image.
    */
  public Image getImage() {
    return cursorImage;
  }

  /**
    * Returns a new instance of this component created using the 
    * default constructor.
    * This method is used by the <CODE>getComponent</CODE> factory 
    * method of <CODE>Elements</CODE>.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance() {
    BLKMComponent x = new BLKMComponent();
    for(int i=x.destinationLine; i < (SimMemory.GetReg(x.lengthLine)); i++){
    	SimMemory.addUser(i, x);
    }
    return x;
  }

  /**
    * Returns a new instance of this component, using the specified 
    * initial parameters.  This method is used by the 
    * <CODE>getComponent</CODE> factory method of <CODE>Elements</CODE> 
    * to instantiate new components during a file open.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance(int[] a) {	
    BLKMComponent x = new BLKMComponent();
    x.sourceLine 	= a[0];
    x.destinationLine 	= a[1];
    x.lengthLine    	= a[2];
    for(int i=x.destinationLine; i < (SimMemory.GetReg(x.lengthLine)); i++){
    	SimMemory.addUser(i, x);
    }
    return x;
  }

  /**
    * Draws this component and any associated parameters at the 
    * specified coordinates in the graphics context. The source 
    * register is written in the top third of the BLKM, the 
    * destination register is written in the middle third of the 
    * BLKM and the length register is written in the bottom third 
    * of the BLKM.
    *
    * @param g  the graphics context in which to draw.
    * @param x  the x-coordinate of the northwest corner.
    * @param y  the y-coordinate of the northwest corner.
    * @param o  the image observer to be notified of image drawing.
    */
  public void drawComponent(Graphics g,int x,int y,ImageObserver o){
    if(Master.isSimActive && powered) {
      // Draw a powered component
      g.drawImage(picLit, x, y, o);
      g.setColor(Network.darkGreen);
    } 
    else {
      // Draw a non-powered component
      g.drawImage(pic, x, y, o);
    }

    // Draw Register assignments onto component at rung
    String text;
    
    // Draw the source register in the top third of the component
    text = "     " + Integer.toString(sourceLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+16);

    // Draw the destination register in the middle third of the comp
    text = "     " + Integer.toString(destinationLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+40);

    // Draw the length register in the bottom third of the component
    text = "     " + Integer.toString(lengthLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+64);

    g.setColor(Color.black);	// in case the color got set to 
  } 				// dark green set back to black
}




/*  To compile and run this Class
	first  - make sure that the class's name is also the file name 
		 -- observe case
	second - at the command prompt in the working directory of the 
		 SDK compile the class.  
			EX. L:\work>N:jdk1.2.1\bin\javac *.java
		 In this case all file *.java will be linked, 
		 therefore take care that they are related or delete 
		 unneeded files.
	third  - run the Class by 
			EX. L:\WORK>N:jdk1.2.1\bin\java figure1511
		 where figure1511 is both the name of the class & file.

    For creating links to, for example, methods ...

	{@link #reset() <CODE>reset</CODE>.}
*/
		