import java.awt.*;
import java.awt.image.*;
import java.lang.*;		// includes math libraries
import java.lang.String.*;
import java.net.*;
import java.util.*;
import javax.swing.*;

/**
  * Class <CODE>BLKTComponent</CODE>.
  *
  * @author Robert Marlowe
  * @author Mike Cammarano
  * @author Charlie So
  *
  * BLKT implementation of the <CODE>CircuitComponent</CODE> 
  * interface.
  *
  * Class for 16-bit Block to Table Function. This is utilized by 
  * <CODE>BLKTComponent</CODE>.
  *
  *  The BLKT instruction combines the functions of R->T and BLKM in a
  *	single instruction.  In one scan, it can copy data from a source 
  *	block to a destination block in a table.  The source block is of 
  *	a fixed length.  The block within the table is of the same 
  *	length, but the overall length of the table is limited only by
  *	the number of registers in you system configuration.
  *
  *	Inputs - 
  *		BLKT has three control inputs.  The input to the top 
  *		node initiates the DX move operation.  The inputs to the 
  *		middle and bottom node can be used to control the 
  *		pointer so that source data is not copied into registers 
  *		that are needed for other purposes in the logic program.  
  *
  *		When the input to the middle node is ON, the value in the 
  *		pointer register is frozen while the BLKT operation 
  *		continues.  This causes the BLKT operation to copy source 
  *		data into the first block of registers in the destination 
  *		table.
  *		
  *		When the input to the bottom node is ON, the value in 
  *		the pointer register is reset to zero.  This causes the 
  *		BLKT operation to copy source data into the first block
  *		of registers in the destination table.
  *	Outputs- 
  *		BLKT can produce two possible outputs.  When the move 
  *		is successful, power is passed to the output from the 
  *		top node.  If an error occurs in the operation, power 
  *		is passed to the output from the middle node.
  *	Top Node Content -
  *		The register entered in the top nide is the first 
  *		holding register in the source block - i.e., the block 
  *		of contiguous registers whose content will be copied to 
  *		a block of registers in the destination table.
  *	Middle Node Content -
  *		The register entered in the middle node is the pointer 
  *		to the destination table.  The first register in the 
  *		destination table is the next contiguous register after 
  *		the pointer - e.g., if the pointer register is 40107, 
  *		then the first register in the destination table is 
  *		40108.
  *
  *		Note:  The destination table is segmented into a series
  * 			of register blocks, each of which is the same 
  *			length as the source block.  Therefore, the size 
  *			of the destination table is a multiple of the 
  *			length of the source block, byt its overall size 
  *			is not specifically defined in the instruction.  
  *			If left uncontrolled, the destination table 
  *			could consue all the registers available.
  *
  *		The value stored in the pointer register indicates where
  *		in the destination table the source data will begin to 
  *		be copied.  This value specifies the block number within
  *		the destination table.
  *	Botton Node Content -
  *		The integer value entered in the bottom node specifies 
  *		block length - i.e., the number of registers - of the 
  *		source block (and the destination block).  The valid 
  *		range can be from 1 ... 1024.
  *
  */
public class BLKTComponent implements CircuitComponent {
  /**
    * The register used to hold this component's top node value.
    */
  private int sourceLine;
  /**
    * The register used to hold this component's middle node value.
    */
  private int pointerLine;
  /**
    * The register used to hold this component's bottom node value.
    */
  private int lengthLine;
  /**
    * Flag indicating whether component should be highlighted. 
    */
  private boolean powered;
  /**
    * Image used to draw this component in a network when it is not 
    * powered.
    */
  private static Image pic;
  /**
    * Image used to draw this component in a network when it is  
    * powered.
    */
  private static Image picLit;
  /**
    * Cursor image used for this component during drag and drop.
    */
  private static Image cursorImage;
  /**
    * Icon used for this component in the component bin.
    */
  private static ImageIcon icon;
  /**
    * ToolTip text used for this component.
    */
  private static String tip = "BLKT";
  /**
    * The simulations internal name for this component.
    */
  private static String name = "BLKT";

  static {
      // This static initializer loads all the image files needed
      URL myURL;
      ImageIcon myIcon;
      myURL = ClassLoader.getSystemResource("BigBLKT.gif");
    	myIcon = new ImageIcon(myURL);
    	pic = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("BigBLKTOn.gif");
        myIcon = new ImageIcon(myURL);
        picLit = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("CursorBLKT.gif");
        myIcon = new ImageIcon(myURL);
        cursorImage = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("ToolBLKT.gif");
        icon = new ImageIcon(myURL);
  }

  /**
    * Construct a 3-rung BLKT function using default source, 
    * pointer, and length values.
    */
  public BLKTComponent() {
    sourceLine  = 0;  // initial registers used
    pointerLine = 1;
    lengthLine  = 2;  // SimMemory auto initializes regs
  }

  /**
    * @return 3, the number of ladder logic rungs occupied by this 
    * component.
    */
  public int size() {
    return 3;
  }

  /**
    * Performs one evaluation of this component using the specified
    * input logic levels. If the 1st rung of the BLKT is logic-low,
    * then the BLKT resets. Otherwise, the operation commences and 
    * the current state of the first rung is passed to the output.  
    *
    * The top, middle, and bottom rungs of the input (b[0],b[1],b[2]) 
    * are read and the top and middle rungs of the output 
    * (boolean[0], boolean[1]) is affected by the operation.
    *
    * @param b - array containing input logic levels of the rungs
    *            occupied by this component. 
    *
    * @return the output logic levels from this component.
    */
  public boolean[] step(boolean[] b) {
     powered = b[0];
     boolean tempArr[];
     tempArr = new boolean[] {false, false, false};
     if(b[2]){				  // if bottom rung is ON
        SimMemory.SetReg(pointerLine, 0); // reset the pointer value
     }					  // to zero
     int length = SimMemory.GetReg(lengthLine); // # registers in source block
     int block	= SimMemory.GetReg(pointerLine);// block in destination table 
     int destinationReg = pointerLine+1+length+length*block;
     // ****************** Initiate Move
     if(b[0] && destinationReg<1024){			// if top rung is ON
	tempArr[0] = true;	// pass input state to out top rung
	for(int i=0; i<length; i++) { // copy source into destination block 
	   int data = SimMemory.GetReg(sourceLine+i);
	   SimMemory.SetReg(pointerLine+1+i+(length*block), data);
	}					     
     }
     else if(b[0] && destinationReg>=1024){
	tempArr[1] = true;
     }
     // ****************** End Initiate Move
     if(!b[1]){
	SimMemory.SetReg(pointerLine, block+1);
     }
     return tempArr;			
  }					

  /**
    * Displays a modal dialog box to allow the user to edit the
    *  	parameters of this component, such as which registers are 
    * 	used. This method will initialize the state of the 
    * 	existing instance of <CODE>Node3Dialog</CODE> in 
    * 	<CODE>Master</CODE>, show it, and then update source and 
    *   destination registers to be used in the operation of the 
    *	AND based on the user's selections from the dialog.
    */
  public void edit() {
    Node3Dialog a = Master.DialogNode3;
    a.setTitle("BLKT Register Setup");
    for(int i=pointerLine; i < (SimMemory.GetReg(lengthLine)); i++){
    	SimMemory.deleteUser(i, this);
    }
    a.setState(sourceLine,pointerLine, lengthLine, this);
    a.setVisible(true);
    sourceLine 		= a.getValue1Line();
    pointerLine 	= a.getValue2Line();
    lengthLine 		= a.getResultLine();
    for(int i=pointerLine; i < (SimMemory.GetReg(lengthLine)); i++){
    	SimMemory.addUser(i, this);
    }
  }

  /**
    * Clears any registers marked as used by this component prior to
    * its deletion.
    */
  public void clear() {
    for(int i=pointerLine; i < (SimMemory.GetReg(lengthLine)); i++){
    	SimMemory.deleteUser(i, this);
    }
  }

  /**
    * Resets this component, and initializes values
    * in <CODE>SimMemory</CODE>.
    */
  public void reset() {
	// Not needed in this component
  }

  /**
    * Returns the text to be used as a tool tip when this component
    * is placed in the component bin.
    *
    * @return the string to display.
    */
  public String getTip() {
    return tip;
  }

  /**
    * Returns the name of this component to be used internally in 
    * saved files and in drag and drop operations.
    *
    * @return the string to use.
    */
  public String getName() {
    return name;
  }

  /**
    * Returns the Icon to be used for this component in the component 
    * bin.
    *
    * @return the icon.
    */
  public ImageIcon getToolIcon() {
    return icon;
  }

  /**
    * Returns a one line string representing the state of this 
    * component for use in a save file. The string will be of the 
    * form: <BR>
    * BLKT 
    * <I>source-register pointer-register length-register
    * </I>.
    *
    * @return the parameter string.
    */
  public String getSaveData() {
    return name + " " + sourceLine + " " + pointerLine 
		+ " " + lengthLine;
  }

  /**
    * Returns whether this component must be the rightmost component 
    * on any rung.  For example, an output coil terminates a rung; 
    * a switch does not.
    *
    * @return <CODE>False</CODE> since BLKTs don't terminate rungs.
    */
  public boolean terminatesRung() {
    return false;
  }

  /**
    * Returns the image to use as a mouse cursor for this component 
    * during a drag and drop operation.
    *
    * @return the cursor image.
    */
  public Image getImage() {
    return cursorImage;
  }

  /**
    * Returns a new instance of this component created using the 
    * default constructor.
    * This method is used by the <CODE>getComponent</CODE> factory 
    * method of <CODE>Elements</CODE>.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance() {
    BLKTComponent x = new BLKTComponent();
    for(int i=x.pointerLine; i < (SimMemory.GetReg(x.lengthLine)); i++){
    	SimMemory.addUser(i, x);
    }
    return x;
  }

  /**
    * Returns a new instance of this component, using the specified 
    * initial parameters.  This method is used by the 
    * <CODE>getComponent</CODE> factory method of <CODE>Elements</CODE> 
    * to instantiate new components during a file open.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance(int[] a) {	
    BLKTComponent x = new BLKTComponent();
    x.sourceLine 	= a[0];
    x.pointerLine 	= a[1];
    x.lengthLine    	= a[2];
    for(int i=x.pointerLine; i < (SimMemory.GetReg(x.lengthLine)); i++){
    	SimMemory.addUser(i, x);
    }
    return x;
  }

  /**
    * Draws this component and any associated parameters at the 
    * specified coordinates in the graphics context. The source 
    * register is written in the top third of the BLKT, the 
    * pointer register is written in the middle third of the 
    * BLKT and the length register is written in the bottom third 
    * of the BLKT.
    *
    * @param g  the graphics context in which to draw.
    * @param x  the x-coordinate of the northwest corner.
    * @param y  the y-coordinate of the northwest corner.
    * @param o  the image observer to be notified of image drawing.
    */
  public void drawComponent(Graphics g,int x,int y,ImageObserver o){
    if(Master.isSimActive && powered) {
      // Draw a powered component
      g.drawImage(picLit, x, y, o);
      g.setColor(Network.darkGreen);
    } 
    else {
      // Draw a non-powered component
      g.drawImage(pic, x, y, o);
    }

    // Draw Register assignments onto component at rung
    String text;
    
    // Draw the source register in the top third of the component
    text = "     " + Integer.toString(sourceLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+16);

    // Draw the destination register in the middle third of the comp
    text = "     " + Integer.toString(pointerLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+40);

    // Draw the length register in the bottom third of the component
    text = "     " + Integer.toString(lengthLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+64);

    g.setColor(Color.black);	// in case the color got set to 
  } 				// dark green set back to black
}




/*  To compile and run this Class
	first  - make sure that the class's name is also the file name 
		 -- observe case
	second - at the command prompt in the working directory of the 
		 SDK compile the class.  
			EX. L:\work>N:jdk1.2.1\bin\javac *.java
		 In this case all file *.java will be linked, 
		 therefore take care that they are related or delete 
		 unneeded files.
	third  - run the Class by 
			EX. L:\WORK>N:jdk1.2.1\bin\java figure1511
		 where figure1511 is both the name of the class & file.

    For creating links to, for example, methods ...

	{@link #reset() <CODE>reset</CODE>.}
*/
		