import java.awt.*;
import java.awt.image.*;
import java.lang.*;		// includes math libraries
import java.lang.String.*;
import java.net.*;
import java.util.*;
import javax.swing.*;

/**
  * Class <CODE>BROTComponent</CODE>.
  *
  * @author Mike Cammarano
  * @author Charlie So
  * @author Robert Marlowe
  *
  * BROT implementation of the <CODE>CircuitComponent</CODE> 
  * interface.
  *
  * Class for 16-bit BROT Function. This is utilized by 
  * <CODE>BROTComponent</CODE>.
  *
  *  The BROT (bit rotate) instruction shifts the bit pattern in a 
  *	source matrix, then posts the shifted bit pattern in a 
  *	destination matrix.  The bit pattern shifts left of right by 
  *	one position per scan.
  *
  *	Inputs - 
  *		BROT ahs three control inputs.  The inputs determine
  *		the way the bits will be shifted.
  *	Output- 
  *		BROT can produce two possible outputs (from the top
  *		and middle nodes).  The output from the middle node 
  *		indicates the sense of the bit that exits the source 
  *		matrix (the leftmost or rightmost bit) as a result of 
  *		the shift.
  *	Top Node Content -
  *		The entry in the top node is the first regerence in 
  *		the source matrix - i.e., in the matrix that will have
  *		its bit pattern shifted.
  *	Middle Node Content -
  *		The entry in the middle node is the first reference 
  *		in the destination matrix - i.e., in the matrix that
  *		shows the shifted bit pattern.
  *	Botton Node Content -
  *		The integer entered in the bottom node specifies the 
  *		matrix length - i.e., the number of registers or 
  *		16-bit words in the two matrixes.  (The source matrix
  *		and the destination matrix have the same length).  The 
  *		matrix length can be in the range 1 ... 1024.  Ex. A 
  *		length of 100 indicates 1600 bit locations.
  *
  */
public class BROTComponent implements CircuitComponent {
  /**
    * The register used to hold this component's top node value.
    */
  private int sourceLine;
  /**
    * The register used to hold this component's middle node value.
    */
  private int destinationLine;
  /**
    * The register used to hold this component's bottom node value.
    */
  private int lengthLine;
  /**
    * Flag indicating whether component should be highlighted. 
    */
  private boolean powered;
  /**
    * Image used to draw this component in a network when it is not 
    * powered.
    */
  private static Image pic;
  /**
    * Image used to draw this component in a network when it is  
    * powered.
    */
  private static Image picLit;
  /**
    * Cursor image used for this component during drag and drop.
    */
  private static Image cursorImage;
  /**
    * Icon used for this component in the component bin.
    */
  private static ImageIcon icon;
  /**
    * ToolTip text used for this component.
    */
  private static String tip = "BROT";
  /**
    * The simulations internal name for this component.
    */
  private static String name = "BROT";

  static {
      // This static initializer loads all the image files needed
      URL myURL;
      ImageIcon myIcon;
      myURL = ClassLoader.getSystemResource("BigBROT.gif");
    	myIcon = new ImageIcon(myURL);
    	pic = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("BigBROTOn.gif");
        myIcon = new ImageIcon(myURL);
        picLit = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("CursorBROT.gif");
        myIcon = new ImageIcon(myURL);
        cursorImage = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("ToolBROT.gif");
        icon = new ImageIcon(myURL);
  }

  /**
    * Construct a 3-rung BROT function using default source, 
    * destination, and iteration values.
    */
  public BROTComponent() {
    sourceLine 	    = 0;  // initial registers used
    destinationLine = 1;
    lengthLine      = 2;  // SimMemory auto initializes regs
  }

  /**
    * @return 3, the number of ladder logic rungs occupied by this 
    * component.
    */
  public int size() {
    return 3;
  }

  /**
    * Performs one evaluation of this component using the specified
    * input logic levels. If the 1st rung of the BROT is logic-low,
    * then the BROT resets. Otherwise, the operation commences and 
    * the current state of the first rung is passed to the output.  
    *
    * The top, middle, and bottom rungs	of the input (b[0], b[1], b[2]) 
    * are read and the top and middle rungs of the output 
    * (boolean[0] and boolean[1]) are affected by the operation.
    *
    * @param b - array containing input logic levels of the rungs
    *            occupied by this component.
    *
    * @return the output logic levels from this component.
    */
  public boolean[] step(boolean[] b) {
     powered = b[0];
     boolean tempArr[];
     tempArr = new boolean[] {false, false, false};// initialize output array
     if(b[0]){	// if ON
        int tempLength = SimMemory.GetReg(lengthLine);// get matrix length
	tempArr[0] = true;	// pass input state to top output
	int exitBit = 0;
	int bit = 0;
	// ****************************  shift right (exit LSB)   *******************************
	if(b[1]){	
	   exitBit = SimMemory.GetReg(sourceLine+tempLength-1) & 1; // get first word and mask off all but LSB
	   tempArr[1] = (exitBit==1);	 			    // reflect state of exit bit to 2nd output rung  				    				
	   if(!b[2]){						    // if bottom input rung is set
		exitBit = 0;					    // exit bit falls out of the destination matrix
	   }
	   for(int i=0; i < tempLength; i++) {
	      int tempSource = SimMemory.GetReg(sourceLine+i);	    // get source reg contents
	      int tempDestination;	
	      if(i == 0){				    	    // if current word is last word
		 bit = exitBit;				    	    // bit to be shifted in is the exit bit 
		 bit = bit << 15;		    	    	    // shift over to MSB position
	      }
	      else{						    // if current word is not last word
	      	 bit = SimMemory.GetReg(sourceLine+i-1);            // Get next word
	      	 bit = bit & 1;			    	    	    // mask off all but LSB of next word
		 bit = bit << 15;		    	    	    // shift over to MSB position
	      }
	      tempDestination = tempSource >> 1;		    // shift once to R
	      tempDestination = tempDestination | bit;	    	    // OR in bit
	      tempDestination = tempDestination & 65535;  	    // remove upper 16 of 32 bits
	      SimMemory.SetReg(destinationLine+i, tempDestination); // save result
	   }
        }
	// ****************************  shift left (exit MSB)   *******************************
	else if(!b[1]){						    		
	   exitBit = SimMemory.GetReg(sourceLine) >> 15;	    // get word and shift MSB to LSB position
	   exitBit = exitBit & 1;				    // mask off all but LSB	
	   tempArr[1] = (exitBit==1);	 			    // reflect state of exit bit to 2nd output rung	
	   if(!b[2]){						    // if bottom input rung is set
		exitBit = 0;					    // exit bit falls out of the destination matrix
	   }
	   for(int i=0; i < tempLength; i++) {
	      int tempSource = SimMemory.GetReg(sourceLine+i);	    // get source reg contents
	      int tempDestination;	
	      if(i == tempLength-1){				    // if current word is last word
		 bit = exitBit;				    	    // bit to be shifted in is the exit bit 
	      }
	      else{						    // if current word is not last word
	      	 bit = SimMemory.GetReg(sourceLine+i+1) >> 15;      // Get next word and shift MSB to LSB position
	      	 bit = bit & 1;			    	    	    // mask off all but LSB of next word
	      }
	      tempDestination = tempSource << 1;		    // shift once to L
	      tempDestination = tempDestination | bit;	    	    // OR in Last bit
	      tempDestination = tempDestination & 65535;  	    // remove upper 16 of 32 bits
	      SimMemory.SetReg(destinationLine+i, tempDestination); // save result
	   }
        }
     }
     return tempArr;	// write outputs	
  }					

  /**
    * Displays a modal dialog box to allow the user to edit the
    *  	parameters of this component, such as which registers are 
    * 	used. This method will initialize the state of the 
    * 	existing instance of <CODE>Node3Dialog</CODE> in 
    * 	<CODE>Master</CODE>, show it, and then update source and 
    *   destination registers to be used in the operation of the 
    *	AND based on the user's selections from the dialog.
    */
  public void edit() {
    Node3Dialog a = Master.DialogNode3;
    a.setTitle("BROT Register Setup");
    for(int i=destinationLine; i < (SimMemory.GetReg(lengthLine)); i++){
    	SimMemory.deleteUser(i, this);
    }
    a.setState(sourceLine, destinationLine, lengthLine, this);
    a.setVisible(true);
    sourceLine 		= a.getValue1Line();
    destinationLine 	= a.getValue2Line();
    lengthLine 		= a.getResultLine();
    for(int i=destinationLine; i < (SimMemory.GetReg(lengthLine)); i++){
    	SimMemory.addUser(i, this);
    }
  }

  /**
    * Clears any registers marked as used by this component prior to
    * its deletion.
    */
  public void clear() {
    for(int i=destinationLine; i < (SimMemory.GetReg(lengthLine)); i++){
    	SimMemory.deleteUser(i, this);
    }
  }

  /**
    * Resets this component, and initializes values
    * in <CODE>SimMemory</CODE>.
    */
  public void reset() {
	// Not needed in this component
  }

  /**
    * Returns the text to be used as a tool tip when this component
    * is placed in the component bin.
    *
    * @return the string to display.
    */
  public String getTip() {
    return tip;
  }

  /**
    * Returns the name of this component to be used internally in 
    * saved files and in drag and drop operations.
    *
    * @return the string to use.
    */
  public String getName() {
    return name;
  }

  /**
    * Returns the Icon to be used for this component in the component 
    * bin.
    *
    * @return the icon.
    */
  public ImageIcon getToolIcon() {
    return icon;
  }

  /**
    * Returns a one line string representing the state of this 
    * component for use in a save file. The string will be of the 
    * form: <BR>
    * BROT 
    * <I>source-register destination-register length-register
    * </I>.
    *
    * @return the parameter string.
    */
  public String getSaveData() {
    return name + " " + sourceLine + " " + destinationLine 
		+ " " + lengthLine;
  }

  /**
    * Returns whether this component must be the rightmost component 
    * on any rung.  For example, an output coil terminates a rung; 
    * a switch does not.
    *
    * @return <CODE>False</CODE> since BROTs don't terminate rungs.
    */
  public boolean terminatesRung() {
    return false;
  }

  /**
    * Returns the image to use as a mouse cursor for this component 
    * during a drag and drop operation.
    *
    * @return the cursor image.
    */
  public Image getImage() {
    return cursorImage;
  }

  /**
    * Returns a new instance of this component created using the 
    * default constructor.
    * This method is used by the <CODE>getComponent</CODE> factory 
    * method of <CODE>Elements</CODE>.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance() {
    BROTComponent x = new BROTComponent();
    for(int i=x.destinationLine; i < (SimMemory.GetReg(x.lengthLine)); i++){
    	SimMemory.addUser(i, x);
    }
    return x;
  }

  /**
    * Returns a new instance of this component, using the specified 
    * initial parameters.  This method is used by the 
    * <CODE>getComponent</CODE> factory method of <CODE>Elements</CODE> 
    * to instantiate new components during a file open.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance(int[] a) {	
    BROTComponent x = new BROTComponent();
    x.sourceLine 	= a[0];
    x.destinationLine 	= a[1];
    x.lengthLine    	= a[2];
    for(int i=x.destinationLine; i < (SimMemory.GetReg(x.lengthLine)); i++){
    	SimMemory.addUser(i, x);
    }
    return x;
  }

  /**
    * Draws this component and any associated parameters at the 
    * specified coordinates in the graphics context. The source 
    * register is written in the top third of the BROT, the 
    * destination register is written in the middle third of the BROT 
    * and the length register is written in the bottom third of the 
    * BROT.
    *
    * @param g  the graphics context in which to draw.
    * @param x  the x-coordinate of the northwest corner.
    * @param y  the y-coordinate of the northwest corner.
    * @param o  the image observer to be notified of image drawing.
    */
  public void drawComponent(Graphics g,int x,int y,ImageObserver o){
    if(Master.isSimActive && powered) {
      // Draw a powered component
      g.drawImage(picLit, x, y, o);
      g.setColor(Network.darkGreen);
    } 
    else {
      // Draw a non-powered component
      g.drawImage(pic, x, y, o);
    }

    // Draw Register assignments onto component at rung
    String text;
    
    // Draw the source register in the top third of the component
    text = "     " + Integer.toString(sourceLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+16);

    // Draw the destination register in the middle third of the comp
    text = "     " + Integer.toString(destinationLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+40);

    // Draw the length register in the bottom third of the component
    text = "     " + Integer.toString(lengthLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+64);

    g.setColor(Color.black);	// in case the color got set to 
  } 				// dark green set back to black
}




/*  To compile and run this Class
	first  - make sure that the class's name is also the file name 
		 -- observe case
	second - at the command prompt in the working directory of the 
		 SDK compile the class.  
			EX. L:\work>N:jdk1.2.1\bin\javac *.java
		 In this case all file *.java will be linked, 
		 therefore take care that they are related or delete 
		 unneeded files.
	third  - run the Class by 
			EX. L:\WORK>N:jdk1.2.1\bin\java figure1511
		 where figure1511 is both the name of the class & file.

    For creating links to, for example, methods ...

	{@link #reset() <CODE>reset</CODE>.}
*/
		