import java.awt.*;
import java.awt.image.*;
import java.lang.*;		// includes math libraries
import java.lang.String.*;
import java.net.*;
import java.util.*;
import javax.swing.*;

/**
  * Class <CODE>CMPRComponent</CODE>.
  *
  * @author Mike Cammarano
  * @author Charlie So
  * @author Robert Marlowe
  *
  * CMPR implementation of the <CODE>CircuitComponent</CODE> 
  * interface.
  *
  * Class for 16-bit CMPR Function. This is utilized by 
  * <CODE>CMPRComponent</CODE>.
  *
  *  The CMPR instruction compares the bit pattern in matrix a against
  *	the bit pattern matrix b for miscompares.  In a single scan,
  *	the two matrices are compared bit position by bit position until
  *	a miscompare is found or the end of the matrices is reached
  *	(without miscompares).
  *
  *	Inputs - 
  *		CMPR has two control input (to the top and middle 
  *		nodes).  The input to the top node initiates the 
  *		comparison.  The state of the input to the middle node
  *		determines the location in the logic program where the 
  *		next comparison will start.
  *	Outputs- 
  *		CMPR produces three possible outputs.  The output from
  *		the top node echoes the state of the top input.  Power
  *		is passed to the output from the middle node when a 
  *		miscompare is found.  The state of the putput from the 
  *		bottom node indicates whether the miscompared bit in 
  *		matrix a is a 1 or a 0.
  *	Top Node Content -
  *		The entry in the top node is the first reference in 
  *		matrix a, one of the two matrices to be compared.
  *	Middle Node Content -
  *		The register entered in the middle node is the pointer 
  *		to matrix b, the other matrix to be compared.  The first
  *		register in matrix b is the next contiguous register
  *		following the pointer register.
  *
  *		The value stored inside the pointer register increments
  *		with each bit position in the two matrices that is being
  *		compared.  As bit position 1 in matrix a and matrix b is
  *		compared, the pointer register contains a value of 1; as 
  *		bit position 2 in the matrices are compared, the pointer 
  *		value increments to 2; etc.
  *
  *		When the outputs signal a miscompare, you can check the 
  *		the accumulated count in the pointer register to determine
  *		the bit position in the matrices of the miscompare.
  *		
  *	Botton Node Content -
  *		The integer value entered in the bottom node specifies 
  *		a length length of the two matrices - i.e., the number 
  *		of registers or 16-bit words in each matrix.  (Matrix a
  *		and Matrix b have the same length.)  The matrix	length 
  *		can be in the range 1 ... 1024.  Ex. A length of 2 
  *		indicates that Matrix a and Matrix b contain 32 bits.
  *
  */
public class CMPRComponent implements CircuitComponent {
  /**
    * The register used to hold this component's top node value.
    */
  private int matrixaLine;
  /**
    * The register used to hold this component's middle node value.
    */
  private int pointerLine;
  /**
    * The register used to hold this component's bottom node value.
    */
  private int lengthLine;
  /**
    * Flag indicating whether component should be highlighted. 
    */
  private boolean powered;
  /**
    * Image used to draw this component in a network when it is not 
    * powered.
    */
  private static Image pic;
  /**
    * Image used to draw this component in a network when it is  
    * powered.
    */
  private static Image picLit;
  /**
    * Cursor image used for this component during drag and drop.
    */
  private static Image cursorImage;
  /**
    * Icon used for this component in the component bin.
    */
  private static ImageIcon icon;
  /**
    * ToolTip text used for this component.
    */
  private static String tip = "CMPR";
  /**
    * The simulations internal name for this component.
    */
  private static String name = "CMPR";

  static {
      // This static initializer loads all the image files needed
      URL myURL;
      ImageIcon myIcon;
      myURL = ClassLoader.getSystemResource("BigCMPR.gif");
    	myIcon = new ImageIcon(myURL);
    	pic = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("BigCMPROn.gif");
        myIcon = new ImageIcon(myURL);
        picLit = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("CursorCMPR.gif");
        myIcon = new ImageIcon(myURL);
        cursorImage = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("ToolCMPR.gif");
        icon = new ImageIcon(myURL);
  }

  /**
    * Construct a 3-rung Add function using default source, 
    * destination, and iteration values.
    */
  public CMPRComponent() {
    matrixaLine = 0;  // initial registers used
    pointerLine = 1;
    lengthLine  = 2;  // SimMemory auto initializes regs
  }

  /**
    * @return 3, the number of ladder logic rungs occupied by this 
    * component.
    */
  public int size() {
    return 3;
  }

  /**
    * Performs one evaluation of this component using the specified
    * input logic levels. If the 1st rung of the CMPR is logic-low,
    * then the CMPR resets. Otherwise, the operation commences and 
    * the current state of the first rung is passed to the output.  
    *
    * The top and middle rungs of the input (b[0], b[1]) 
    * are read and the top, middle, and bottom rungs of the output 
    * (boolean[0], boolean[1], and boolean[2]) is affected by the 
    * operation.
    *
    * @param b - array containing input logic levels of the rungs
    *            occupied by this component. 
    *
    * @return the output logic levels from this component.
    */
  public boolean[] step(boolean[] b) {
     powered = b[0];
     boolean tempArr[];
     tempArr = new boolean[] {false, false, false};
     if(b[0]){	// if ON
     	int length  = SimMemory.GetReg(lengthLine);
     	boolean miscompare = false;
	tempArr[0] = true;	// pass input state of top rung state to output top rung
	int i = 0;		// word position within matrix
	int pointer = 0;	// bit position within matrix
	if(!b[1]){					// if rung 2 - restart at last miscompare
     	    pointer = SimMemory.GetReg(pointerLine);	// initialize pointer to last bit miscompare
	    i = pointer/16;				// initialize word position "  "  "
	}
	for(; i < length; i++){				// i == word
	    int matrixA = SimMemory.GetReg(matrixaLine+i);	
	    int matrixB = SimMemory.GetReg(pointerLine+1+i);// next contiguous reg after pointer-reg
	    int bit = pointer%16;			    // bit position within word
	    while(pointer<=(length*16) && !miscompare && bit<17) { // step through word
		int bitA = matrixA & (1 << bit);
		int bitB = matrixB & (1 << bit);
		if(bitA != bitB){
		    tempArr[1] = true;				// set middle output rung
		    miscompare = true;				// i.e. miscompare detected
		    SimMemory.SetReg(pointerLine, pointer + 1);	// save value/bitPosition of pointer
		    if(bitA!=0){					// in miscompare, if matrixa's bit=1
		    	tempArr[2] = true;			// set bottom output rung
		    }
		}
		pointer++;
		bit++;
	    }
	}
     }
     return tempArr;			
  }					

  /**
    * Displays a modal dialog box to allow the user to edit the
    *  	parameters of this component, such as which registers are 
    * 	used. This method will initialize the state of the 
    * 	existing instance of <CODE>Node3Dialog</CODE> in 
    * 	<CODE>Master</CODE>, show it, and then update source and 
    *   destination registers to be used in the operation of the 
    *	AND based on the user's selections from the dialog.
    */
  public void edit() {
    Node3Dialog a = Master.DialogNode3;
    a.setTitle("CMPR Register Setup");
    SimMemory.deleteUser(pointerLine, this);
    a.setState(matrixaLine,pointerLine, lengthLine, this);
    a.setVisible(true);
    matrixaLine 	= a.getValue1Line();
    pointerLine 	= a.getValue2Line();
    lengthLine 		= a.getResultLine();
    SimMemory.addUser(pointerLine, this);
  }

  /**
    * Clears any registers marked as used by this component prior to
    * its deletion.
    */
  public void clear() {
    SimMemory.deleteUser(pointerLine, this);
  }

  /**
    * Resets this component, and initializes values
    * in <CODE>SimMemory</CODE>.
    */
  public void reset() {
	// Not needed in this component
  }

  /**
    * Returns the text to be used as a tool tip when this component
    * is placed in the component bin.
    *
    * @return the string to display.
    */
  public String getTip() {
    return tip;
  }

  /**
    * Returns the name of this component to be used internally in 
    * saved files and in drag and drop operations.
    *
    * @return the string to use.
    */
  public String getName() {
    return name;
  }

  /**
    * Returns the Icon to be used for this component in the component 
    * bin.
    *
    * @return the icon.
    */
  public ImageIcon getToolIcon() {
    return icon;
  }

  /**
    * Returns a one line string representing the state of this 
    * component for use in a save file. The string will be of the 
    * form: <BR>
    * CMPR 
    * <I>matrixa-register pointer-register length-register
    * </I>.
    *
    * @return the parameter string.
    */
  public String getSaveData() {
    return name + " " + matrixaLine + " " + pointerLine 
		+ " " + lengthLine;
  }

  /**
    * Returns whether this component must be the rightmost component 
    * on any rung.  For example, an output coil terminates a rung; 
    * a switch does not.
    *
    * @return <CODE>False</CODE> since CMPRs don't terminate rungs.
    */
  public boolean terminatesRung() {
    return false;
  }

  /**
    * Returns the image to use as a mouse cursor for this component 
    * during a drag and drop operation.
    *
    * @return the cursor image.
    */
  public Image getImage() {
    return cursorImage;
  }

  /**
    * Returns a new instance of this component created using the 
    * default constructor.
    * This method is used by the <CODE>getComponent</CODE> factory 
    * method of <CODE>Elements</CODE>.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance() {
    CMPRComponent x = new CMPRComponent();
    SimMemory.addUser(x.pointerLine, x);
    return x;
  }

  /**
    * Returns a new instance of this component, using the specified 
    * initial parameters.  This method is used by the 
    * <CODE>getComponent</CODE> factory method of <CODE>Elements</CODE> 
    * to instantiate new components during a file open.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance(int[] a) {	
    CMPRComponent x = new CMPRComponent();
    x.matrixaLine 	= a[0];
    x.pointerLine 	= a[1];
    x.lengthLine    	= a[2];
    SimMemory.addUser(x.pointerLine, x);
    return x;
  }

  /**
    * Draws this component and any associated parameters at the 
    * specified coordinates in the graphics context. The matrixa 
    * register is written in the top third of the CMPR, the 
    * pointer register is written in the middle third of the 
    * CMPR and the length register is written in the bottom third 
    * of the CMPR.
    *
    * @param g  the graphics context in which to draw.
    * @param x  the x-coordinate of the northwest corner.
    * @param y  the y-coordinate of the northwest corner.
    * @param o  the image observer to be notified of image drawing.
    */
  public void drawComponent(Graphics g,int x,int y,ImageObserver o){
    if(Master.isSimActive && powered) {
      // Draw a powered component
      g.drawImage(picLit, x, y, o);
      g.setColor(Network.darkGreen);
    } 
    else {
      // Draw a non-powered component
      g.drawImage(pic, x, y, o);
    }

    // Draw Register assignments onto component at rung
    String text;
    
    // Draw the source register in the top third of the component
    text = "     " + Integer.toString(matrixaLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+16);

    // Draw the destination register in the middle third of the comp
    text = "     " + Integer.toString(pointerLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+40);

    // Draw the length register in the bottom third of the component
    text = "     " + Integer.toString(lengthLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+64);

    g.setColor(Color.black);	// in case the color got set to 
  } 				// dark green set back to black
}




/*  To compile and run this Class
	first  - make sure that the class's name is also the file name 
		 -- observe case
	second - at the command prompt in the working directory of the 
		 SDK compile the class.  
			EX. L:\work>N:jdk1.2.1\bin\javac *.java
		 In this case all file *.java will be linked, 
		 therefore take care that they are related or delete 
		 unneeded files.
	third  - run the Class by 
			EX. L:\WORK>N:jdk1.2.1\bin\java figure1511
		 where figure1511 is both the name of the class & file.

    For creating links to, for example, methods ...

	{@link #reset() <CODE>reset</CODE>.}
*/
		