import java.awt.*;
import java.awt.image.*;
import javax.swing.*;

/**
  * <CODE>CircuitComponent</CODE> is the interface for any ladder logic component,
  * such as a switch or output coil.
  *
  * @author Mike Cammarano
  * @author Charlie So
  */
public interface CircuitComponent 
{
  /**
    * Returns the number of ladder logic rungs occupied by this component.
    *
    * @return the number of ladder logic rungs occupied by this component.
    */
  public abstract int size();

  /**
    * Performs one evaluation of this component using the specified input
    * logic levels.
    *
    * @param b  array containing input logic levels of the rungs
    *           occupied by this component.
    *
    * @return the output logic levels from this component.
    */
  public abstract boolean[] step(boolean[] b);

  /**
    * Displays a modal dialog box to allow the user to edit the parameters
    * of this component, such as which registers are used. If there are no
    * user editable parameters for this component, then the method will
    * simply return.
    */
  public abstract void edit();

  /**
    * Resets all the values for the component. Called at the start of each
    * new simulation run to perform any needed initialization.
    */
  public abstract void reset();

  /**
    * Clears any registers marked as used by this component prior to
    * its deletion.
    */
  public abstract void clear();

  /**
    * Returns the text to be used as a tool tip when this component
    * is placed in the component bin.
    *
    * @return the string to display.
    */
  public abstract String getTip();

  /**
    * Returns the name of this component to be used internally in saved files
    * and in drag and drop operations.
    *
    * @return the string to use.
    */
  public abstract String getName();

  /**
    * Returns the Icon to be used for this component in the component bin.
    *
    * @return the icon.
    */
  public abstract ImageIcon getToolIcon();


  /**
    * Returns a new instance of this component created using the default constructor.
    * This method is used by the <CODE>getComponent</CODE> factory method of
    * <CODE>Elements</CODE>.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public abstract CircuitComponent getInstance();

  /**
    * Returns a new instance of this component, using the specified initial parameters.
    * This method is used by the <CODE>getComponent</CODE> factory method of
    * <CODE>Elements</CODE> to instantiate new components during a file open.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public abstract CircuitComponent getInstance(int[] a);

  /**
    * Returns a one line string representing the state of this component for use in
    * a save file. The string should be the whitespace separated concatenation of
    * this component's internal name (as returned by the <CODE>getName</CODE> method)
    * with up to six integers representing its parameters.
    * <P>
    * For example, an instance of the concrete class OutputComponent that is tied
    * to bit 12 of register 0 in the ladder logic diagram would return the string:
    * <BR>
    * <PRE>OP 0 12</PRE><BR>
    * from this method.
    *
    * @return the parameter string.
    */
  public abstract String getSaveData();

  /**
    * Returns whether this component must be the rightmost component on any rung.
    * For example, an output coil terminates a rung; a switch does not.
    *
    * @return <CODE>True</CODE> if no other components may be placed after this
    *         one on a ladder logic rung.
    */
  public abstract boolean terminatesRung();
  
  /**
    * Returns the image to use as a mouse cursor for this component during
    * a drag and drop operation.
    *
    * @return the cursor image.
    */
  public abstract Image getImage();

  /**
    * Draws this component and any associated parameters at the specified coordinates
    * in the graphics context.
    *
    * @param g  the graphics context in which to draw.
    * @param x  the x-coordinate of the northwest corner.
    * @param y  the y-coordinate of the northwest corner.
    * @param o  the image observer to be notified of image drawing.
    */
  public abstract void drawComponent(Graphics g, int x, int y, ImageObserver o);
}
