import java.awt.*;
import java.awt.image.*;
import java.lang.String.*;
import java.net.*;
import java.util.*;
import javax.swing.*;

/**
  * Class <CODE>DivComponent</CODE>.
  *
  * @author Mike Cammarano
  * @author Charlie So
  * @author Robert Marlowe
  *
  * Division implementation of the <CODE>CircuitComponent</CODE> 
  * interface.
  *
  * Class for 16-bit Division Function. This is utilized by 
  * <CODE>DivComponent</CODE>.
  *
  *  The DIV instruction divides unsigned value1 (its top node) 
  *	by unsigned value2 (its middle node) and posts the 
  *	quotient and remainder in two contiguous holding 
  *	registers in the bottom node.
  *
  *	Input - 
  *		DIV has two control inputs (to the top and 
  *		middle nodes).  The top input initiates the 
  *		operation when it is ON.
  *		The state of the input to the middle node
  *		indicates whether the remainder will be specified
  *		as a decimal or as a fraction.  For example, if 
  *		value1 = 8 and value 2 = 3, the dicimal remainder 
  *		(middle input ON) is 6666; the fractional 
  *		remainder (middle input OFF) is 2.
  *	Output- 
  *		DIV can produce one of three possible outputs.  
  *		Power passed at the top output indicates the 
  *		successful completion of a DIV operation.  Power
  *		passed from the middle or bottom output indicates 
  *		an error in the operation.
  *	Top and Middle Node Content -
  *		The top and middle nodes contain value1 and value2,
  *		respectively, which can be an integer in the 
  *		range 1 ... 999.
  *	Bottom Node Content -
  *		The register entered in the borrom node is the 
  *		first of two contiguous holding registers.  The
  *		result of the division is posted in the displayed 
  *		register.  The remainder is posted in the implied 
  *		register as either a decimal or a fraction 
  *		(depending on the state of the middle input).
  *
  */
public class DivComponent implements CircuitComponent {
  /**
    * The register used to hold this component's top node value.
    */
  private int value1Line;
  /**
    * The register used to hold this component's middle node value.
    */
  private int value2Line;
  /**
    * The register used to hold this component's bottom node 
    *	quotient value.
    */
  private int quoLine;
  /**
    * The register used to hold this component's bottom node 
    *	remainder value.
    */
  private int remLine;
  /**
    * Flag indicating whether component should be highlighted. 
    */
  private boolean powered;
  /**
    * Image used to draw this component in a network when it is not 
    * powered.
    */
  private static Image pic;
  /**
    * Image used to draw this component in a network when it is  
    * powered.
    */
  private static Image picLit;
  /**
    * Cursor image used for this component during drag and drop.
    */
  private static Image cursorImage;
  /**
    * Icon used for this component in the component bin.
    */
  private static ImageIcon icon;
  /**
    * ToolTip text used for this component.
    */
  private static String tip = "DIV";
  /**
    * The simulations internal name for this component.
    */
  private static String name = "DIV";

  static {
      // This static initializer loads all the image files needed
      URL myURL;
      ImageIcon myIcon;
      myURL = ClassLoader.getSystemResource("BigDiv.gif");
    	myIcon = new ImageIcon(myURL);
    	pic = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("BigDivOn.gif");
        myIcon = new ImageIcon(myURL);
        picLit = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("CursorDiv.gif");
        myIcon = new ImageIcon(myURL);
        cursorImage = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("ToolDiv.gif");
        icon = new ImageIcon(myURL);
  }

  /**
    * Construct a 3-rung Divide function using default value1, 
    *	value2, quotient, and remainder values.
    */
  public DivComponent() {
    value1Line 	= 0;	// initial registers used
    value2Line 	= 1;
    quoLine	= 2;
    remLine    	= 3;	// SimMemory automatically initializes regs
  }

  /**
    * @return 3, the number of ladder logic rungs occupied by this 
    * component.
    */
  public int size() {
    return 3;
  }

  /**
    * Performs one evaluation of this component using the specified
    *	input logic levels. If the 1st rung of the DIV is logic-low,
    *	then the DIV resets. Otherwise, the divide operation continues.
    * 
    *
    * The input top rung (b[0]), input middle rung (b[1]), output top
    *	rung boolean[0], output middle rung boolean[1], and output
    *	bottom rung boolean[2], is affected by the operation.
    *
    * @param b - array containing input logic levels of the rungs
    *            occupied by this component.
    *
    * @return the output logic levels from this component.
    */
  public boolean[] step(boolean[] b) {
    	int tempValue1 = SimMemory.GetReg(value1Line);
    	int tempValue2 = SimMemory.GetReg(value2Line);
	int tempQuo = 0;
	int tempRem = 0;
        powered = b[0];
	boolean tempArr[];
	tempArr = new boolean[] {false, false, false};
	if(b[0]) {
	    if(tempValue2==0){	   	// !!! Division by 0
		tempArr[2]=true;
	    }
	    else{
		tempQuo = tempValue1/tempValue2;
	    	if(tempQuo > 999){	// Overflow
		    tempArr[1] = true;
		}
		else{ 			// Division success
		    tempArr[0] = true;
		}
		// Get fractional remainder
	    	tempRem = tempValue1%tempValue2;   
	    	if(b[1]){ // convert to decimal remainder
		    tempRem = tempRem*10000/tempValue2;
		}
		SimMemory.SetReg(quoLine, tempQuo);// write 3rd Rung
	        SimMemory.SetReg(remLine, tempRem);// write 3rd Rung
	    }
 	}
  	return tempArr;				 
  }						

  /**
    * Displays a modal dialog box to allow the user to edit the
    *  	parameters of this component, such as which registers are 
    * 	used. This method will initialize the state of the 
    * 	existing instance of <CODE>DivDialog</CODE> in 
    * 	<CODE>Master</CODE>, show it, and then update value1 and 
    *   value2 values to be used in the operation of the Division 
    *   based on the user's selections from the dialog.
    */
  public void edit() {
    DivDialog a = Master.DialogDiv;
    SimMemory.deleteUser(quoLine, this);
    SimMemory.deleteUser(remLine, this);
    a.setState(value1Line,value2Line, quoLine, remLine, this);
    a.setVisible(true);
    value1Line 	= a.getValue1Line();
    value2Line 	= a.getValue2Line();
    quoLine 	= a.getQuoLine();
    remLine	= a.getRemLine();
    SimMemory.addUser(quoLine, this);
    SimMemory.addUser(remLine, this);
  }


  /**
    * Clears any registers marked as used by this component prior to
    * its deletion.
    */
  public void clear() {
    SimMemory.deleteUser(quoLine, this);
    SimMemory.deleteUser(remLine, this);
  }

  /**
    * Resets this component, and initializes values
    * in <CODE>SimMemory</CODE>.
    */
  public void reset() {
	// Not needed in this component
  }

  /**
    * Returns the text to be used as a tool tip when this component
    * is placed in the component bin.
    *
    * @return the string to display.
    */
  public String getTip() {
    return tip;
  }

  /**
    * Returns the name of this component to be used internally in 
    * saved files and in drag and drop operations.
    *
    * @return the string to use.
    */
  public String getName() {
    return name;
  }

  /**
    * Returns the Icon to be used for this component in the component 
    * bin.
    *
    * @return the icon.
    */
  public ImageIcon getToolIcon() {
    return icon;
  }

  /**
    * Returns a one line string representing the state of this 
    * component for use in a save file. The string will be of the 
    * form: <BR>
    * Add 
    * <I>value1-register value2-register quotient-register 
    * remainder-register</I>.
    *
    * @return the parameter string.
    */
  public String getSaveData() {
    return name + " " + value1Line + " " + value2Line 
		+ " " + quoLine    + " " + remLine;
  }

  /**
    * Returns whether this component must be the rightmost component 
    * on any rung.  For example, an output coil terminates a rung; 
    * a switch does not.
    *
    * @return <CODE>False</CODE> since DIV's don't terminate rungs.
    */
  public boolean terminatesRung() {
    return false;
  }

  /**
    * Returns the image to use as a mouse cursor for this component 
    * during a drag and drop operation.
    *
    * @return the cursor image.
    */
  public Image getImage() {
    return cursorImage;
  }

  /**
    * Returns a new instance of this component created using the 
    * default constructor.
    * This method is used by the <CODE>getComponent</CODE> factory 
    * method of <CODE>Elements</CODE>.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance() {
    DivComponent x = new DivComponent();
    SimMemory.addUser(x.quoLine, x);
    SimMemory.addUser(x.remLine, x);
    return x;
  }

  /**
    * Returns a new instance of this component, using the specified 
    * initial parameters.  This method is used by the 
    * <CODE>getComponent</CODE> factory method of <CODE>Elements</CODE> 
    * to instantiate new components during a file open.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance(int[] a) {	
    DivComponent x = new DivComponent();
    x.value1Line = a[0];
    x.value2Line = a[1];
    x.quoLine    = a[2];
    x.remLine    = a[3];
    SimMemory.addUser(x.quoLine, x);
    SimMemory.addUser(x.remLine, x);
    return x;
  }

  /**
    * Draws this component and any associated parameters at the 
    * specified coordinates in the graphics context. The value1 value 
    * is written in the top third of the Divider, the value2 value is 
    * written in the middle third of the Divider and the
    * quotient and remainder values are written in the bottom third 
    * of the Divider.
    *
    * @param g  the graphics context in which to draw.
    * @param x  the x-coordinate of the northwest corner.
    * @param y  the y-coordinate of the northwest corner.
    * @param o  the image observer to be notified of image drawing.
    */
  public void drawComponent(Graphics g,int x,int y,ImageObserver o){
    if(Master.isSimActive && powered) {
      // Draw a powered component
      g.drawImage(picLit, x, y, o);
      g.setColor(Network.darkGreen);
    } 
    else {
      // Draw a non-powered component
      g.drawImage(pic, x, y, o);
    }

    // Draw Register assignments onto component at rung
    String text;
    
    // Draw the value1 register in the top third of the component
    text = "     " + Integer.toString(value1Line);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+16);

    // Draw the value2 register in the middle third of the component
    text = "     " + Integer.toString(value2Line);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+40);

    // Draw the quotient register in the bottom L third of component
    text = "     " + Integer.toString(quoLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+4, y+64);

    // Draw the remainder register in the bottom R third of comp.
    text = "     " + Integer.toString(remLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+16, y+64);

    g.setColor(Color.black);	// in case the color got set to 
				// dark green set back to black
  }
}




/*  To compile and run this Class
	first  - make sure that the class's name is also the file name 
		 -- observe case
	second - at the command prompt in the working directory of the 
		 SDK compile the class.  
			EX. L:\work>N:jdk1.2.1\bin\javac *.java
		 In this case all file *.java will be linked, 
		 therefore take care that they are related or delete 
		 unneeded files.
	third  - run the Class by 
			EX. L:\WORK>N:jdk1.2.1\bin\java figure1511
		 where figure1511 is both the name of the class & file.

    For creating links to, for example, methods ...

	{@link #reset() <CODE>reset</CODE>.}
*/
		