import java.lang.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.net.*;
import java.util.*;
import javax.swing.*;

/**
  * An instance of<CODE>Elements</CODE> is a graphical component bin, and also
  * a factory for circuit components. The graphical component bin shows all the
  * user selectable components in a grid of DragLabels. All the components made
  * available through this class are dynamically loaded based on specifications
  * given in the file <PRE>components.cfg</PRE>. This makes it possible for
  * new components to be added to the simulation by writing a single class
  * that implements <CODE>CircuitComponent</CODE> and adding its name to
  * <PRE>components.cfg</PRE>.<P>
  *
  * The <PRE>components.cfg</PRE> file should resemble the following:<P>
  *
  * <PRE>
  * #Ladder Logic Simulator 1.0b
  * #
  * #  When new implementations of CircuitComponent are created,
  * #  they must be added to this file.
  * #
  * #LABEL Switches
  * NOComponent
  * NCComponent
  * PTComponent
  * NTComponent
  * #LABEL Outputs
  * OPComponent
  * TimerComponent
  * UCTRComponent
  * DCTRComponent
  * </PRE><P>
  *
  * Lines prefixed by the '#' symbol are either comments or directives. The first
  * line must consist of the correct version string, as shown. Lines of the form:<BR>
  * <PRE>#LABEL <PRE><I>label-name</I><BR>
  * indicate that a label should be placed in the graphical component bin.
  * All lines not prefixed by the '#' symbol are expected to be names of classes
  * within the Java class path that implement <CODE>CircuitComponent</CODE>.
  * <P>
  * If <PRE>components.cfg</CODE> is inaccessible, the default component bin is
  * used.
  *
  * @author Mike Cammarano
  * @author Charlie So
  */
public class Elements extends JDialog 
{
  /**
    * Reference to the checkbox in the view menu that shows and hides
    * the graphical component bin.
    */
  private JCheckBoxMenuItem check;

  /**
    * Hashtable indexing circuit components by their internal names.
    */
  private static Hashtable hash = new Hashtable(16);
  /**
    * Linked list that temporarily holds circuit components until a <PRE>#LABEL</PRE>
    * directive or the end of the <PRE>components.cfg</PRE> file is encountered,
    * at which point they are placed into a <CODE>JPanel</CODE> with a
    * <CODE>GridLayout</CODE>.
    */
  private LinkedList tempList = new LinkedList();
  /**
    * JPanel used temporarily during initialization.
    */
  private JPanel p = new JPanel();

  /**
    * Common instance of <CODE>EmptyComponent</CODE> to be retrieved by the
    * <CODE>getComponent</CODE> factory method.
    */
  private static CircuitComponent emptyComp = new EmptyComponent();
  /**
    * Common instance of <CODE>HShortComponent</CODE> to be retrieved by the
    * <CODE>getComponent</CODE> factory method.
    */
  private static CircuitComponent hshortComp = new HShortComponent();
  /**
    * Common instance of <CODE>NullComponent</CODE> to be retrieved by the
    * <CODE>getComponent</CODE> factory method.
    */
  private static CircuitComponent nullComp = new NullComponent();
  /**
    * Common instance of <CODE>EndRowComponent</CODE> to be retrieved by the
    * <CODE>getComponent</CODE> factory method.
    */
  private static CircuitComponent endrowComp = new EndRowComponent();
  /**
    * Common instance of <CODE>UnknownComponent</CODE> to be retrieved by the
    * <CODE>getComponent</CODE> factory method.
    */
  private static CircuitComponent unknownComp = new UnknownComponent();

  /**
    * Instantiates the component bin window and the component factory.
    *
    * @param owner     this dialogs owner.
    * @param checkbox  a reference to the checkbox in the view menu that shows
    *                  and hides the graphical component bin.
    */
  public Elements(Frame owner, JCheckBoxMenuItem checkbox)
  {
    super(owner, "Component Bin");
    check = checkbox;
    p.setLayout (new BoxLayout(p, BoxLayout.Y_AXIS));

    // EmptyComponent, HShortComponent, NullComponent, and EndRowComponent
    // are all special indicators within a Network, so we need to put them
    // all in the hash table and be able to build them in the Factory methods.
    hash.put(emptyComp.getName(), emptyComp);
    hash.put(hshortComp.getName(), hshortComp);
    hash.put(nullComp.getName(), nullComp);
    hash.put(endrowComp.getName(), endrowComp);
    hash.put(unknownComp.getName(), unknownComp);

    JLabel lab = new JLabel("Shorts");
    lab.setHorizontalAlignment(JLabel.CENTER);
    p.add(lab);
    addComponent(hshortComp);
    DragButton button;
    URL myURL = ClassLoader.getSystemResource("ToolVShort.gif");
    ImageIcon icon = new ImageIcon(myURL);
    button = new DragButton("VSHORT", icon, icon.getImage());
    button.setToolTipText("Vertical Short");
    tempList.add(button);
    addComponentPanel();

    if(!open()) { // if we can't read components.cfg, use defaults
      System.err.println("Unable to read components.cfg. Using default components.");
      lab = new JLabel("Switches");
      lab.setHorizontalAlignment(JLabel.CENTER);
      p.add(lab);
      addComponent(new NOComponent());
      addComponent(new NCComponent());
      addComponent(new PTComponent());
      addComponent(new NTComponent());
      addComponentPanel();
      lab = new JLabel("Outputs");
      lab.setHorizontalAlignment(JLabel.CENTER);
      p.add(lab);
      addComponent(new OPComponent());
      addComponent(new TimerComponent());
      addComponent(new UCTRComponent());
      addComponent(new DCTRComponent());
      addComponentPanel();
      lab = new JLabel("Math");
      lab.setHorizontalAlignment(JLabel.CENTER);
      p.add(lab);
      addComponent(new AddComponent());
      addComponent(new SubComponent());
    }

    addComponentPanel();

    JScrollPane sp = new JScrollPane(p);
    getContentPane().add(sp);
    pack();
    setSize(new Dimension(160, 220));	// size of Component Bin
    setResizable(false);
  }

  /**
    * Sets the checkbox associated with this component bin to the correct state,
    * and show or hide the window.
    *
    * @param b  whether to show or hide the component bin.
    */
  public void setVisible(boolean b) {
    check.setSelected(b);
    super.setVisible(b);
  }

  /**
    * Builds a new circuit component of a specific type, with optional
    * parameters. This factory method is used to build the new component
    * after a drag and drop operation, or when a file is loaded.
    *
    * @param name  the internal name associated with a component type.
    */
  public static CircuitComponent getComponent(String name) {
    int[] a = new int[6];
    int i=0;
    StringTokenizer st = new StringTokenizer(name);
    name = st.nextToken();
    boolean passParams = false;
    while(st.hasMoreTokens() && i<6) {
      passParams = true;
      a[i] = Integer.parseInt(st.nextToken());
      i++;
    }
    CircuitComponent x = (CircuitComponent) hash.get(name);
    if(x == null)
      // Return the unknown component. This is really an error condition
      x = unknownComp;
    if(passParams) {
      // If the String contained parameters as well as the name of the component,
      // construct the new component with those parameters.
      return x.getInstance(a);
    } else {
      // Otherwise, just construct a component with the default parameters.
      return x.getInstance();
    }
  }

  /**
    * Adds another circuit component to the hashtable of components and queues
    * it for addition to the component bin.  This method prepares the factory
    * method, <CODE>getComponent</CODE>, to build new instances as needed.
    *
    * @param newComp  the component to include.
    */
  public void addComponent(CircuitComponent newComp) {
    DragButton button;
    hash.put(newComp.getName(), newComp);
    button = new DragButton(newComp.getName(), newComp.getToolIcon(),
                            newComp.getImage());
    button.setToolTipText(newComp.getTip());
    tempList.add(button);
  }

  /**
    * Adds a new <CODE>JPanel</CODE> with a <CODE>GridLayout</CODE> to the
    * component bin and adds the queued circuit components to it.
    *
    * @param newComp  the component to include.
    */
  public void addComponentPanel() {
    ListIterator li = tempList.listIterator();
    int rows = tempList.size();
    if((rows % 2) == 1)
      rows++;
    rows /= 2;
    if(rows > 0) {
      JPanel inner = new JPanel();
      inner.setLayout(new GridLayout(rows, 2, 4, 8));
      while(li.hasNext()) {
        inner.add((DragButton) li.next());
      }
      p.add(inner);
    }
    tempList.clear();
  }

  /**
    * Opens <PRE>components.cfg</PRE> and dynamically initiallizes the
    * component bin based on the file's specifications.
    *
    * @return  <CODE>True</CODE> if successful.
    */
  public boolean open() {
    boolean success = false;
    URL myURL = ClassLoader.getSystemResource("components.cfg");
    if(myURL!=null) {
      File tempFile = new File(myURL.getFile());
      ClassLoader sysLoader = ClassLoader.getSystemClassLoader();
      BufferedReader loadReader;
      Class tempClass;
      String data;
      String name;
      if(tempFile.exists()) {
        try {   //  IOException
          loadReader = new BufferedReader(new FileReader(tempFile));
          data = loadReader.readLine();
          if(data.equals("#"+Master.versionString)) {
            data = loadReader.readLine();
            while(data != null) {
              if(data.startsWith("#")) {        // Comments or directives
                if(data.startsWith("#LABEL")) {
                  addComponentPanel();
                  JLabel lab = new JLabel(data.substring(7));
	            lab.setHorizontalAlignment(JLabel.CENTER); // Component Bin text justification
                  p.add(lab);
                }
              } else {                          // Class names of CircuitComponents
                try {
                  tempClass = sysLoader.loadClass(data);
                  CircuitComponent comp = (CircuitComponent) tempClass.newInstance();
                  addComponent(comp);
                  if(Master.verbose)
                    System.err.println("Loaded class: " + data);
                  success = true;
                } catch(Throwable e) {
                  System.err.println("ERROR: Failed to initialize class: " + data);
                }
              }
              data = loadReader.readLine();
            }
          }
          loadReader.close();
        } catch  (IOException io) {
          System.err.println(io);
        }
      }
    }
    return success;
  }
}
