import java.util.*;
import javax.swing.*;

/**
  * An implementation of <CODE>ListModel</CODE> (for a <CODE>JList</CODE>) that
  * uses a linked list of <CODE>NetworkFrame</CODE>s as its underlying model.
  * Used in <CODE>NetworkList</CODE> as the model for the list of NetworkFrames.<P>
  *
  * It implements a basic set of list operations needed for managing the NetworkFrames.
  *
  * @author Mike Cammarano
  * @author Charlie So
  */
public class LinkedListModel extends AbstractListModel
{
  /**
    * The linked list represented by this model.
    */
  private LinkedList myList;

  /**
    * Constructs a new model using an empty linked list.
    */
  LinkedListModel() {
    myList = new LinkedList();
  }

  /**
    * Constructs a new model using the specified linked list.
    */
  LinkedListModel(LinkedList l) {
    myList = l;
  }

  /**
    * Retrieves the title of a <CODE>NetworkFrame</CODE> from a particular
    * slot in the list.
    *
    * @param index  which network to retrieve.
    *
    * @return       the title of the network at the specified location.
    */
  public Object getElementAt(int index) {
    NetworkFrame network = ((NetworkFrame) myList.get(index));
    return network.getTitle();
  }

  /**
    * Retrieves a <CODE>NetworkFrame</CODE> from a particular slot in the list.
    *
    * @param index  which network to retrieve.
    *
    * @return       the network at the specified location.
    */
  public Object get(int index) {
    return myList.get(index);
  }

  /**
    * Returns the number of networks in the list.
    *
    * @return the number of networks in the list.
    */
  public int getSize() {
    return myList.size();
  }

  /**
    * Adds a <CODE>NetworkFrame</CODE> to the end of the list and updates
    * the list model's listeners.
    *
    * @param ob  the network to add.
    */
  public void add(Object ob) {
    myList.add(ob);
    fireContentsChanged(this, myList.size(), myList.size());
  }

  /**
    * Inserts a <CODE>NetworkFrame</CODE> into the list at the specified index
    * and updates the list model's listeners.
    *
    * @param index  where to insert the new network.
    * @param ob     the network to add.
    */
  public void add(int index, Object ob) {
    myList.add(index, ob);
    fireIntervalAdded(this, index, index);
  }

  /**
    * Removes a <CODE>NetworkFrame</CODE> from the specified index and updates
    * the list model's listeners.
    *
    * @param index  the location to remove.
    *
    * @return       the removed <CODE>NetworkFrame</CODE>.
    */
  public Object remove(int index) {
    Object ob = myList.remove(index);
    fireIntervalRemoved(this, index, index);
    return ob;
  }

  /**
    * Removes a <CODE>NetworkFrame</CODE> from the end of the list and updates
    * the list model's listeners.
    *
    * @return   the removed <CODE>NetworkFrame</CODE>.
    */
  public Object removeLast() {
    Object ob = myList.removeLast();
    int index = myList.size();
    fireIntervalRemoved(this, index, index);
    return ob;
  }

  /**
    * Returns a <CODE>ListIterator</CODE> for this list.
    *
    * @param start  the starting index for the iterator.
    *
    * @return       the new iterator for this list.
    */
  public ListIterator listIterator(int start) {
    return myList.listIterator(start);
  }
}
