import java.awt.*;
import java.awt.event.*;
import java.awt.print.*;
import java.io.*;
import java.net.*;
import java.util.*;
import javax.swing.*;

/**
  * <Code>Master</Code> is the main class that handles all the interaction
  * between all the classes.  <Code>Master</Code> class handles all the 
  * actions and frames.  Actions include creating new <Code>Network</Code>,
  * saving the current <Code>Network</Code>, opening a previously saved
  * <Code> Network</Code> and printing the current <Code>Network</Code>.
  * 
  * There is a property file associated with <Code>Master</Code>, which keeps
  * track of many properties at the closing of <Code>Master</Code>.
  */

public class Master extends JFrame
                    implements ActionListener, WindowListener,
                               Pageable, Printable, Runnable
{
  /**
    * Whether lots of debugging output should be produced.
    */
  public static boolean verbose;
  /**
    * Whether the current project is saved.
    */
  public static boolean isSaved;
  /**
    * Whether the software is in simulation mode (rather than editing mode).
    */
  public static boolean isSimActive;
  /**
    * The single instance of this main application window.
    */
  public static Master frame;
  /**
    * The single instance of the component bin window.
    */
  public static JDialog toolframe;
  /**
    * The checkbox governing the component bin window.
    */
  public static JCheckBoxMenuItem components;
  /**
    * The checkbox governing the network manager window.
    */
  public static JCheckBoxMenuItem networks;
  /**
    * The file menu.
    */
  public static JMenu file;
  /**
    * The simulation menu.
    */
  public static JMenu simulate;
  /**
    * The file menu.
    */
  public static LinkedListModel net;
  /**
    * The desktop for all the <CODE>JInternalFrame</CODE> network frames.
    */
  public static JDesktopPane desktop;
  /**
    * The single instance of the network manager window.
    */
  public static NetworkList netList;
  /**
    * The single instance of the simulation window.
    */
  public static Simulation simWindow;
  /**
    * The single instance of the plugin window.
    */
  public static PluginRunner plugin;
  /**
    * Table of all plugins, indexed by name.
    */
  public static Hashtable pluginTable;
  /**
    * The single instance of the dialog box for switches.
    */
  public static NONCDialog DialogNONC;
  /**
    * The single instance of the dialog box for output coils.
    */
  public static OutputDialog DialogOutput;
  /**
    * The single instance of the dialog box for timers.
    */
  public static TimerDialog DialogTimer;
  /**
    * The single instance of the dialog box for counters.
    */
  public static CounterDialog DialogCounter;
  /**
    * The single instance of the dialog box for multiplication.
    */
  public static MulDialog DialogMul;
  /**
    * The single instance of the dialog box for division.
    */
  public static DivDialog DialogDiv;
  /**
    * The single instance of the dialog box for miscellaneous 3 register components.
    */
  public static Node3Dialog DialogNode3;
  /**
    * The single instance of the printing status window.
    */
  public static PrintingDialog print;
  /**
    * The version string for this software that is expected at the beginning of
    * various files.
    */
  public static final String versionString = "Ladder Logic Simulator 1.0b";

  /**
    * The single instance of the file chooser (open/save window).
    */
  private static JFileChooser chooser;
  /**
    * The single instance of the frame containing the help mini-browser.
    */
  private static JFrame help;
  /**
    * The single instance of the help mini-browser.
    */
  private static Browser helpbrowser;
  /**
    * The icon for the SimPLC application.
    */
  private static ImageIcon icon;
  /**
    * The properties table used to access the file <CODE>sim.properties</CODE>.
    */
  private static Properties props;
  /**
    * The active printer job.
    */
  private PrinterJob pj;


  public void windowActivated(WindowEvent e) {}
  public void windowClosed(WindowEvent e) {}
  public void windowClosing(WindowEvent e) {
    quit();
  }
  public void windowDeactivated(WindowEvent e) {}
  public void windowDeiconified(WindowEvent e) {}
  public void windowIconified(WindowEvent e) {}
  public void windowOpened(WindowEvent e) {}


	/**
	 * Updates the look and feel of all the nodes in the tree.
	 *
	 */
  public void updateAllUI() {
    SwingUtilities.updateComponentTreeUI(this);
    SwingUtilities.updateComponentTreeUI(toolframe);
    SwingUtilities.updateComponentTreeUI(chooser);
    SwingUtilities.updateComponentTreeUI(simWindow);
    SwingUtilities.updateComponentTreeUI(netList);
    SwingUtilities.updateComponentTreeUI(DialogNONC);
    SwingUtilities.updateComponentTreeUI(DialogOutput);
    SwingUtilities.updateComponentTreeUI(DialogTimer);
    SwingUtilities.updateComponentTreeUI(DialogCounter);
    SwingUtilities.updateComponentTreeUI(print);
  }


	/**
	 * Checks all the mouse listener for action commands.
	 *
	 * @param   e  is an action event.
	 */
  public void actionPerformed (ActionEvent e) 
  {
    String s = e.getActionCommand();
    if(s.equals("Exit")) {
      quit();
    } else if(s.equals("New")) {
      if(isSaved || promptSave("New")) {
        // clear everything
        clear();
        isSaved = true;
      }
    }	else if(s.equals("Open ...")) {
      open();
    } else if(s.equals("Save ...")) {
      // save the thing ...
      save();
    }	else if(s.equals("Print")) {
      pj = PrinterJob.getPrinterJob();
      pj.setPageable(Master.frame);
      pj.setJobName("Simulator printout");
      print = new PrintingDialog(frame, net.getSize());
      if(pj.printDialog()) {
        Thread printThread = new Thread(this);
        printThread.start();
        print.setVisible(true);
      }
    }	else if(s.equals("Component Bin")) {
      toolframe.setVisible(components.getState());
    } else if(s.equals("Network Manager")) {
      netList.setVisible(networks.getState());
    } else if(s.equals("System Look & Feel")) {
      try {
        UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
        updateAllUI();
      } catch (Exception exc) {
      System.err.println("Error loading L&F: " + exc);
      }
    } else if(s.equals("Java Look & Feel")) {
      try {
        UIManager.setLookAndFeel(UIManager.getCrossPlatformLookAndFeelClassName());
        updateAllUI();
      } catch (Exception exc) {
        System.err.println("Error loading L&F: " + exc);
      }
    } else if(s.equals("Standard")) {
      simWindow.setVisible(true);
    } else if(s.equals("Contents")) {
      helpbrowser.loadPage("..\\docs\\Index.html");
      help.show();
    } else if(s.equals("About ...")) {
      helpbrowser.loadPage("..\\docs\\about.html");
      help.show();
    } else {
      // PLUGIN!
      plugin.setPlugin((SimulatorPlugin) pluginTable.get(s));
      plugin.setVisible(true);
    }
  }


	/**
	 * Constructor.  The main frame is created in this constructor.
	 * Which includes all the menu items in the menu bar.
	 *
	 */
  public Master() 
  {
    super ("Ladder Logic Simulator");

    //  File
    JMenuBar jmb = new JMenuBar();
    file = new JMenu ("File");
    JMenuItem item;
    file.add (item = new JMenuItem ("New"));
    item.addActionListener (this);
    file.add (item = new JMenuItem ("Open ..."));
    item.addActionListener (this);
    file.add (item = new JMenuItem ("Save ..."));
    item.addActionListener (this);
    file.addSeparator();
    file.add (item = new JMenuItem ("Print"));
    item.addActionListener (this);
    file.addSeparator();
    file.add (item = new JMenuItem ("Exit"));
    item.addActionListener (this);
    jmb.add (file);

    //  View
    JMenu view = new JMenu("View");
    ButtonGroup rbg = new ButtonGroup();
    components = new JCheckBoxMenuItem("Component Bin");
    view.add(components);
    components.setSelected(true);
    components.addActionListener (this);

    networks = new JCheckBoxMenuItem("Network Manager");
    view.add(networks);
    networks.setSelected(true);
    networks.addActionListener (this);

    view.addSeparator();
    JRadioButtonMenuItem rad;
    rad = new JRadioButtonMenuItem("System Look & Feel");
    view.add(rad);
    rbg.add(rad);
    rad.addActionListener (this);
    rad = new JRadioButtonMenuItem("Java Look & Feel");
    view.add(rad);
    rbg.add(rad);
    rad.setSelected(true);
    rad.addActionListener (this);
    jmb.add(view);

    //  Simulate
    simulate = new JMenu("Simulate");
    simulate.add (item = new JMenuItem ("Standard"));
    item.addActionListener (this);
    simulate.addSeparator();
    jmb.add(simulate);

    //  Help
    JMenu help = new JMenu ("Help");
    help.add (item = new JMenuItem ("Contents"));
    item.addActionListener (this);
    help.addSeparator();
    help.add (item = new JMenuItem ("About ..."));
    item.addActionListener (this);
    jmb.add (help);

    setJMenuBar (jmb);
    }
 

	/**
	 * Since we are working with an application, we need to create a main function
	 * that will run all the classes that are needed. 
	 *
	 * @param   s  is a string that can be used in the invocation of the 
	 *             program.
	 */
  public static void main(String s[]) 
  {
    isSimActive = false;
    verbose = false;
    if(s.length > 1)
      verbose = true;

    JWindow splash = new JWindow();
    URL myURL = ClassLoader.getSystemResource("splash.gif");
    JLabel splashPic = new JLabel(new ImageIcon(myURL));
    splash.getContentPane().add(splashPic);
    splash.setSize(300, 200);
    Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
    splash.setLocation(screenSize.width/2-150, screenSize.height/2-100);
    splash.setVisible(true);

    props = new Properties();

    frame = new Master();
    frame.setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);

    chooser = new JFileChooser();
    javax.swing.filechooser.FileFilter ext =
                             new ExtensionFilter("net", "Simulator files (.net)");
    chooser.addChoosableFileFilter(ext);
    chooser.setFileFilter(ext);

    toolframe = new Elements(frame, components);
    desktop = new JDesktopPane();
    help = new JFrame("Ladder Simulator Help");
    help.getContentPane().add(helpbrowser = new Browser());
    myURL = ClassLoader.getSystemResource("help.gif");
    icon = new ImageIcon(myURL);
    help.setIconImage(icon.getImage());
    myURL = ClassLoader.getSystemResource("icon.gif");
    icon = new ImageIcon(myURL);
    frame.setIconImage(icon.getImage());

    DialogNONC = new NONCDialog(frame);
    DialogOutput = new OutputDialog(frame);
    DialogTimer = new TimerDialog(frame);
    DialogCounter = new CounterDialog(frame);
    DialogNode3 = new Node3Dialog(frame);
    DialogMul = new MulDialog(frame);
    DialogDiv = new DivDialog(frame);

    isSaved = true;
    SimMemory.init();

    frame.addWindowListener(frame);
    frame.setSize(600, 400);
    frame.setLocation(150, 100);
    toolframe.setLocation(600, 145);
    help.setLocation(400, 150);
    help.pack();
    help.setSize(400, 450);

    simWindow = new Simulation(frame);
    plugin = new PluginRunner(frame);

    frame.getContentPane().add(desktop);

    net = new LinkedListModel();
    netList = new NetworkList(frame, net, networks);
    clear();

    netList.setLocation(580, 295);

    loadProps();
    pluginTable = new Hashtable();
    if(openSimPlugins(simulate, "plugins.cfg")) {
      System.out.println("Loaded plugins successfully.");
    } else {
      System.out.println("No plugins loaded.");
    }
    frame.show();
    toolframe.show();
    netList.show();

    splash.dispose();
  }


	/**
	 * Prompts the user if they want to have the network saved.
	 *
	 * @param   title  is the String that will be displayed.
	 * @return     true to have the next command execute
	 *             false to stop the next command from executing.
	 */
  public boolean promptSave(String title)
  {
    int ans = JOptionPane.showConfirmDialog(this,
        "The current network is unsaved. Would you like to save before proceeding?",
        title,
        JOptionPane.YES_NO_CANCEL_OPTION,
        JOptionPane.WARNING_MESSAGE);
    if(ans == JOptionPane.YES_OPTION)
      return save();
    else if(ans == JOptionPane.NO_OPTION)
      return true;
    else
      return false;
  }


	/**
	 * Function that will open the file and read in the necessary data to 
	 * recreate the original saved <Code>Network</Code>.
	 *
	 */
  public void open()
  {
    if(isSaved || promptSave("Open")) {
      int returnVal = chooser.showOpenDialog(this);
      if(returnVal == JFileChooser.APPROVE_OPTION) {
        File tempFile = chooser.getSelectedFile();
        BufferedReader loadReader;
        LinkedList tempList;
        boolean readable = true;
        boolean eof = false;
        String data;
        String name;
        NetworkFrame tempNet;
        if(!tempFile.exists()) {
          //pop up error dialog box here
          JOptionPane.showMessageDialog(this,
                            "File not found.",
                            "Open failed",
                            JOptionPane.ERROR_MESSAGE);
          readable = false;
        }

        if(readable) {
          clear();
          try  //  IOException
          {
            loadReader = new BufferedReader(new FileReader(tempFile));
            String check = loadReader.readLine();
            if(check.equals(versionString)) {
              String more = loadReader.readLine();
              while(more != null && more.startsWith("NETWORK")) {
                if(more.length() > 8)
                  name = more.substring(8);
                else
                  name = "";
                tempNet = new NetworkFrame(net.getSize(), name);
                for(int row=0;row<8;row++)
                  for(int col=0;col<11;col++) {
                    data = loadReader.readLine();
                    tempNet.getNetwork().setNode(row,col,Elements.getComponent(data));
                  }
                for(int row=0;row<7;row++) {
                  data = loadReader.readLine();
                  tempNet.getNetwork().setVShorts(row,data);
                }
                data = loadReader.readLine();
                String Comment = new String();
                while(!data.equals("~ENDCOMMENT")) {
                  Comment = Comment + data + "\n";
                  data = loadReader.readLine();
                }
                tempNet.setText(Comment.substring(0,Comment.length()-1));
                net.add(tempNet);
                desktop.add(tempNet);
                tempNet.setVisible(false);
                more = loadReader.readLine();
              }
              isSaved = true;
            } else {
              // Version String didn't match.
              JOptionPane.showMessageDialog(this,
                              "Not A Simulator File.",
                              "Open failed",
                              JOptionPane.ERROR_MESSAGE);
            } 
            loadReader.close();
          } catch  (IOException io)
          {
            System.err.println(io);
          }
        }
      }
    }
  }


	/**
	 * Save the current <Code>Network</Code> to the file.
	 *
	 * @return     success or failure.
	 */
  public boolean save() 
  {
    int returnVal = chooser.showSaveDialog(this);
    if(returnVal == JFileChooser.APPROVE_OPTION) {
      File tempFile = chooser.getSelectedFile();
      String path = tempFile.getPath();
      if(!path.endsWith(".net")) {
        path = path + ".net";
        tempFile = new File(path);
      }
      DataOutputStream tempStream;
      String data;
      boolean writeable = true;
      ListIterator SLI = net.listIterator(0);
      NetworkFrame tempNet;
 
      if(tempFile.exists()) {
        int ans = JOptionPane.showConfirmDialog(this,
             "The specified file exists. Overwrite?",
             "Save",
             JOptionPane.YES_NO_OPTION,
             JOptionPane.WARNING_MESSAGE);

        //pop up dialog box here
        if(ans == JOptionPane.NO_OPTION)
          writeable = false;
      }
      if(writeable) {
        try {
  	    tempStream = new DataOutputStream(new FileOutputStream(tempFile));
  	    tempStream.writeBytes(versionString+"\n");
          while(SLI.hasNext()) {
  	      tempNet = (NetworkFrame) SLI.next();
            tempStream.writeBytes("NETWORK " + tempNet.getName() + "\n");
  	      for(int row=0;row<8;row++)
              for(int col=0;col<11;col++) {
                data = tempNet.getNetwork().getNode(row,col).getSaveData();
                tempStream.writeBytes(data + "\n");
              }
            for(int row=0;row<7;row++) {
              data = tempNet.getNetwork().getVShorts(row);
              tempStream.writeBytes(data + "\n");
            }
            data = tempNet.getText();
            tempStream.writeBytes(data + "\n~ENDCOMMENT\n");
          }
          tempStream.close(); 
        } catch  (IOException io) {
  	    System.err.println(io);
        }
      }
      isSaved = true;
      return true;
    } else {
      return false;
    }
  }


	/**
	 * Clears all the <Code>Network</Code> from the linked list of <Code>Networks</Code>.
	 *
	 */
  public static void clear()
  {
    NetworkFrame n;
    SimMemory.clear();
    SimMemory.clearReg();
    simWindow.resetReg();
    SimMemory.clearUsed();
    while(net.getSize() > 0) {
      n = ((NetworkFrame)net.removeLast());
      try {
        n.setClosed(true);
      } catch(Exception ex) {
        System.err.println("Exception!");
      }
      n.dispose();
    }
  }


	/**
	 * Dialog to quit the program.
	 * If it needs it it will prompt you for save, and will save
	 * all the properties that are needed in the next run of the 
	 * program.
	 *
	 */
  public void quit()
  {
    int ans;
    if(! isSaved) {
      ans = JOptionPane.showConfirmDialog(this,
               "Save before exiting?",
               "Quit",
               JOptionPane.YES_NO_CANCEL_OPTION,
               JOptionPane.WARNING_MESSAGE);
      if(ans == JOptionPane.YES_OPTION) {
        if(save()) {
          saveProps();
          System.exit(0);
        }
      } else if(ans == JOptionPane.NO_OPTION) {
        saveProps();
        System.exit(0);
      }
    } else {
      ans = JOptionPane.showConfirmDialog(this,
             "Exit Ladder Simulator?",
             "Quit",
             JOptionPane.YES_NO_OPTION,
             JOptionPane.QUESTION_MESSAGE);
      if(ans == JOptionPane.YES_OPTION) {
        saveProps();
        System.exit(0);
      }
    }
  }


	/**
	 * Number of pages to be printed based on the size of the <Code>Network</Code> list.
	 *
	 * @return     pages of <Code>Network</Code>.
	 */
  public int getNumberOfPages()
  {
    return (net.getSize()/2) + (net.getSize()%2);
  }


	/**
	 * Format of the printable page.
	 *
	 * @param   pageIndex  is the index of pages
	 * @return     the format of the page.
	 */
  public PageFormat getPageFormat(int pageIndex)
  {
    return new PageFormat();
  }


	/**
	 * Gets the printable image.
	 *
	 * @param   pageIndex  tells where the image is
	 * @return     a printable object
	 */
  public Printable getPrintable(int pageIndex)
  {
    return this;
  }


	/**
	 * A thread which runs the print job.  We implemented this to also show
	 * the progress bar.
	 *
	 */
  public void run()
  {
    try {
      pj.print();
    } catch (PrinterException pe) {
    }
    try {
      Thread.sleep(500);
    } catch (InterruptedException e) {
    }
    print.setVisible(false);
  }


  /**
    * Actual print implementation of the <Code>Network</Code>.
    *
    * @param   g  where to print
    * @param   pf  format of the page
    * @param   pageIndex  source of the image
    * @return     an error code used by print method.
    * @exception   PrinterException  is to see that the print job goes well.
    */
  public int print(Graphics g, PageFormat pf, int pageIndex)throws PrinterException
  {
    NetworkFrame tempNet;
    int numNet = pageIndex * 2;
    String data;

    if(numNet > net.getSize())
    {
      return Printable.NO_SUCH_PAGE;
    }
  
    g.translate((int)pf.getImageableX(),(int)pf.getImageableY());
  
    g.setColor(Color.black);
    g.drawString("Network " + numNet, 0, 12);
  
    tempNet = (NetworkFrame) net.get(numNet);
    tempNet.getNetwork().renderNetwork(g,0,20);
    data = tempNet.getText();
    
    StringTokenizer st = new StringTokenizer(data,"\n");
    int yTemp = 225;
    
    while(st.hasMoreElements() && yTemp < 332)
    {
      g.drawString(st.nextToken(),0,yTemp);
      yTemp+= 12;
    }

    if(print.bar.getValue() < numNet)
      print.bar.setValue(numNet);
  
    if((numNet + 1) < net.getSize())
    {
      g.drawString("Network " + (numNet+1), 0, 352);
      tempNet = (NetworkFrame) net.get(numNet+1);
      tempNet.getNetwork().renderNetwork(g,0,364);
      data = tempNet.getText();
      st = new StringTokenizer(data,"\n");
      yTemp = 569;
  
      while(st.hasMoreElements() && yTemp < 636)
      {
        g.drawString(st.nextToken(),0,yTemp);
        yTemp+=12;
      }

      if(print.bar.getValue() < (numNet+1))
        print.bar.setValue(numNet+1);
    }
  
    return Printable.PAGE_EXISTS;
  }

  /**
    * Opens the simulation plugin config file and configures the menu bar with
    * the file's specifications.
    *
    * @return  <CODE>True</CODE> if successful.
    */
  public static boolean openSimPlugins(JMenu pluginMenu, String configFile) {
    URL myURL = ClassLoader.getSystemResource(configFile);
    File tempFile = new File(myURL.getFile());
    ClassLoader sysLoader = ClassLoader.getSystemClassLoader();
    BufferedReader loadReader;
    Class tempClass;
    String data;
    String name;
    boolean success = false;
    if(tempFile!=null && tempFile.exists()) {
      try {   //  IOException
        loadReader = new BufferedReader(new FileReader(tempFile));
        data = loadReader.readLine();
        if(data.equals("#"+versionString)) {
          data = loadReader.readLine();
          while(data != null) {
            if(data.startsWith("#")) {        // Comments or directives
            } else {                   // Class names of Plugins
              try {
                tempClass = sysLoader.loadClass(data);
                if(verbose)
                  System.out.println("Loaded plugin: " + data);
                addPlugin(pluginMenu, (SimulatorPlugin) tempClass.newInstance());
                success = true;
              } catch(Exception e) {
                System.err.println("Failed to load plugin: " + data);
                e.printStackTrace();
              }
            }
            data = loadReader.readLine();
          }
        }
        loadReader.close();
      } catch  (IOException io) {
        System.err.println(io);
      }
    }
    return success;
  }

  /**
    * Add a new plugin to the menu.
    */
  private static void addPlugin(JMenu pluginMenu, SimulatorPlugin plugin) {
    pluginTable.put(plugin.getTitle(), plugin);
    JMenuItem item = new JMenuItem(plugin.getTitle());
    item.addActionListener(frame);
    pluginMenu.add(item);
  }

  /**
    * Loading properties when the program is first ran.
    *
    */
  public static void loadProps() {
    int x, y, h, w;
    String s1, s2;

    try {
      props.load(new FileInputStream("sim.properties"));
    } catch (IOException e) {
    }
    s1 = props.getProperty("frameX");
    s2 = props.getProperty("frameY");
    if(s1 != null && s2 != null) {
      x = Integer.parseInt(s1);
      y = Integer.parseInt(s2);
      frame.setLocation(x,y);
      s1 = props.getProperty("frameWidth");
      s2 = props.getProperty("frameHeight");
      if(s1 != null && s2 != null) {
        w = Integer.parseInt(s1);
        h = Integer.parseInt(s2);
        frame.setSize(w,h);
      }
    }

    s1 = props.getProperty("helpX");
    s2 = props.getProperty("helpY");
    if(s1 != null && s2 != null) {
      x = Integer.parseInt(s1);
      y = Integer.parseInt(s2);
      help.setLocation(x,y);
      s1 = props.getProperty("helpWidth");
      s2 = props.getProperty("helpHeight");
      if(s1 != null && s2 != null) {
        w = Integer.parseInt(s1);
        h = Integer.parseInt(s2);
        help.setSize(w,h);
      }
    }

    s1 = props.getProperty("toolframeX");
    s2 = props.getProperty("toolframeY");
    if(s1 != null && s2 != null) {
      x = Integer.parseInt(s1);
      y = Integer.parseInt(s2);
      toolframe.setLocation(x,y);
    }

    s1 = props.getProperty("netListX");
    s2 = props.getProperty("netListY");
    if(s1 != null && s2 != null) {
      x = Integer.parseInt(s1);
      y = Integer.parseInt(s2);
      netList.setLocation(x,y);
    }

    s1 = props.getProperty("simWindowX");
    s2 = props.getProperty("simWindowY");
    if(s1 != null && s2 != null) {
      x = Integer.parseInt(s1);
      y = Integer.parseInt(s2);
      simWindow.setLocation(x,y);
    }

    s1 = props.getProperty("DialogNONCX");
    s2 = props.getProperty("DialogNONCY");
    if(s1 != null && s2 != null) {
      x = Integer.parseInt(s1);
      y = Integer.parseInt(s2);
      DialogNONC.setLocation(x,y);
    }

    s1 = props.getProperty("DialogOutputX");
    s2 = props.getProperty("DialogOutputY");
    if(s1 != null && s2 != null) {
      x = Integer.parseInt(s1);
      y = Integer.parseInt(s2);
      DialogOutput.setLocation(x,y);
    }

    s1 = props.getProperty("DialogCounterX");
    s2 = props.getProperty("DialogCounterY");
    if(s1 != null && s2 != null) {
      x = Integer.parseInt(s1);
      y = Integer.parseInt(s2);
      DialogCounter.setLocation(x,y);
    }

    s1 = props.getProperty("DialogTimerX");
    s2 = props.getProperty("DialogTimerY");
    if(s1 != null && s2 != null) {
      x = Integer.parseInt(s1);
      y = Integer.parseInt(s2);
      DialogTimer.setLocation(x,y);
    }
 
    s1 = props.getProperty("DialogMulX");
    s2 = props.getProperty("DialogMulY");
    if(s1 != null && s2 != null) {
      x = Integer.parseInt(s1);
      y = Integer.parseInt(s2);
      DialogMul.setLocation(x,y);
    }
 
    s1 = props.getProperty("DialogDivX");
    s2 = props.getProperty("DialogDivY");
    if(s1 != null && s2 != null) {
      x = Integer.parseInt(s1);
      y = Integer.parseInt(s2);
      DialogDiv.setLocation(x,y);
    }
 
    s1 = props.getProperty("DialogNode3X");
    s2 = props.getProperty("DialogNode3Y");
    if(s1 != null && s2 != null) {
      x = Integer.parseInt(s1);
      y = Integer.parseInt(s2);
      DialogNode3.setLocation(x,y);
    }

    s1 = props.getProperty("SimPluginX");
    s2 = props.getProperty("SimPluginY");
    if(s1 != null && s2 != null) {
      x = Integer.parseInt(s1);
      y = Integer.parseInt(s2);
      plugin.setLocation(x,y);
    }

    s1 = props.getProperty("SimEvalDelay");
    if(s1 != null) {
      simWindow.setEvalDelay(Integer.parseInt(s1));
    }

    s1 = props.getProperty("SimRedrawDelay");
    if(s1 != null) {
      simWindow.setRedrawDelay(Integer.parseInt(s1));
    }

    s1 = props.getProperty("SimMinDelay");
    if(s1 != null) {
      simWindow.setMinDelay(Integer.parseInt(s1));
    }

    s1 = props.getProperty("Path");
    if(s1 != null) {
      chooser.setCurrentDirectory(new File(s1));
    }
 }


	/**
	 * Saving properties of the last view.
	 *
	 */
  public static void saveProps() {
    props.setProperty("frameX", Integer.toString(frame.getX()));
    props.setProperty("frameY", Integer.toString(frame.getY()));
    props.setProperty("frameWidth", Integer.toString(frame.getWidth()));
    props.setProperty("frameHeight", Integer.toString(frame.getHeight()));

    props.setProperty("toolframeX", Integer.toString(toolframe.getX()));
    props.setProperty("toolframeY", Integer.toString(toolframe.getY()));

    props.setProperty("netListX", Integer.toString(netList.getX()));
    props.setProperty("netListY", Integer.toString(netList.getY()));

    props.setProperty("helpX", Integer.toString(help.getX()));
    props.setProperty("helpY", Integer.toString(help.getY()));
    props.setProperty("helpWidth", Integer.toString(help.getWidth()));
    props.setProperty("helpHeight", Integer.toString(help.getHeight()));

    props.setProperty("simWindowX", Integer.toString(simWindow.getX()));
    props.setProperty("simWindowY", Integer.toString(simWindow.getY()));

    props.setProperty("DialogNONCX", Integer.toString(DialogNONC.getX()));
    props.setProperty("DialogNONCY", Integer.toString(DialogNONC.getY()));

    props.setProperty("DialogOutputX", Integer.toString(DialogOutput.getX()));
    props.setProperty("DialogOutputY", Integer.toString(DialogOutput.getY()));

    props.setProperty("DialogCounterX", Integer.toString(DialogCounter.getX()));
    props.setProperty("DialogCounterY", Integer.toString(DialogCounter.getY()));

    props.setProperty("DialogTimerX", Integer.toString(DialogTimer.getX()));
    props.setProperty("DialogTimerY", Integer.toString(DialogTimer.getY()));

    props.setProperty("DialogMulX", Integer.toString(DialogMul.getX()));
    props.setProperty("DialogMulY", Integer.toString(DialogMul.getY()));

    props.setProperty("DialogDivX", Integer.toString(DialogDiv.getX()));
    props.setProperty("DialogDivY", Integer.toString(DialogDiv.getY()));

    props.setProperty("DialogNode3X", Integer.toString(DialogNode3.getX()));
    props.setProperty("DialogNode3Y", Integer.toString(DialogNode3.getY()));

    props.setProperty("SimPluginX", Integer.toString(plugin.getX()));
    props.setProperty("SimPluginY", Integer.toString(plugin.getY()));

    props.setProperty("SimEvalDelay", Integer.toString(simWindow.getEvalDelay()));
    props.setProperty("SimRedrawDelay", Integer.toString(simWindow.getRedrawDelay()));
    props.setProperty("SimMinDelay", Integer.toString(simWindow.getMinDelay()));

    props.setProperty("Path", chooser.getCurrentDirectory().getPath());

    try {
      props.store(new FileOutputStream("sim.properties"), versionString);
    } catch (IOException e) {
    }
  }
}