import java.awt.*;
import java.awt.image.*;
import java.lang.*;		// includes math libraries
import java.lang.String.*;
import java.net.*;
import java.util.*;
import javax.swing.*;

/**
  * Class <CODE>MulComponent</CODE>.
  *
  * @author Mike Cammarano
  * @author Charlie So
  * @author Robert Marlowe
  *
  * Multiplication implementation of the 
  * 	<CODE>CircuitComponent</CODE> interface.
  *
  * Class for 16-bit Multiplication Function. This is utilized by 
  * <CODE>MulComponent</CODE>.
  *
  *  The MUL instruction multiplies unsigned value1 (its top node)
  *	and stroes the product a single 16-bit register in the 
  *	bottom node..
  *
  *	Input - 
  *		MUL has one control input (to the top node), which 
  *		initiates the operation when it is ON.
  *	Output- 
  *		MUL produces an output from the top node, which 
  *		echoes the state of the top input.
  *	Top and Middle Node Content -
  * 		Teh top and middle nodes contain value1 and value2,
  *		respectively.  The value in each node may be an 
  * 		integer in the range of 1 ... 65535.
  *	Bottom Node Content -
  *		The register entered in the bottom node is the 
  *		first of two contiguous holding registers where the 
  *		product is stored. The high-order digits in the 
  *		product are stored in the highLine register, and 
  *		the low-order digits are sored in the lowLine 
  *		register.
  *
  *  If the input is ON (= MULtiply value1 and value2).  The MUL  
  *	multiplies unsigned value1 (its top node) to unsigned value2 
  *	(its middle node) and stores the product in two holding 
  *	registers (the bottom node).
  *
  */
public class MulComponent implements CircuitComponent {
  /**
    * The register used to hold this component's top node value.
    */
  private int value1Line;
  /**
    * The register used to hold this component's middle node value.
    */
  private int value2Line;
  /**
    * The register used to hold this component's bottom node value.
    */
  private int highLine;
  /**
    * The register used to hold this component's bottom node value.
    */
  private int lowLine;
  /**
    * Flag indicating whether component should be highlighted. 
    */
  private boolean powered;
  /**
    * Image used to draw this component in a network when it is not 
    * powered.
    */
  private static Image pic;
  /**
    * Image used to draw this component in a network when it is  
    * powered.
    */
  private static Image picLit;
  /**
    * Cursor image used for this component during drag and drop.
    */
  private static Image cursorImage;
  /**
    * Icon used for this component in the component bin.
    */
  private static ImageIcon icon;
  /**
    * ToolTip text used for this component.
    */
  private static String tip = "MUL";
  /**
    * The simulations internal name for this component.
    */
  private static String name = "MUL";

  static {
      // This static initializer loads all the image files needed
      URL myURL;
      ImageIcon myIcon;
      myURL = ClassLoader.getSystemResource("BigMul.gif");
    	myIcon = new ImageIcon(myURL);
    	pic = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("BigMulOn.gif");
        myIcon = new ImageIcon(myURL);
        picLit = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("CursorMul.gif");
        myIcon = new ImageIcon(myURL);
        cursorImage = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("ToolMul.gif");
        icon = new ImageIcon(myURL);
  }

  /**
    * Construct a 3-rung Multiply function using default value1, 
    *	value2, and product values.
    */
  public MulComponent() {
    value1Line 	= 0;	// initial registers used
    value2Line 	= 1;
    highLine 	= 2;
    lowLine     = 3;	// SimMemory automatically initializes regs
  }

  /**
    * @return 3, the number of ladder logic rungs occupied by this 
    * component.
    */
  public int size() {
    return 3;
  }

  /**
    * Performs one evaluation of this component using the specified
    * input logic levels. If the 1st rung of the MUL is logic-low,
    * then the MUL resets. Otherwise, the current state of the
    * first rung is Echoed to the output.
    *
    * Only the top rung	of the input (b[0]) and the top node of the 
    *	output boolean[0] is affected by the operation.
    *
    * @param b - array containing input logic levels of the rungs
    *            occupied by this component.  Only b[0] is observed.
    *
    * @return the output logic levels from this component.
    */
  public boolean[] step(boolean[] b) {
    	int tempValue1 = SimMemory.GetReg(value1Line);
    	int tempValue2 = SimMemory.GetReg(value2Line);
	int tempProd = 0;
	int tempHigh = 0;
	int tempLow  = 0;
        powered = b[0];
	boolean tempArr[];
	tempArr = new boolean[] {false, false, false};
	if(b[0] && (tempValue1 < 65535 && tempValue1 > 0) 
		&& (tempValue2 < 65535 && tempValue2 > 0)) {
	   tempProd = tempValue1 * tempValue2;
	   tempHigh = tempProd >> 16;	// Bit-Shift Right
	   tempLow  = tempProd & 65535;	// Bit-Wise AND
	   SimMemory.SetReg(highLine, tempHigh);
	   SimMemory.SetReg(lowLine,  tempLow);  // write to Memory 
	   tempArr[0] = true;			 // ( 3rd Rung )
	}
  	return tempArr;				
  }						 

  /**
    * Displays a modal dialog box to allow the user to edit the
    *  	parameters of this component, such as which registers are 
    * 	used. This method will initialize the state of the 
    * 	existing instance of <CODE>MulDialog</CODE> in 
    * 	<CODE>Master</CODE>, show it, and then update value1 and 
    *   value2 values to be used in the operation of the Add 
    *   based on the user's selections from the dialog.
    */
  public void edit() {
    MulDialog a = Master.DialogMul;
    SimMemory.deleteUser(highLine, this);
    SimMemory.deleteUser(lowLine, this);
    a.setState(value1Line,value2Line, highLine, lowLine, this);
    a.setVisible(true);
    value1Line 	= a.getValue1Line();
    value2Line 	= a.getValue2Line();
    highLine 	= a.getHighLine();
    lowLine 	= a.getLowLine();
    SimMemory.addUser(highLine, this);
    SimMemory.addUser(lowLine, this);
  }


  /**
    * Clears any registers marked as used by this component prior to
    * its deletion.
    */
  public void clear() {
    SimMemory.deleteUser(highLine, this);
    SimMemory.deleteUser(lowLine, this);
  }


  /**
    * Resets this component, and initializes values
    * in <CODE>SimMemory</CODE>.
    */
  public void reset() {
	// Not needed in this component
  }

  /**
    * Returns the text to be used as a tool tip when this component
    * is placed in the component bin.
    *
    * @return the string to display.
    */
  public String getTip() {
    return tip;
  }

  /**
    * Returns the name of this component to be used internally in 
    * saved files and in drag and drop operations.
    *
    * @return the string to use.
    */
  public String getName() {
    return name;
  }

  /**
    * Returns the Icon to be used for this component in the component 
    * bin.
    *
    * @return the icon.
    */
  public ImageIcon getToolIcon() {
    return icon;
  }

  /**
    * Returns a one line string representing the state of this 
    * component for use in a save file. The string will be of the 
    * form: <BR>
    * Add 
    * <I>value1-register value2-register highLine-register 
    * lowLine-register</I>.
    *
    * @return the parameter string.
    */
  public String getSaveData() {
    return name + " " + value1Line + " " + value2Line 
		+ " " + highLine   + " " + lowLine;
  }

  /**
    * Returns whether this component must be the rightmost component 
    * on any rung.  For example, an output coil terminates a rung; 
    * a switch does not.
    *
    * @return <CODE>False</CODE> since MULs don't terminate rungs.
    */
  public boolean terminatesRung() {
    return false;
  }

  /**
    * Returns the image to use as a mouse cursor for this component 
    * during a drag and drop operation.
    *
    * @return the cursor image.
    */
  public Image getImage() {
    return cursorImage;
  }

  /**
    * Returns a new instance of this component created using the 
    * default constructor.
    * This method is used by the <CODE>getComponent</CODE> factory 
    * method of <CODE>Elements</CODE>.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance() {
    MulComponent x = new MulComponent();
    SimMemory.addUser(x.highLine, x);
    SimMemory.addUser(x.lowLine, x);
    return x;
  }

  /**
    * Returns a new instance of this component, using the specified 
    * initial parameters.  This method is used by the 
    * <CODE>getComponent</CODE> factory method of <CODE>Elements</CODE> 
    * to instantiate new components during a file open.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance(int[] a) {	
    MulComponent x = new MulComponent();
    x.value1Line = a[0];
    x.value2Line = a[1];
    x.highLine   = a[2];
    x.lowLine    = a[3];
    SimMemory.addUser(x.highLine, x);
    SimMemory.addUser(x.lowLine, x);
    return x;
  }

  /**
    * Draws this component and any associated parameters at the 
    * specified coordinates in the graphics context. The value1 value 
    * is written in the top third of the MUL, the value2 value is 
    * written in the middle third of the MUL and the
    * product values are written in the bottom third of the MUL.
    *
    * @param g  the graphics context in which to draw.
    * @param x  the x-coordinate of the northwest corner.
    * @param y  the y-coordinate of the northwest corner.
    * @param o  the image observer to be notified of image drawing.
    */
  public void drawComponent(Graphics g,int x,int y,ImageObserver o){
    if(Master.isSimActive && powered) {
      // Draw a powered component
      g.drawImage(picLit, x, y, o);
      g.setColor(Network.darkGreen);
    } 
    else {
      // Draw a non-powered component
      g.drawImage(pic, x, y, o);
    }

    // Draw Register assignments onto component at rung
    String text;
    
    // Draw the value1 value in the top third of the component
    text = "     " + Integer.toString(value1Line);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+16);

    // Draw the value2 value in the middle third of the component
    text = "     " + Integer.toString(value2Line);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+40);

    // Draw the High-order value in the bottom third of the comp
    text = "     " + Integer.toString(highLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+4, y+64);

    // Draw the Low-order value in the bottom third of the comp
    text = "     " + Integer.toString(lowLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+16, y+64);

    g.setColor(Color.black);	// in case the color got set to 
				// dark green set back to black
  }
}




/*  To compile and run this Class
	first  - make sure that the class's name is also the file name 
		 -- observe case
	second - at the command prompt in the working directory of the 
		 SDK compile the class.  
			EX. L:\work>N:jdk1.2.1\bin\javac *.java
		 In this case all file *.java will be linked, 
		 therefore take care that they are related or delete 
		 unneeded files.
	third  - run the Class by 
			EX. L:\WORK>N:jdk1.2.1\bin\java figure1511
		 where figure1511 is both the name of the class & file.

    For creating links to, for example, methods ...

	{@link #reset() <CODE>reset</CODE>.}
*/
		