import java.io.*;
import java.awt.*;
import java.awt.dnd.*;
import java.awt.datatransfer.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;


/**
  * An instance of <Code>Network</Code> is a graphical representation of
  * circuit diagram on one network. <Code>Network</Code> is also the target
  * of a drag and drop method.
  *
  * @author Mike Cammarano
  * @author Charlie So
  */

public class Network extends JPanel
                     implements DropTargetListener, ActionListener
{
  /**
    * Small font to place writing on either side of the dropped elements
    * to denote the input, output, or register usage.
    */
  public final static Font myFont = new Font("SansSerif", Font.PLAIN, 7);
  /**
    * Slightly larger font used for the M, I, O, F, or R symbols used to
    * denote the type of line a component is attached to.
    */
  public final static Font bigFont = new Font("SansSerif", Font.PLAIN, 9);
  /**
    * Specified color to denote the charging of each element.
    * Dark green means the element is charged, black means it is not.
    */
  public final static Color darkGreen = new Color(0, 128, 0);

  /**
    * An array of <Code>CircuitComponent</Code>.
    */
  private CircuitComponent[][] node;
  /**
    * An array of booleans that will stand for all the vertical shorts.
    */
  private boolean[][] vshort;
  /**
    * An instance of EmptyComponent to be used in the <Code>Network</Code>.
    */
  private EmptyComponent empty = new EmptyComponent();
  /**
    * An instance of HShortComponent to be used in the <Code>Network</Code>.
    */
  private HShortComponent hshort = new HShortComponent();
  /**
    * An instance of NullComponent to be used in the <Code>Network</Code>.
    */
  private NullComponent nullComponent = new NullComponent();
  /**
    * An instance of EndRowComponent to be used in the <Code>Network</Code>.
    */
  private EndRowComponent EndRow = new EndRowComponent();
  /**
    * Variables to keep track of which element was selected by denoting the 
    * row and column of the element selected.
    */
  private int selectedRow, selectedCol;


  /**
   * A constructor for <Code>Network</Code>. It resets all the components
   * on the diagram. Drop target and mouse listener for drag and drop are
   * instantiated in this constructor.
   */
  public Network() 
  {
    node = new CircuitComponent[8][11];
    vshort = new boolean[7][10];
    for(int i=0;i<8;i++)
      for(int j=0;j<11;j++)
        node[i][j] = empty;
    for(int i=0;i<7;i++)
      for(int j=0;j<10;j++)
        vshort[i][j] = false;

    new DropTarget(this, DnDConstants.ACTION_COPY_OR_MOVE, this);
    // DnD is weird. Just have to instantiate a DropTarget, without maintaining
    // any reference to it.

    // POPUP stuff
    final JPopupMenu popNode = new JPopupMenu();
    JMenuItem item;
    popNode.add(item = new JMenuItem("Edit"));
    item.addActionListener(this);
    popNode.add(item = new JMenuItem("Delete"));
    item.addActionListener(this);
    popNode.setInvoker(this);
    popNode.addPopupMenuListener(new PopupMenuListener() {
      public void popupMenuCanceled(PopupMenuEvent e) {}
      public void popupMenuWillBecomeInvisible(PopupMenuEvent e) {}
      public void popupMenuWillBecomeVisible(PopupMenuEvent e) {}
    });

    final JPopupMenu popShort = new JPopupMenu();
    popShort.add(item = new JMenuItem("Place short"));
    item.addActionListener(this);
    popShort.add(item = new JMenuItem("Remove short"));
    item.addActionListener(this);
    popShort.setInvoker(this);
    popShort.addPopupMenuListener(new PopupMenuListener() {
      public void popupMenuCanceled(PopupMenuEvent e) {}
      public void popupMenuWillBecomeInvisible(PopupMenuEvent e) {}
      public void popupMenuWillBecomeVisible(PopupMenuEvent e) {}
    });

    this.addMouseListener( new MouseAdapter() {
      public void showPopup(MouseEvent e) {
        if(!Master.isSimActive) {
          int x = e.getX();
          int y = e.getY();
          if(x > 8 && x < 458 && y > 8) {
            if( ((x-8)%41 > 3) && ((x-8)%41 < 37)) {
              x = col(x);
              y = row(y);
              if(!(node[y][x] instanceof EmptyComponent)&&
                 !(node[y][x] instanceof EndRowComponent)) {
                while(node[y][x] instanceof NullComponent)
                  y--;
                selectedCol = x;
                selectedRow = y;
                popNode.show(e.getComponent(), e.getX(), e.getY());
              }
            } else if( (((x-8)%41 <= 3) || ((x-8)%41 >= 37))
                                && (y>20 || y<188) ) {
              selectedCol = col(x-4);
              selectedRow = row(y-12);
              popShort.show(e.getComponent(), e.getX(), e.getY());
            }
          }
        }
      }

      public void mousePressed(MouseEvent e) {
        if(e.isPopupTrigger()) {
          showPopup(e);
        }
      }

      public void mouseReleased(MouseEvent e) {
        if(e.isPopupTrigger()) {
          showPopup(e);
        }
      }

      public void mouseClicked(MouseEvent e) {
        if(!(Master.isSimActive || e.isPopupTrigger())) {
          int x = e.getX();
          int y = e.getY();
          if(x > 8 && x < 458 && y > 8) {
            if( ((x-8)%41 > 3) && ((x-8)%41 < 37)) {
              x = col(x);
              y = row(y);
              if(node[y][x] instanceof EmptyComponent) {
                node[y][x] = hshort;
                repaint(300, 0, 0, 468, 200);
              }
            } else if( (((x-8)%41 <= 3) || ((x-8)%41 >= 37))
                                && (y>20 || y<188) ) {
              vshort[row(y-12)][col(x-4)] = true;
              repaint(300, 0, 0, 468, 200);
            }
          }
        }
      }
    });
    // End of POPUP stuff

    setMinimumSize(new Dimension(468, 204));
    setPreferredSize(new Dimension(468, 204));
  }


  /**
   * Determines the number of rows that can be placed in the <Code>Network</Code>
   * from the max y value.
   *
   * @param   y  max number of pixels in y direction.
   * @return     number or rows determined from the y coordinate.
   */
  private int row(int y) {
    return (y-8)/24;
  }


  /**
   * Determines the number of columns that can be placed in the <Code>Network</Code>
   * from the max x value.
   *
   * @param   x  max number os pixels in x direction.
   * @return     number of columns available from the x coordinate.
   */
  private int col(int x) {
    return (x-8)/41;
  }


  /**
   * Action listener to the mouse. This listener determines what action to
   * take when the right mouse button is used.
   *
   * @param   e  is an action event.
   */
  public void actionPerformed(ActionEvent e) {
    Master.isSaved = false;
    String s = e.getActionCommand();
    if(s.equals("Delete")) {
      delete(selectedRow, selectedCol);
    } else if (s.equals("Edit")) {
      node[selectedRow][selectedCol].edit();
    } else if (s.equals("Place short")) {
       vshort[selectedRow][selectedCol] = true;
    } else if (s.equals("Remove short")) {
       vshort[selectedRow][selectedCol] = false;
    }
    repaint(300, 0, 0, 468, 200);
  }
  

  /**
   * Runs through each of the components in the network
   * and evaluates.
   */
  public void simulate() {
    boolean[] tempcol = new boolean[8];
    boolean[] in = new boolean[8], out;
    boolean temp = false;
    int row, col, k;
    for(row=0; row<8; row++)
      tempcol[row] = true;
    for(col=0; col<11; col++) {
      // evaluate nodes in column
     	for(row=0; row<8; row++) {
     	  for(k=0; k<node[row][col].size(); k++)
     	    in[k] = tempcol[row+k];
        out = node[row][col].step(in);
     	  for(k=0; k<node[row][col].size(); k++)
     	    tempcol[row+k] = out[k];
     	}

      if(col < 10) {
        // evaluate vshorts
        for(row=0; row<7; row++) {
          temp = tempcol[row];
          for(k=row; k<7 && vshort[k][col]; k++) {
            temp = temp || tempcol[k+1];
          }
          tempcol[row] = temp;
          for(k=row; k<7 && vshort[k][col]; k++) {
            tempcol[k+1] = temp;
          }
        }
      }
    }
  }


  /**
   * <Code>delete</Code> deletes an element from the rung.
   *
   * @param   row  specific row where the element exists
   * @param   col  specific column where the element exists
   * @return       true when it finishes.
   */
  private boolean delete(int row, int col) {
    CircuitComponent comp = node[row][col];
    for(int r = row; r < (row+comp.size()); r++)
      node[r][col] = empty;
    if(comp.terminatesRung()) {
      for(int r = row; r< (row+comp.size()); r++)
        for(int c = col + 1; c < 11; c++)
          node[r][c] = empty;
    }
    comp.clear();
    repaint(300, 0, 0, 468, 200);
    return true;
  }


  /**
   * Resets all the nodes in the network to the original
   * state for new simulation purposes.
   */
  public void reset()
  {
    for(int row = 0; row < 8; row++)
      for(int col = 0; col < 11; col++)
        node[row][col].reset();
  }


  /**
   * Sets a particular node in the <Code>Network</Code> array to be 
   * specified component.
   *
   * @param   row  specific row where the component exists.
   * @param   col  specific column where the component exists
   * @param   comp  the component to be inserted
   */
  public void setNode(int row, int col, CircuitComponent comp)
  {
    node[row][col] = comp;
  }


  /**
   * Gets the particular type of component that exists at the
   * particular spot in the <Code>Network</Code> array.
   *
   * @param   row  specific row where the component exists
   * @param   col  specific column where the component exists
   * @return     an instance of <Code>CircuitComponent</Code>
   *             that exists in the particular row and column.
   */
  public CircuitComponent getNode(int row, int col)
  {
    return node[row][col];
  }


  /**
   * Converts the verticle short information of a particular row to
   * a string.
   *
   * @param   row  specified row where the information about verticle shorts
   *               is needed.
   * @return     a String of 1's and 0's that denote the verticle shorts.
   */
  public String getVShorts(int row)
  {
    String data = new String();
    for(int i=0; i<10; i++) {
      data = data.concat(vshort[row][i] ? "1" : "0");
    }
    return data;
  }


  /**
   * From the string data, sets each verticle shorts that need to be
   * set in the <Code>Network</Code>.
   *
   * @param   row  particular row where the verticle shorts information goes.
   * @param   data  string of 1's and 0's that denote where the shorts are.
   */
  public void setVShorts(int row, String data)
  {
    for(int i=0; i<10; i++) {
      vshort[row][i] = (data.charAt(i) == '1' ? true : false);
    }
  }

  public void dragEnter(DropTargetDragEvent e) {}
  public void dragExit(DropTargetEvent e) {}
  public void dragOver(DropTargetDragEvent e) {}
  public void dropActionChanged(DropTargetDragEvent e) {}

  /**
   * Drag and drop code.  This one is the "drop" side of the code.
   * The function will accept the data being transfered from the
   * <Code>Elements</Code> and use the data to place each of the 
   * <Code>CircuitComponent</Code> into the <Code>Network</Code>.
   *
   * @param   e  is a DropTargetDropEvent.
   */
  public void drop(DropTargetDropEvent e)
  {
    try {
      DataFlavor stringFlavor = DataFlavor.stringFlavor;
 	Transferable tr = e.getTransferable();
 	if(e.isDataFlavorSupported(stringFlavor)) {
	  String componentname = (String)tr.getTransferData(stringFlavor);
 	  e.acceptDrop(DnDConstants.ACTION_COPY_OR_MOVE);
	  e.dropComplete(true);
        int y = (int)e.getLocation().getY();
        int x = (int)e.getLocation().getX();
        if(componentname.equals("VSHORT")) {
          if( (((x-8)%41 <= 3) || ((x-8)%41 >= 37)) && (y>20 || y<188) ) {
            vshort[row(y-12)][col(x-4)] = true;
          }
          repaint(300, 0, 0, 468, 200);
        } else if(8<=y && y<=192 && 8<=x && x<=466) {
          CircuitComponent comp = Elements.getComponent(componentname);
          x = col(x);
          y = row(y);
          if(y+comp.size()<=8) {
            boolean onRung = true;
            boolean clobbers = false;
            for(int i=y; i<y+comp.size(); i++) {
              if(node[i][x] instanceof EndRowComponent)
                onRung = false;
            }
            if(comp.terminatesRung()) {
              for(int i=y; i<y+comp.size(); i++) {
                for(int j=x+1; j<11; j++)
                  if(!(node[i][j] instanceof EmptyComponent))
                    clobbers = true;
              }
            }
            // onRung designates whether the component would extend past
            // the end of of rung terminated early by an output coil.
            // clobbers indicates whether the component would terminate
            // a rung that is already occupied by other components
            if(onRung && !clobbers) {
              if(!(node[y][x] instanceof HShortComponent ||
                   node[y][x] instanceof EmptyComponent)) {
                int y2 = y;
                while(node[y2][x] instanceof NullComponent)
                  y2--;
                if(delete(y2, x)) {
                  // components above deleted, insert new
                  Master.isSaved = false;
                  for(int i = y+1; i<y+comp.size(); i++) {
                    if(!(node[i][x] instanceof NullComponent)) {
                      delete(i, x);
                    }
                  }
                  node[y][x] = comp;
                  for(int i = y+1; i<y+comp.size(); i++) {
                    node[i][x] = nullComponent;
                  }
                  if(comp.terminatesRung()) {
                    for(int i = y; i< y+comp.size(); i++)
                      for(int j = x+1; j<11; j++) {
                        node[i][j] = EndRow;
                      }
                  }
                }
              } else {
                // go ahead and place component
                Master.isSaved = false;
                for(int i = y+1; i<y+comp.size(); i++) {
                  if(!(node[i][x] instanceof NullComponent)) {
                    delete(i, x);
                  }
                }
                node[y][x] = comp;
                for(int i = y+1; i<y+comp.size(); i++)
                  node[i][x] = nullComponent;
                if(comp.terminatesRung()) {
                  for(int i = y; i< y+comp.size(); i++)
                    for(int j = x+1; j<11; j++) {
                      node[i][j] = EndRow;
                    }
                }
              }
            }
          }
          repaint(300, 0, 0, 468, 200);
        }
	} else {
 	  e.rejectDrop();
 	}
    } catch(IOException ioe) {
      ioe.printStackTrace();
    } catch(UnsupportedFlavorException ufe) {
      ufe.printStackTrace();
    }
  }
 	    				

  /**
   * Paints the components in to the <Code>Network</Code> frame.
   *
   * @param   g  is a graphics panel, or where to place the picture.
   */
  public void paintComponent(Graphics g) 
  {
    super.paintComponent(g);
    renderNetwork(g, 0, 0);
  }


  /**
   * Draws a <Code>Network</Code>.
   *
   * @param   g  a graphics panel to place the graphics
   * @param   x  x-coordinate of upper left corner
   * @param   y  y-coordinate of upper left corner
   */
  public void renderNetwork(Graphics g, int x, int y)
  {
    /*
    float[] dash = {1.0f, 1.0f};
    BasicStroke bs = new BasicStroke(1.0f, BasicStroke.CAP_SQUARE,
                                     BasicStroke.JOIN_MITER, 10.0f,
                                     dash, 0.0f);
    Stroke ds = ((Graphics2D) g).getStroke();
    */
    g.setColor(Color.black);
    g.setFont(myFont);
    g.drawLine(x+7,y+8,x+7,y+192);
    for(int i=0;i<8;i++) {
      for(int j=0;j<11;j++) {
        if(node[i][j].getImage() != null) {
          node[i][j].drawComponent(g, x+8+41*j, y+8+24*i, this);
        }
      }
    }
    for(int i=0;i<7;i++)
      for(int j=0;j<10;j++) {
        if(vshort[i][j]) {
          g.drawLine(x+48+41*j, y+19+24*i, x+48+41*j, y+43+24*i);
        } else {
          g.drawLine(x+48+41*j, y+19+24*i, x+48+41*j, y+21+24*i);
        }
      }
    for(int j=0;j<10;j++)
      g.drawLine(x+48+41*j, y+187, x+48+41*j, y+189);
  }
}