import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

/**
  * <Code>NetworkList</Code> keeps track of all the <Code>NetworkFrame</Code>.
  *
  * @author Mike Cammarano
  * @author Charlie So
  */

class NetworkList extends JDialog
                  implements ActionListener, InternalFrameListener
{
  /**
    * The checkbox controlling whether this window is visible or not.
    */
  private JCheckBoxMenuItem check;
  /**
    * The actual list(model) of <CODE>NetworkFrame</CODE>s.
    */
  private LinkedListModel net;
  /**
    * The UI list of <CODE>NetworkFrame</CODE>s.
    */
  private final JList netList;


  /**
   * Main constructor for <Code>NetworkFrame</Code>. This creates a window to 
   * display the network as well as a small window to keep track of all the 
   * networks.
   *
   * @param   owner  would be the main window from <Code>Master</Code>
   * @param   networks  is a linked list of all the <Code>Network</Code>
   * @param   ch  is a check box
   */
  public NetworkList(Frame owner, LinkedListModel networks, 
                     JCheckBoxMenuItem ch) {
    super(owner, "Network Manager");
    net = networks;
    check = ch;
    JPanel p = new JPanel();
    p.setLayout(new BoxLayout(p, BoxLayout.Y_AXIS));
    netList = new JList(net);
    netList.setPrototypeCellValue("Network 9999");

    final JPopupMenu pop = new JPopupMenu();
    JMenuItem item;
    item = new JMenuItem("Rename");
    item.addActionListener(this);
    pop.add(item);
    item = new JMenuItem("Insert Above");
    item.addActionListener(this);
    pop.add(item);
    item = new JMenuItem("Delete");
    item.addActionListener(this);
    pop.add(item);
    pop.setInvoker(netList);

    MouseListener mouseListener = new MouseAdapter() {
        public void mouseClicked(MouseEvent e) {
            if (e.getClickCount() == 2) {
                int index = netList.locationToIndex(e.getPoint());
                if(index != -1) {
                  NetworkFrame nf = ((NetworkFrame)net.get(index));
                  if(nf.isVisible()) {
                    nf.setVisible(false);
                    Master.frame.repaint();
                  }
                }
             } else if(e.getClickCount() == 1) {
                int index = netList.locationToIndex(e.getPoint());
                if(index != -1) {
                  NetworkFrame nf = ((NetworkFrame)net.get(index));
                  nf.show();
                }
             }
        }

        public void mousePressed(MouseEvent e) {
          if(e.isPopupTrigger())
            showPopup(e);
        }

        public void mouseReleased(MouseEvent e) {
         if(e.isPopupTrigger())
           showPopup(e);
        }

        public void showPopup(MouseEvent e) {
          pop.show(e.getComponent(), e.getX(), e.getY());
        }
    };
    netList.addMouseListener(mouseListener);

    JScrollPane scroller = new JScrollPane(netList);
    p.add(scroller);

    JPanel buttons = new JPanel();
    buttons.setLayout(new GridLayout(3,1));
    JButton b;
    b = new JButton("Close All");
    b.addActionListener(this);
    buttons.add(b);
    p.add(buttons);
    b = new JButton("Add Network");
    b.addActionListener(this);
    buttons.add(b);
    b = new JButton("Simulate");
    b.addActionListener(this);
    buttons.add(b);

    JPanel buttonHolder = new JPanel();
    buttonHolder.add(buttons);
    p.add(buttonHolder);
    getContentPane().add(p);

    pack();
    setSize(160, 250);
    setResizable(false);
  }


  /**
   * Takes care of all the action caused by the mouse in the windows.
   *
   * @param   e  is an ActionEvent.
   */
  public void actionPerformed(ActionEvent e)
   {
     NetworkFrame network;
     String s = e.getActionCommand();
     int lastValid = net.getSize();
     if(s.equals("Close All")) {
       for(int i=0; i<net.getSize(); i++)
         ((NetworkFrame)net.get(i)).setVisible(false);
     } else if(s.equals("Add Network")) {
       Master.isSaved = false;
       network = new NetworkFrame(lastValid, "");
       lastValid++;
       net.add(network);
       Master.desktop.add(network);
       network.show();
     } else if(s.equals("Simulate")) {
       Master.simWindow.setVisible(true);
     } else if(s.equals("Delete")) {
       lastValid = netList.getSelectedIndex();
       if(lastValid != -1) {
         int ans = JOptionPane.showConfirmDialog(this,
               "Delete Network " + lastValid + "?",
               "Delete",
               JOptionPane.YES_NO_OPTION,
               JOptionPane.WARNING_MESSAGE);
         if(ans == JOptionPane.YES_OPTION) {
           Master.isSaved = false;
           network = (NetworkFrame) net.remove(lastValid);
           try {
             network.setClosed(true);
           } catch(Exception ex) {
             System.err.println("Exception!");
           }
           network.dispose();
           lastValid--;
           if(lastValid < 0)
             lastValid = 0;
           for(;lastValid < net.getSize(); lastValid++)
             ((NetworkFrame)net.get(lastValid)).setName(lastValid);
         }
       }
     } else if(s.equals("Insert Above")) {
        lastValid = netList.getSelectedIndex();
        if(lastValid != -1) {
          Master.isSaved = false;
          network = new NetworkFrame(lastValid, "");
          net.add(lastValid, network);
          Master.desktop.add(network);
          network.show();
          for(;lastValid < net.getSize(); lastValid++)
            ((NetworkFrame)net.get(lastValid)).setName(lastValid);
        }
     } else if(s.equals("Rename")) {
        lastValid = netList.getSelectedIndex();
        if(lastValid != -1) {
          Master.isSaved = false;
          network = (NetworkFrame) net.get(lastValid);
          String newName = (String) JOptionPane.showInputDialog(this,
                      "Enter name for Network " + lastValid,
                      "Rename Network", JOptionPane.PLAIN_MESSAGE,
                      null, null, network.getName());
          if(newName != null)
            network.setName(newName);
        }
     }

     repaint();
     Master.frame.repaint();
   }


  /**
   * To check an see if the <Code>NetworkFrame</Code> is supposed to 
   * be visible or not and take action according to the result.
   *
   * @param   b  is a boolean, true for visible
   *                           false for not visible
   */
  public void setVisible(boolean b) {
    check.setSelected(b);
    super.setVisible(b);
  }


  /**
   * Checks to see if any of the internal frame has beenselected
   * and denote the selection in the network list window.
   *
   * @param   e  is an action event.
   */
  public void internalFrameActivated(InternalFrameEvent e) {
    NetworkFrame nf = (NetworkFrame) e.getSource();
    netList.setSelectedValue(nf.getTitle(), true);
  }

//  These functions needed to be defined, but not used in this program.

  public void internalFrameClosed(InternalFrameEvent e) {}
  public void internalFrameClosing(InternalFrameEvent e) {}
  public void internalFrameDeactivated(InternalFrameEvent e) {}
  public void internalFrameIconified(InternalFrameEvent e) {}
  public void internalFrameDeiconified(InternalFrameEvent e) {}
  public void internalFrameOpened(InternalFrameEvent e) {}
}