import java.awt.*;
import java.awt.image.*;
import java.lang.String.*;
import java.net.*;
import javax.swing.*;

/**
  * Output coil implementation of the <CODE>CircuitComponent</CODE> interface.
  *
  * @author Mike Cammarano
  * @author Charlie So
  */
public class OPComponent implements CircuitComponent 
{
  /**
    * The memory line (either a numbered register, the output line, or the flag line)
    * that this output writes its state to.
    */
  private int line;
  /**
    * The individual bit of a memory line that this output writes its state to.
    */
  private int bit;
  /**
    * Whether this component should be drawn highlighted on the next repaint.
    */
  private boolean powered;
  /**
    * Image used to draw this component in a network when it is not powered.
    */
  private static Image pic;
  /**
    * Image used to draw this component in a network when it is powered.
    */
  private static Image picLit;
  /**
    * Cursor image used for this component during drag and drop.
    */
  private static Image cursorImage;
  /**
    * Icon used for this component in the component bin.
    */
  private static ImageIcon icon;
  /**
    * ToolTip text used for this component.
    */
  private static String tip = "Output Coil";
  /**
    * The simulations internal name for this component.
    */
  private static String name = "OP"; 

  static {
      // This static initializer loads all the image files needed
      URL myURL = ClassLoader.getSystemResource("ToolOP.gif");
    	icon = new ImageIcon(myURL);
    	pic = icon.getImage();
      myURL = ClassLoader.getSystemResource("ToolOPOn.gif");
      ImageIcon myIcon = new ImageIcon(myURL);
      picLit = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("CursorOP.gif");
      myIcon = new ImageIcon(myURL);
      cursorImage = myIcon.getImage();
  }

  /**
    * Construct an output coil using the default output line and bit.
    */
  public OPComponent() {
    line = SimMemory.OUTPUT;
    bit = 0;
  }

  /**
    * Returns 1, the number of ladder logic rungs occupied by this component.
    *
    * @return 1, the number of ladder logic rungs occupied by this component.
    */
  public int size() {
    return 1;
  }

  /**
    * Performs one evaluation of this component using the specified input
    * logic levels. This will write the input logic level to the memory
    * line and bit associated with this output coil.
    *
    * @param b  array containing input logic levels of the rungs
    *           occupied by this component.
    *
    * @return the output logic levels from this component.
    */
  public boolean[] step(boolean[] b) {
    SimMemory.SetBit(line, bit, b[0]);
    powered = b[0];
    return b;
  }

  /**
    * Displays a modal dialog box to allow the user to edit the parameters
    * of this component. This method will initialize the state of the existing
    * instance of <CODE>OutputDialog</CODE> in <CODE>Master</CODE>, show it, and
    * then update the memory line and bit associated with this output based on
    * the user's selections in that dialog.
    */
  public void edit() {
    if(line < 1024) {
      SimMemory.deleteUser(line, this);
    }
    Master.DialogOutput.setState(line, bit, this);
    Master.DialogOutput.setVisible(true);
    line = Master.DialogOutput.getLine();
    if(line < 1024) {
      SimMemory.addUser(line, this);
    }
    bit = Master.DialogOutput.getBit();
  }

  /**
    * Clears any registers marked as used by this component prior to
    * its deletion.
    */
  public void clear() {
    if(line < 1024) {
      SimMemory.deleteUser(line, this);
    }
  }

  /**
    * Resets this component.
    */
  public void reset() {
    powered = false;
  }

  /**
    * Returns the text to be used as a tool tip when this component
    * is placed in the component bin.
    *
    * @return the string to display.
    */
  public String getTip() {
    return tip;
  }

  /**
    * Returns the name of this component to be used internally in saved files
    * and in drag and drop operations.
    *
    * @return the string to use.
    */
  public String getName() {
    return name;
  }

  /**
    * Returns the Icon to be used for this component in the component bin.
    *
    * @return the icon.
    */
  public ImageIcon getToolIcon() {
    return icon;
  }

  /**
    * Returns a one line string representing the state of this component for use in
    * a save file. The string will be of the form:
    * <BR>
    * OP <I>memory-line memory-bit</I>.
    *
    * @return the parameter string.
    */
  public String getSaveData() {
    return name + " " + line + " " + bit;
  }

  /**
    * Returns whether this component must be the rightmost component on any rung.
    * For example, an output coil terminates a rung; a switch does not.
    *
    * @return <CODE>True</CODE> since output coils terminate rungs.
    */
  public boolean terminatesRung() {
    return true;
  }

  /**
    * Returns the image to use as a mouse cursor for this component during
    * a drag and drop operation.
    *
    * @return the cursor image.
    */
  public Image getImage() {
    return cursorImage;
  }

  /**
    * Returns a new instance of this component created using the default constructor.
    * This method is used by the <CODE>getComponent</CODE> factory method of
    * <CODE>Elements</CODE>.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance() {
    OPComponent x = new OPComponent();
    if(x.line < 1024) {
      SimMemory.addUser(x.line, x);
    }
    return x;
  }

  /**
    * Returns a new instance of this component, using the specified initial parameters.
    * This method is used by the <CODE>getComponent</CODE> factory method of
    * <CODE>Elements</CODE> to instantiate new components during a file open.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance(int[] a) {
    OPComponent x = new OPComponent();
    x.line = a[0];
    x.bit = a[1];
    if(x.line < 1024) {
      SimMemory.addUser(x.line, x);
    }
    return x;
  }

  /**
    * Draws this component and any associated parameters at the specified coordinates
    * in the graphics context. The memory line is written in the northwest corner
    * of the node, and the memory bit is written in the northeast.
    *
    * @param g  the graphics context in which to draw.
    * @param x  the x-coordinate of the northwest corner.
    * @param y  the y-coordinate of the northwest corner.
    * @param o  the image observer to be notified of image drawing.
    */
  public void drawComponent(Graphics g, int x, int y, ImageObserver o) {
    if(Master.isSimActive && powered) {
      // Draw a powered component
      g.drawImage(picLit, x, y, o);
      g.setColor(Network.darkGreen);
    } else {
      // Draw a non-powered component
      g.drawImage(pic, x, y, o);
    }

    // Draw the letter or register number of the input line in the upper left corner
    String left;
    if(line == SimMemory.OUTPUT) {
      g.setFont(Network.bigFont);
      g.drawString("O", x+4, y+8);
      g.setFont(Network.myFont);
    } else if(line == SimMemory.FLAG) {
      g.setFont(Network.bigFont);
      g.drawString("F", x+4, y+8);
      g.setFont(Network.myFont);
    } else {
      left = ("   " + line);
      left = left.substring(left.length()-4);
      g.drawString(left, x-2, y+7);
    }

    // Draw the bit of the input line used in the upper right corner
    g.drawString(Integer.toString(bit), x+30, y+7);
    g.setColor(Color.black);
  }
}
