import java.awt.*;
import java.awt.event.*;
import java.lang.String.*;
import javax.swing.*;
import javax.swing.border.*;

/**
  * Dialog box for output coils. This is utilized by OutputComponent as the
  * pop up dialog in response to its <CODE>edit</CODE> method.
  * This modal Dialog allows the user to specify one bit from any output
  * line (output, flag, or any register) to use as an output.
  *
  * @author Mike Cammarano
  * @author Charlie So
  */
public class OutputDialog extends JDialog implements ActionListener
{
  private ButtonGroup rbg;

  private JRadioButton      outputButton;
  private JComboBox         outputCombo;
  private JRadioButton      flagButton;
  private JComboBox         flagCombo;
  private JRadioButton      regButton;
  private JComboBox         regCombo;
  private RegisterListModel regComboModel;
  private JComboBox         bitCombo;

  private int line, bit;

  /**
    * Construct a new instance of the dialog.
    */
  public OutputDialog(Frame o) {
    	super(o, "Outputs", true);

    	rbg = new ButtonGroup();
      ButtonModel bm;
    
    	getContentPane().setLayout(new BoxLayout(getContentPane(), BoxLayout.Y_AXIS));

      JPanel mainPane = new JPanel();
      mainPane.setLayout(new GridLayout(1, 2));
      JPanel left = new JPanel();
      JPanel right = new JPanel();
      JPanel regPanel = new JPanel();
      regPanel.setBorder(BorderFactory.createTitledBorder(
            BorderFactory.createEtchedBorder(), "Register"));

      outputButton = new JRadioButton((new String("Output")));
      bm = outputButton.getModel();
      bm.setActionCommand("O");
      left.add(outputButton);
      rbg.add(outputButton);
      outputCombo = new JComboBox(new NumberListModel("", 32));
      left.add(outputCombo);
      flagButton = new JRadioButton((new String("Flag")));
      bm = flagButton.getModel();
      bm.setActionCommand("F");
      right.add(flagButton);
      rbg.add(flagButton);
      flagCombo = new JComboBox(new NumberListModel("", 32));
      right.add(flagCombo);
      mainPane.add(left);
      mainPane.add(right);
      regButton = new JRadioButton();
      bm = regButton.getModel();
      bm.setActionCommand("R");
      regPanel.add(regButton);
      rbg.add(regButton);
      regCombo = new JComboBox(regComboModel = new RegisterListModel("Reg ", 1024));
      regPanel.add(regCombo);
      bitCombo = new JComboBox(new NumberListModel("bit ", 16));
      regPanel.add(bitCombo);

      getContentPane().add(mainPane);
      getContentPane().add(new JLabel(" ")); // spacing
      getContentPane().add(regPanel);

    //  Build button panel, containing OK and Cancel buttons.		
    JPanel buttons = new JPanel();
    JButton b;
    b = new JButton("OK");
    b.addActionListener(this);
    buttons.add(b);
    b = new JButton("Cancel");
    b.addActionListener(this);
    buttons.add(b);

    getContentPane().add(buttons);
    pack();
    setSize(260,180);
    setResizable(false);
  }

  /**
    * Process events on the OK and Cancel buttons.
    *
    * @see ActionListener
    */
  public void actionPerformed (ActionEvent e) {
    String s = e.getActionCommand();
    if(s.equals("OK")) {
      updateData();
    }
    setVisible(false);
  }

  /**
    * Prior to showing this Dialog, the setState function is called to set
    * all of the radio button and combo box selections to correspond to the
    * current settings of the component being edited.
    *
    * @param newline  initially selected memory line.
    * @param newbit   initially selected bit.
    * @param c        the component this dialog will represent.
    */
  public void setState(int newline, int newbit, CircuitComponent c) {
    regComboModel.setComponent(c);
    line = newline;
    bit = newbit;
    outputCombo.setSelectedIndex(0);
    flagCombo.setSelectedIndex(0);
    regCombo.setSelectedIndex(0);
    bitCombo.setSelectedIndex(0);
    if(line == SimMemory.OUTPUT) {
      rbg.setSelected(outputButton.getModel(), true);
      outputCombo.setSelectedIndex(bit);
    } else if(line == SimMemory.FLAG) {
      rbg.setSelected(flagButton.getModel(), true);
      flagCombo.setSelectedIndex(bit);
    } else {          //register
      rbg.setSelected(regButton.getModel(), true);
      regCombo.setSelectedIndex(line);
      bitCombo.setSelectedIndex(bit);
    }
  }

  /**
    * Returns the memory line selected in this dialog.
    *
    * @return the memory line selected in this dialog.
    */
  public int getLine() {
    return line;
  }

  /**
    * Returns the bit selected in this dialog.
    *
    * @return the bit selected in this dialog.
    */
  public int getBit() {
    return bit;
  }

  /**
    * If the user selects the OK button, this function is called to capture
    * the states of the controls and prepare all the accessor methods to
    * return the updated values. If this function is not called (i.e. if the
    * user chooses Cancel), then the accessor methods will simply return the
    * values that were initially specified by the <CODE>setState</CODE> method.
    */
  public void updateData() {
    String s = rbg.getSelection().getActionCommand();
    if(s.equals("O")) {
      line = SimMemory.OUTPUT;
      bit = outputCombo.getSelectedIndex();
    } else if(s.equals("F")) {
      line = SimMemory.FLAG;
      bit = flagCombo.getSelectedIndex();
    } else {  // register
      line = regCombo.getSelectedIndex();
      bit = bitCombo.getSelectedIndex();
    }
  }
}
