import java.awt.*;
import java.awt.image.*;
import java.lang.String.*;
import java.net.*;
import javax.swing.*;

/**
  * Positive-transition switch implementation of the <CODE>CircuitComponent</CODE>
  * interface.
  *
  * @author Mike Cammarano
  * @author Charlie So
  */
public class PTComponent implements CircuitComponent 
{
  /**
    * The memory line (either a numbered register, the input line, the momentary
    * line, the output line, or the flag line) that toggles this switch.
    */
  private int line;
  /**
    * The individual bit of a memory line that toggles this switch.
    */
  private int bit;
  /**
    * The input to this component during the last call to the <CODE>step</CODE>
    * method.
    */
  private boolean last;
  /**
    * Whether this component should be drawn highlighted on the next repaint.
    */
  private boolean powered;
  /**
    * Image used to draw this component in a network when it is not powered.
    */
  private static Image pic;
  /**
    * Image used to draw this component in a network when it is powered.
    */
  private static Image picLit;
  /**
    * Cursor image used for this component during drag and drop.
    */
  private static Image cursorImage;
  /**
    * Icon used for this component in the component bin.
    */
  private static ImageIcon icon;
  /**
    * ToolTip text used for this component.
    */
  private static String tip = "Pos. Trans. Switch";
  /**
    * The simulations internal name for this component.
    */
  private static String name = "PT";

  static {
      // This static initializer loads all the image files needed
      URL myURL = ClassLoader.getSystemResource("ToolPT.gif");
    	icon = new ImageIcon(myURL);
    	pic = icon.getImage();
      myURL = ClassLoader.getSystemResource("ToolPTOn.gif");
      ImageIcon myIcon = new ImageIcon(myURL);
      picLit = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("CursorPT.gif");
      myIcon = new ImageIcon(myURL);
      cursorImage = myIcon.getImage();
  }

  /**
    * Construct a switch using the default input line and bit.
    */
  public PTComponent() {
    line = SimMemory.INPUT;
    bit = 0;
    reset();
  }

  /**
    * Returns 1, the number of ladder logic rungs occupied by this component.
    *
    * @return 1, the number of ladder logic rungs occupied by this component.
    */
  public int size() {
    return 1;
  }

  /**
    * Performs one evaluation of this component using the specified input
    * logic levels. The switch is closed for one cycle after a positive
    * transition on the input.
    *
    * @param b  array containing input logic levels of the rungs
    *           occupied by this component.
    *
    * @return the output logic levels from this component.
    */
  public boolean[] step(boolean[] b) {
    if(!last && SimMemory.GetBit(line,bit)) {
      last = SimMemory.GetBit(line, bit);
      powered = true;
      return b;
    } else {
      last = SimMemory.GetBit(line, bit);
      boolean[] r = {false};
      powered = false;
      return r;
    }
  }

  /**
    * Displays a modal dialog box to allow the user to edit the parameters
    * of this component. This method will initialize the state of the existing
    * instance of <CODE>NONCDialog</CODE> in <CODE>Master</CODE>, show it, and
    * then update the memory line and bit associated with this switch based on
    * the user's selections in that dialog.
    */
  public void edit() {
    // Pop up the Dialog prompting the user for settings.
    Master.DialogNONC.setState(line, bit);
    Master.DialogNONC.setVisible(true);
    line = Master.DialogNONC.getLine();
    bit = Master.DialogNONC.getBit();
  }

  /**
    * Clears any registers marked as used by this component prior to
    * its deletion.
    */
  public void clear() {}

  /**
    * Resets this component.
    */
  public void reset() {
    last = false;
    powered = false;
  }

  /**
    * Returns the text to be used as a tool tip when this component
    * is placed in the component bin.
    *
    * @return the string to display.
    */
  public String getTip() {
    return tip;
  }

  /**
    * Returns the name of this component to be used internally in saved files
    * and in drag and drop operations.
    *
    * @return the string to use.
    */
  public String getName() {
    return name;
  }

  /**
    * Returns the Icon to be used for this component in the component bin.
    *
    * @return the icon.
    */
  public ImageIcon getToolIcon() {
    return icon;
  }

  /**
    * Returns a one line string representing the state of this component for use in
    * a save file. The string will be of the form:
    * <BR>
    * PT <I>memory-line memory-bit</I>.
    *
    * @return the parameter string.
    */
  public String getSaveData() {
    return name + " " + line + " " + bit;
  }

  /**
    * Returns whether this component must be the rightmost component on any rung.
    * For example, an output coil terminates a rung; a switch does not.
    *
    * @return <CODE>False</CODE> since switches don't terminate rungs.
    */
  public boolean terminatesRung() {
    return false;
  }

  /**
    * Returns the image to use as a mouse cursor for this component during
    * a drag and drop operation.
    *
    * @return the cursor image.
    */
  public Image getImage() {
    return cursorImage;
  }

  /**
    * Returns a new instance of this component created using the default constructor.
    * This method is used by the <CODE>getComponent</CODE> factory method of
    * <CODE>Elements</CODE>.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance() {
    return new PTComponent();
  }

  /**
    * Returns a new instance of this component, using the specified initial parameters.
    * This method is used by the <CODE>getComponent</CODE> factory method of
    * <CODE>Elements</CODE> to instantiate new components during a file open.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance(int[] a) {
    PTComponent x = new PTComponent();
    x.line = a[0];
    x.bit = a[1];
    return x;
  }

  /**
    * Draws this component and any associated parameters at the specified coordinates
    * in the graphics context. The memory line is written in the northwest corner
    * of the node, and the memory bit is written in the northeast.
    *
    * @param g  the graphics context in which to draw.
    * @param x  the x-coordinate of the northwest corner.
    * @param y  the y-coordinate of the northwest corner.
    * @param o  the image observer to be notified of image drawing.
    */
  public void drawComponent(Graphics g, int x, int y, ImageObserver o) {
    if(Master.isSimActive && powered) {
      // Draw a powered component
      g.drawImage(picLit, x, y, o);
      g.setColor(Network.darkGreen);
    } else {
      // Draw a non-powered component
      g.drawImage(pic, x, y, o);
    }
    // Draw the letter or register number of the input line in the upper left corner
    String left;
    if(line == SimMemory.MOMENTARY) {
      g.setFont(Network.bigFont);
      g.drawString("M", x+7, y+8);
      g.setFont(Network.myFont);
    } else if(line == SimMemory.INPUT) {
      g.setFont(Network.bigFont);
      g.drawString("I", x+7, y+8);
      g.setFont(Network.myFont);
    } else if(line == SimMemory.OUTPUT) {
      g.setFont(Network.bigFont);
      g.drawString("O", x+7, y+8);
      g.setFont(Network.myFont);
    } else if(line == SimMemory.FLAG) {
      g.setFont(Network.bigFont);
      g.drawString("F", x+7, y+8);
      g.setFont(Network.myFont);
    } else {
      left = ("   " + line);
      left = left.substring(left.length()-4);
      g.drawString(left, x, y+7);
    }

    // Draw the bit of the input line used in the upper right corner
    g.drawString(Integer.toString(bit), x+27, y+7);
    g.setColor(Color.black);
  }
}
