import java.awt.*;
import java.awt.event.*;
import java.lang.*;
import java.lang.reflect.*;
import java.net.*;
import java.util.*;
import javax.swing.*;

/**
  * Plugin runner performs a continuous looping evaluation of the networks
  * while using a <CODE>SimulatorPlugin</CODE> as a front end.
  *
  * @author Mike Cammarano
  */

public class PluginRunner extends JDialog
                          implements Runnable
{
  /**
    * The plugin managed by this PluginRunner.
    */
  SimulatorPlugin myPlugin;

	/**
	 * Denotes whether the execution is in continuous play mode.
	 */
  boolean playing;
	/**
	 * An execution <Code>Thread</Code> to take care of the continuous
	 * execution as to not interfere with the <Code>ActionListener</Code>
	 */
  Thread runThread;

	/**
	 * Eveluation delay, can be changed to suit the need of the user.
	 */
  int evalDelay = 40;
	/**
	 * This is the minimum delay to make the simulation work well.
	 */
  int minDelay = 8;
	/**
	 * Delay taken before the whole <Code>Simulation</Code> window is redrawn.
	 * If the number is too large, then the updates are done fewer times,
	 * but is the number is too small, it will take up lots of system 
	 * resources and slow down the <Code>Simulation</Code> speed.
	 */
  int redrawDelay = 100;

	/**
	 * Time that the <Code>Simulation</Code> begins. We will be using the 
	 * <Code>getTime()</Code> from <Code>Date</Code> class
	 */
  long startTime;
	/**
	 * Calculate the total time used to evaluate one cycle, that is running
	 * through all the <Code>NetworkFrame</Code> once.
	 */
  int timeEval;
	/**
	 * Number of evaluations performed.
	 */
  int numEval = 0;

	/**
	 * <Code>ListIterator</Code> used to traverse <Code>NetworkList</Code>
	 */
  ListIterator  list;

	/**
	 * Checks the visibilities of <Code>Elements</Code> window and
	 * <Code>Network</Code> list window.
	 */
  boolean stateComp, stateNet;

  /**
    * Constructor. Draws all the needed images as well as place all the
    * buttons.
    *
    * @param   owner  is the frame on which to place the simulation
    */
  public PluginRunner(Frame owner) 
  {
    super(owner, "Simulation");

    pack();
    setResizable(false);
  }

  /**
    * Assign a particular plugin to this runner.
    */
  public void setPlugin(SimulatorPlugin p) {
    getContentPane().removeAll();
    setTitle(p.getTitle());
    myPlugin = p;
    getContentPane().add(p);
    pack();
  }

  /**
    * setVisible has to do a fair bit of work; when the simulation window is shown,
    * we have to hide the component bin, the network manager, and the file menu,
    * and restore them when we're done. Also, various initializations prior to the
    * start of simulation are required here as well.	 
    *
    * @param   b  visible if true, not visible is false.
    */  
  public void setVisible(boolean b)
  {
    if(Master.isSimActive != b) {
      Master.isSimActive = b;
      if(b) {
        stateComp = Master.toolframe.isVisible();
        stateNet = Master.netList.isVisible();
        Master.toolframe.setVisible(false);
        Master.netList.setVisible(false);
        Master.components.setEnabled(false);
        Master.networks.setEnabled(false);
        Master.file.setEnabled(false);
        Master.simulate.setEnabled(false);
        list = Master.net.listIterator(0);
        playing = true;

        SimMemory.clear();
        while(list.hasNext())
          ((NetworkFrame)list.next()).getNetwork().reset();

        runThread = new Thread(this);
        runThread.start();

        super.setVisible(true);
      } else {
        super.setVisible(false);
        playing = false;
        runThread.interrupt();

        SimMemory.clear();
        while(list.hasNext())
          ((NetworkFrame)list.next()).getNetwork().reset();

        Master.components.setEnabled(true);
        Master.networks.setEnabled(true);
        Master.file.setEnabled(true);
        Master.simulate.setEnabled(true);
        Master.toolframe.setVisible(stateComp);
        Master.netList.setVisible(stateNet);
        Master.frame.repaint(redrawDelay);
      }
    }
  }

  /**
    * Evaluation loop.
    */
  public void run() {
    // We shouldn't call the plugin's step method directly within this Thread,
    // since it affects Swing components. So, we will invoke it within the
    // event handler thread when we need it.
    Thread pluginThread = new Thread() {
      public void run() {
        myPlugin.step();
      }
    };

    int delay = evalDelay;
    if(Master.net.getSize()>0)
      delay = (evalDelay/Master.net.getSize())+minDelay;
    numEval = 0;
    startTime = (new Date()).getTime();
    while(playing) {
      try {
        try {
          SwingUtilities.invokeAndWait(pluginThread);
        } catch(InvocationTargetException e) {}
        Master.frame.repaint(redrawDelay);
        do {
            Thread.sleep(delay);
        } while(!stepNetwork());

        // There's some debugging code here to display the rate of evaluation.
        if(Master.verbose) {
          numEval++;
          if(numEval >= 5) {
            timeEval = (int) ((new Date()).getTime() - startTime);
            if(timeEval == 0)
              timeEval = 1;
            System.err.println("Rate: " + (1000*numEval)/timeEval);
            numEval = 0;
            startTime = (new Date()).getTime();
          }
        }
      } catch(InterruptedException e) {}
    }
  }


  /**
    * returns true after all networks are evaluated
    * returns false otherwise
    *
    * @return     true or false depending on the evaluations.
    */
  public boolean stepNetwork() {
    if(Master.net.getSize() > 0) {
      if(!list.hasNext()) {
        list = Master.net.listIterator(0);
      }
      ((NetworkFrame)list.next()).getNetwork().simulate();
      return !list.hasNext();
    }
    return true;
  }

  /**
    * Sets the evaluation delay.
    *
    * @param   nv  sleep time that will be read in from the file
    *              <Code>Master</Code> takes care of this.
    */
  public void setEvalDelay(int nv) {
    evalDelay = nv;
  }


  /**
    * Sets the redraw delay.
    *
    * @param   nv  sleep time for the redrawing the screen.
    */
  public void setRedrawDelay(int nv) {
    redrawDelay = nv;
  }


  /**
   * Sets the minimum delay for evaluation.
   *
   * @param   nv  minimum delay read in from <Code>Master</Code>
   */
  public void setMinDelay(int nv) {
    minDelay = nv;
  }

  /**
    * Returns the current evaluation delay.
    *
    * @return     the evaluation delay for the current <Code>Simuation</Code>
    */
  public int getEvalDelay() {
    return evalDelay;
  }

  /**
    * Returns the current redraw delay.
    *
    * @return     the redraw delay for the current <Code>Simulation</Code>
    */
  public int getRedrawDelay() {
    return redrawDelay;
  }

  /**
    * Returns the current minimum delay.
    *
    * @return     the minimum delay for the current <Code>Simulation</Code>
    */
  public int getMinDelay() {
    return minDelay;
  }
}
