import java.lang.*;
import javax.swing.*;

/**
  * An implementation of <CODE>ListModel</CODE> (for a <CODE>JList</CODE>) that
  * provides a list of the simulators 1024 registers, with an indication whether
  * or not they are already being written to by other circuit components.
  * Used for register combo boxes in timer and counter dialogs.
  *
  * @author Mike Cammarano
  * @author Charlie So
  */
public class RegisterListModel extends AbstractListModel
                             implements ComboBoxModel
{
  /**
    * The prefix shown before all the numbers in this list.
    */
  private String head;
  /**
    * The number of elements in this list.
    */
  private int size;
  /**
    * The selected index in this list (for combo boxes).
    */
  private int selection;
  /**
    * The component whose parameter this list model represents.
    */
  private CircuitComponent myComp;

  /**
    * Creates a new register list model with the specified prefix and range.
    *
    * @param name  the string prefixed to all registers in the list.
    * @param max   the largest register index plus one.
    */
  RegisterListModel(String name, int max) {
     head = name;
     size = max;
   }

  /**
    * Sets the component that this list model is representing.
    * The list model needs to be able to check whether components other than
    * this one have already registered to use particular <CODE>SimMemory</CODE>
    * registers.
    */
  public void setComponent(CircuitComponent c) {
    myComp = c;
  }

  /**
    * Retrieves the string to show for the specified index.
    *
    * @param index  the index in the list.
    *
    * @return       the string to display.
    */
  public Object getElementAt(int index) {
    if(myComp == null)
      return head + index;
    else
      return head + index + (SimMemory.isUsedByOther(index, myComp) ? " U" : "");
  }

  /**
    * Returns the number of registers in the list.
    *
    * @return  the number of registers in the list.
    */
  public int getSize() {
    return size;
  }

  /**
    * Returns the selected string (number appended to the list's prefix).
    *
    * @return  the selected string.
    */
  public Object getSelectedItem() {
    if(myComp == null)
    return head + selection;
    else
      return head + selection + (SimMemory.isUsedByOther(selection, myComp) ? " U" : "");
  }

  /**
    * Sets the selected item.
    *
    * @param item  a string in the form <I>prefix number</I>.
    */
  public void setSelectedItem(Object item) {
    String s = ((String) item).substring(head.length());
    if(s.endsWith(" U"))
    s = s.substring(0,s.length()-2);
    selection = Integer.parseInt(s);
    fireContentsChanged(this, -1, -1);
  }
}
