import java.awt.*;
import java.awt.image.*;
import java.lang.*;		// includes math libraries
import java.lang.String.*;
import java.net.*;
import java.util.*;
import javax.swing.*;

/**
  * Class <CODE>SENSComponent</CODE>.
  *
  * @author Mike Cammarano
  * @author Charlie So
  * @author Robert Marlowe
  *
  * SENS implementation of the <CODE>CircuitComponent</CODE> 
  * interface.
  *
  * Class for 16-bit SENS Function. This is utilized by 
  * <CODE>SENSComponent</CODE>.
  *
  *  The SENS examines and reports the sinse - 1 or 0 - of a specific
  *	bit location in a data matrix.  One bit location is sensed
  *	per scan.
  *
  *	Input - 
  *		SENS has three possible control inputs.  The input to the 
  *		top node initiates the bit sense operation.  An input to
  *		the middle node causes the bit location specified in the 
  *		top node to increment by one on the next scan.  An input
  *		to the bottom node causes the bit loc to be reset to 1.
  *	Output- 
  *		SENS can produce three possible outputs.  The state of the
  * 		output from the top node echoes the state of the top input.
  *		The state of the output from the middle node indicates the
  *		sense of the current bit location.  Power is passed to the
  *		output from the bottom node if an invalid bit location is
  *		entered in the top node.
  *	Top Node Content -
  *		The entry in the top node is the specific bit location that 
  *		is to be sensed in the data matrix.  It may be an integer
  * 		value in the range of 1 ... 1024 (64 registers wide).  
  *	Middle Node Content -
  *		The middle node is the forst word or register in the data 
  *		matrix.
  *	Botton Node Content -
  *		The integer value entered in the bottom node specifies 
  *		a matrix length - i.e., the number of registers or 
  *		16-bit words in the data matrix.  Matrix length 
  *		can be in the range 1 ... 1024.  Ex. A length of 2 
  *		indicates that 32 bits in the matrix will be SENSed.
  *	NOTE:  Values entered into the top and bottom nodes must 
  *		correspond.  Ex. If bit 1024 (16 X 64) is to be SENSed 
  *		then a register-width/matrix-length of 64 must be entered 
  *		into the bottom node.
  *
  */
public class SENSComponent implements CircuitComponent {
  /**
    * The register used to hold this component's top node value.
    */
  private int bitLocLine;
  /**
    * The register used to hold this component's middle node value.
    */
  private int dataMatrixLine;
  /**
    * The register used to hold this component's bottom node value.
    */
  private int lengthLine;
  /**
    * Flag indicating whether component should be highlighted. 
    */
  private boolean powered;
  /**
    * Image used to draw this component in a network when it is not 
    * powered.
    */
  private static Image pic;
  /**
    * Image used to draw this component in a network when it is  
    * powered.
    */
  private static Image picLit;
  /**
    * Cursor image used for this component during drag and drop.
    */
  private static Image cursorImage;
  /**
    * Icon used for this component in the component bin.
    */
  private static ImageIcon icon;
  /**
    * ToolTip text used for this component.
    */
  private static String tip = "SENS";
  /**
    * The simulations internal name for this component.
    */
  private static String name = "SENS";

  static {
      // This static initializer loads all the image files needed
      URL myURL;
      ImageIcon myIcon;
      myURL = ClassLoader.getSystemResource("BigSENS.gif");
    	myIcon = new ImageIcon(myURL);
    	pic = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("BigSENSOn.gif");
        myIcon = new ImageIcon(myURL);
        picLit = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("CursorSENS.gif");
        myIcon = new ImageIcon(myURL);
        cursorImage = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("ToolSENS.gif");
        icon = new ImageIcon(myURL);
  }

  /**
    * Construct a 3-rung Add function using default source, 
    * destination, and iteration values.
    */
  public SENSComponent() {
    bitLocLine     = 0;  // initial registers used
    dataMatrixLine = 1;
    lengthLine     = 2;  // SimMemory auto initializes regs
  }

  /**
    * @return 3, the number of ladder logic rungs occupied by this 
    * component.
    */
  public int size() {
    return 3;
  }

  /**
    * Performs one evaluation of this component using the specified
    * input logic levels. If the 1st rung of the SENS is logic-low,
    * then the SENS resets. Otherwise, the operation commences and 
    * the current state of the first rung is passed to the output.  
    *
    * The top, middle, and bottom rungs	of the input (b[0], b[1], b[2]) 
    * are read and the top, middle, and bottom rungs of the output 
    * (boolean[0], boolean[1], and boolean[2]) is affected by the 
    * operation.
    *
    * @param b - array containing input logic levels of the rungs
    *            occupied by this component.
    *
    * @return the output logic levels from this component.
    */
  public boolean[] step(boolean[] b) {
     powered = b[0];
     boolean tempArr[];
     tempArr = new boolean[] {false, false, false};
     if(b[0]){
	tempArr[0] = true;	// pass input state to output
	int length = SimMemory.GetReg(lengthLine);			// # of words/registers in dataMatrix
	int bitLocation = 0;
	if(!b[2]){
	    bitLocation = SimMemory.GetReg(bitLocLine);
	}
	int wordLoc = bitLocation/16;					// wordLoc == index in matrix
	if(wordLoc < length){				
	    int dataMatrix = SimMemory.GetReg(dataMatrixLine+wordLoc);	// get contents of register w/in matrix
	    int test = dataMatrix >> (bitLocation%16); //shift
	    int state = test & 1;					// mask off bit
	    if(state!=0){						// if bit sense is 1
	    	tempArr[1] = true;					// set output rung 2
	    }
	    if(b[1]){		  					// save incremented bit location
	    	SimMemory.SetReg(bitLocLine, bitLocation+1);
	    }				   	        
	}
	else if(wordLoc >= length){
	    tempArr[2] = true;	// error->  bit loc > matrix length
	}
     }
     return tempArr;		// write outputs	
  }					

  /**
    * Displays a modal dialog box to allow the user to edit the
    *  	parameters of this component, such as which registers are 
    * 	used. This method will initialize the state of the 
    * 	existing instance of <CODE>Node3Dialog</CODE> in 
    * 	<CODE>Master</CODE>, show it, and then update source and 
    *   destination registers to be used in the operation of the 
    *	SENS based on the user's selections from the dialog.
    */
  public void edit() {
    Node3Dialog a = Master.DialogNode3;
    a.setTitle("SENS Register Setup");
    SimMemory.deleteUser(bitLocLine, this);
    a.setState(bitLocLine,dataMatrixLine, lengthLine, this);
    a.setVisible(true);
    bitLocLine 		= a.getValue1Line();
    dataMatrixLine 	= a.getValue2Line();
    lengthLine 		= a.getResultLine();
    SimMemory.addUser(bitLocLine, this);
  }

  /**
    * Clears any registers marked as used by this component prior to
    * its deletion.
    */
  public void clear() {
    SimMemory.deleteUser(bitLocLine, this);
  }

  /**
    * Resets this component, and initializes values
    * in <CODE>SimMemory</CODE>.
    */
  public void reset() {
	// Not needed in this component
  }

  /**
    * Returns the text to be used as a tool tip when this component
    * is placed in the component bin.
    *
    * @return the string to display.
    */
  public String getTip() {
    return tip;
  }

  /**
    * Returns the name of this component to be used internally in 
    * saved files and in drag and drop operations.
    *
    * @return the string to use.
    */
  public String getName() {
    return name;
  }

  /**
    * Returns the Icon to be used for this component in the component 
    * bin.
    *
    * @return the icon.
    */
  public ImageIcon getToolIcon() {
    return icon;
  }

  /**
    * Returns a one line string representing the state of this 
    * component for use in a save file. The string will be of the 
    * form: <BR>
    * SENS 
    * <I>bitLoc-register dataMatrix-register length-register
    * </I>.
    *
    * @return the parameter string.
    */
  public String getSaveData() {
    return name + " " + bitLocLine + " " + dataMatrixLine 
		+ " " + lengthLine;
  }

  /**
    * Returns whether this component must be the rightmost component 
    * on any rung.  For example, an output coil terminates a rung; 
    * a switch does not.
    *
    * @return <CODE>False</CODE> since SENSs don't terminate rungs.
    */
  public boolean terminatesRung() {
    return false;
  }

  /**
    * Returns the image to use as a mouse cursor for this component 
    * during a drag and drop operation.
    *
    * @return the cursor image.
    */
  public Image getImage() {
    return cursorImage;
  }

  /**
    * Returns a new instance of this component created using the 
    * default constructor.
    * This method is used by the <CODE>getComponent</CODE> factory 
    * method of <CODE>Elements</CODE>.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance() {
    SENSComponent x = new SENSComponent();
    SimMemory.addUser(x.bitLocLine, x);
    return x;
  }

  /**
    * Returns a new instance of this component, using the specified 
    * initial parameters.  This method is used by the 
    * <CODE>getComponent</CODE> factory method of <CODE>Elements</CODE> 
    * to instantiate new components during a file open.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance(int[] a) {	
    SENSComponent x = new SENSComponent();
    x.bitLocLine 	= a[0];
    x.dataMatrixLine 	= a[1];
    x.lengthLine    	= a[2];
    SimMemory.addUser(x.bitLocLine, x);
    return x;
  }

  /**
    * Draws this component and any associated parameters at the 
    * specified coordinates in the graphics context. The bitLoc 
    * register is written in the top third of the SENS, the 
    * dataMatrix register is written in the middle third of the 
    * SENS and the length register is written in the bottom third 
    * of the SENS.
    *
    * @param g  the graphics context in which to draw.
    * @param x  the x-coordinate of the northwest corner.
    * @param y  the y-coordinate of the northwest corner.
    * @param o  the image observer to be notified of image drawing.
    */
  public void drawComponent(Graphics g,int x,int y,ImageObserver o){
    if(Master.isSimActive && powered) {
      // Draw a powered component
      g.drawImage(picLit, x, y, o);
      g.setColor(Network.darkGreen);
    } 
    else {
      // Draw a non-powered component
      g.drawImage(pic, x, y, o);
    }

    // Draw Register assignments onto component at rung
    String text;
    
    // Draw the source register in the top third of the component
    text = "     " + Integer.toString(bitLocLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+16);

    // Draw the destination register in the middle third of the comp
    text = "     " + Integer.toString(dataMatrixLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+40);

    // Draw the length register in the bottom third of the component
    text = "     " + Integer.toString(lengthLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+64);

    g.setColor(Color.black);	// in case the color got set to 
  } 				// dark green set back to black
}




/*  To compile and run this Class
	first  - make sure that the class's name is also the file name 
		 -- observe case
	second - at the command prompt in the working directory of the 
		 SDK compile the class.  
			EX. L:\work>N:jdk1.2.1\bin\javac *.java
		 In this case all file *.java will be linked, 
		 therefore take care that they are related or delete 
		 unneeded files.
	third  - run the Class by 
			EX. L:\WORK>N:jdk1.2.1\bin\java figure1511
		 where figure1511 is both the name of the class & file.

    For creating links to, for example, methods ...

	{@link #reset() <CODE>reset</CODE>.}
*/
		