import java.util.*;

/**
  * The memory for the simulator. Basically just a big array of momentaries, inputs,
  * outputs, flags, and registers, with associated accessor methods.
  *
  * @author Mike Cammarano
  * @author Charlie So
  */
public class SimMemory 
{
  /**
    * The stored values for the momentary input lines.
    */
  private static boolean[] momentary;
  /**
    * The stored values for the regular input lines.
    */
  private static boolean[] input;
  /**
    * The stored values for the output lines.
    */
  private static boolean[] output;
  /**
    * The stored values for the flag lines.
    */
  private static boolean[] flag;
  /**
    * The register lines.
    */
  private static int[] register;
  /**
    * A list for every register keeping track of which components output to it.
    */
  private static LinkedList[] regUsers;

  public final static int MOMENTARY = 1024;
  public final static int INPUT = 1025;
  public final static int OUTPUT = 1026;
  public final static int FLAG = 1027;

  /**
    * This class can't be instantiated.
    */
  private SimMemory() {}

  /**
    * Initializes the memory.
    */
  public static void init() {
    	momentary = new boolean[32];
    	input = new boolean[32];
    	output = new boolean[32];
    	flag = new boolean[32];
      register = new int[1024];
      regUsers = new LinkedList[1024];
      for(int i=0; i<1024; i++)
        regUsers[i] = new LinkedList();
  }

  /**
    * Clears all non-register lines. Called prior to beginning a new simulation.
    */
  public static void clear() {
    for(int i=0; i<32; i++) {
      momentary[i] = false;
      input[i] = false;
      output[i] = false;
      flag[i] = false;
    }
  }

  /**
    * Clears all registers in memory.
    */
  public static void clearReg() {
    for(int j=0; j<1024; j++) {
      register[j] = 0;
    }
  }

  /**
    * Retrieves the bit at the specified memory location.
    *
    * @param line the memory line to access.
    * @param bit  the bit to read.
    *
    * @return     the value of the specified bit.
    */
  public static boolean GetBit(int line, int bit) {
    switch(line) {
      case MOMENTARY : return momentary[bit];
      case INPUT     : return input[bit];
      case OUTPUT    : return output[bit];
      case FLAG      : return flag[bit];
      default        : return (((register[line]>>bit)&1)==1);
    }
  }

  /**
    * Sets the value of the bit at the specified memory location.
    *
    * @param line   the memory line to access.
    * @param bit    the bit to write.
    * @param newval the value to assign to the specified bit.
    */
  public static void SetBit(int line, int bit, boolean newval) {
    switch(line) {
      case MOMENTARY : momentary[bit] = newval; break;
      case INPUT     : input[bit] = newval; break;
      case OUTPUT    : output[bit] = newval; break;
      case FLAG      : flag[bit] = newval; break;
      default        : if (newval) register[line] |= (1<<bit);
                       else register[line] &= (65535^(1<<bit)); break;
      }
  }

  /**
    * Retrieves the value of the specified register.
    *
    * @param regAddress  the register to read.
    *
    * @return     the value of the specified register.
    */
  public static int GetReg(int regAddress) {
    return register[regAddress];
  }

  /**
    * Sets the value of the specified register.
    *
    * @param regAddress  the register to write.
    * @param value       the value to assign to the register.
    */
  public static void SetReg(int regAddress, int value) {
    register[regAddress] = (value%65536);
  }

  /**
    * Clears the lists of users for every register.
    */
  public static void clearUsed() {
    for(int j=0; j<1024; j++) {
      if(regUsers[j] == null)
        regUsers[j] = new LinkedList();
      else
        regUsers[j].clear();
    }
  }

  /**
    * Adds the specified component to the list of those writing to a particular register.
    *
    * @param reg   the register being used.
    * @param comp  the component to add.
    */
  public static void addUser(int reg, CircuitComponent comp) {
     if(! regUsers[reg].contains(comp)) {
       regUsers[reg].add(comp);
     }
  }

  /**
    * Removes the specified component from the list of those writing
    * to a particular register.
    *
    * @param reg   the register being used.
    * @param comp  the component to remove.
    */
  public static void deleteUser(int reg, CircuitComponent comp) {
     regUsers[reg].remove(comp);
  }

  /**
    * Returns whether any components other than the one specified are currently
    * listed as users of a particular register.
    *
    * @param reg   the register to check.
    * @param comp  the component to ignore.
    *
    * @return <CODE>True</CODE> if some component other than the one specified
    *         are listed as users of the register.
    */
  public static boolean isUsedByOther(int reg, CircuitComponent comp) {
    if(regUsers[reg].size() == 0)
      return false;
    else if(regUsers[reg].size() == 1)
      return ! regUsers[reg].contains(comp);
    else
      return true;
  }
}
