import java.awt.*;
import java.awt.event.*;
import java.lang.*;
import java.lang.reflect.*;
import java.net.*;
import java.util.*;
import javax.swing.*;

/**
  * Simulation class takes care of the simulation window as well as the 
  * speed of simulation.
  *
  * @author Mike Cammarano
  * @author Charlie So
  */

public class Simulation extends JDialog
                        implements ActionListener, ItemListener, Runnable
{
	/**
	 * Icon used to denote the input switch being open
	 */
  private ImageIcon SwitchOpen;
	/**
	 * Icon used to denote the input switch being closed
	 */
  private ImageIcon SwitchClosed;
	/**
	 * Icon used to denote the momentary switch being open
	 */
  private ImageIcon MomentaryOpen;
	/**
	 * Icon used to denote the momentary switch being closed
	 */
  private ImageIcon MomentaryClosed;
	/**
	 * Icon used to denote the output as off.
	 */
  private ImageIcon OutputOff;
	/**
	 * Icon used to denote the output as on.
	 */
  private ImageIcon OutputOn;

	/**
	 * Icon used for the buttons that control the speed of 
	 * execution.  In this case, a stop.
	 */
  private ImageIcon VCRstop;
	/**
	 * Icon used for the buttons that control the speed of 
	 * execution.  In this case, one <Code>Network</Code> at a time.
	 */
  private ImageIcon VCRframe;
	/**
	 * Icon used for the buttons that control the speed of 
	 * execution.  In this case, one cycle at a time.
	 */
  private ImageIcon VCRslow;
	/**
	 * Icon used for the buttons that control the speed of 
	 * execution.  In this case, a continuous execution.
	 */
  private ImageIcon VCRplay;

	/**
	 * Buttons for the execution speed selection
	 */
  JButton stopButton, frameButton, slowButton, playButton;

	/**
	 * An array of <Code>ButtonModel</Code> for inputs
	 */
  ButtonModel[] inputModel = new ButtonModel[32];
	/**
	 * An array of <Code>ButtonModel</Code> for momentaries
	 */
  ButtonModel[] momentaryModel = new ButtonModel[8];
	/**
	 * An array of <Code>JLabel</Code> for outputs
	 */
  JLabel[]      outputLabel = new JLabel[32];
	/**
	 * An array of <Code>JLabel</Code> for flags
	 */
  JLabel[]      flagLabel = new JLabel[32];
	/**
	 * An array of <Code>JComboBox</Code> for registers.
	 */
  JComboBox[]   regCombo = new JComboBox[8];
	/**
	 * An array of <Code>JLabel</Code> for register values.
	 */
  JLabel[]      regLabel = new JLabel[8];
	/**
	 * A <Code>JLabel</Code> to show which <Code>Network</Code> was the last 
	 * evaluated, and also <Code>JLabel</Code> used to show status of the
	 * execution.
	 */
  JLabel        status = new JLabel("Last evaluated network: 999999");

	/**
	 * Denotes whether the execution is in continuous play mode.
	 */
  boolean playing;
	/**
	 * An execution <Code>Thread</Code> to take care of the continuous
	 * execution as to not interfere with the <Code>ActionListener</Code>
	 */
  Thread runThread;

	/**
	 * Eveluation delay, can be changed to suit the need of the user.
	 */
  int evalDelay = 40;
	/**
	 * This is the minimum delay to make the simulation work well.
	 */
  int minDelay = 8;
	/**
	 * Delay taken before the whole <Code>Simulation</Code> window is redrawn.
	 * If the number is too large, then the updates are done fewer times,
	 * but is the number is too small, it will take up lots of system 
	 * resources and slow down the <Code>Simulation</Code> speed.
	 */
  int redrawDelay = 100;

	/**
	 * Time that the <Code>Simulation</Code> begins. We will be using the 
	 * <Code>getTime()</Code> from <Code>Date</Code> class
	 */
  long startTime;
	/**
	 * Calculate the total time used to evaluate one cycle, that is running
	 * through all the <Code>NetworkFrame</Code> once.
	 */
  int timeEval;
	/**
	 * Number of evaluations performed.
	 */
  int numEval = 0;

	/**
	 * <Code>ListIterator</Code> allows for an easier traverse through the 
	 * <Code>NetworkList</Code>
	 */
  ListIterator  list;

	/**
	 * Checks the visibilities of <Code>Elements</Code> window and
	 * <Code>Network</Code> list window.
	 */
  boolean stateComp, stateNet;


	/**
	 * Constructor. Draws all the needed images as well as place all the
	 * buttons.
	 *
	 * @param   owner  is the frame on which to place the simulation
	 */
  public Simulation(Frame owner) 
  {
    super(owner, "Simulation");
    URL myURL = ClassLoader.getSystemResource("SwitchOpen.gif");
    SwitchOpen = new ImageIcon(myURL);
    myURL = ClassLoader.getSystemResource("SwitchClosed.gif");
    SwitchClosed = new ImageIcon(myURL);
    myURL = ClassLoader.getSystemResource("MomentaryOpen.gif");
    MomentaryOpen = new ImageIcon(myURL);
    myURL = ClassLoader.getSystemResource("MomentaryClosed.gif");
    MomentaryClosed = new ImageIcon(myURL);
    myURL = ClassLoader.getSystemResource("OutputOff.gif");
    OutputOff = new ImageIcon(myURL);
    myURL = ClassLoader.getSystemResource("OutputOn.gif");
    OutputOn = new ImageIcon(myURL);

    myURL = ClassLoader.getSystemResource("VCRstop.gif");
    VCRstop = new ImageIcon(myURL);
    myURL = ClassLoader.getSystemResource("VCRframe.gif");
    VCRframe = new ImageIcon(myURL);
    myURL = ClassLoader.getSystemResource("VCRslow.gif");
    VCRslow = new ImageIcon(myURL);
    myURL = ClassLoader.getSystemResource("VCRplay.gif");
    VCRplay = new ImageIcon(myURL);

    getContentPane().setLayout(new BoxLayout(getContentPane(), BoxLayout.X_AXIS));

    JPanel a = new JPanel();
    a.setLayout(new BoxLayout(a, BoxLayout.Y_AXIS));
    JPanel b = new JPanel();
    b.setLayout(new BoxLayout(b, BoxLayout.Y_AXIS));

      JCheckBox cb;
      JButton tb;
      JButton button;
      Insets nomargin = new Insets(0,0,0,0);

      JPanel input = new JPanel();
      input.setLayout(new GridLayout(4, 8));
      input.setBorder(BorderFactory.createTitledBorder(
            BorderFactory.createEtchedBorder(), "Input"));
      for(int i=0; i<32; i++) {
        cb = new JCheckBox(Integer.toString(i), SwitchOpen);
        cb.setSelectedIcon(SwitchClosed);
        cb.setFocusPainted(false);
        cb.setHorizontalTextPosition(SwingConstants.CENTER);
        cb.setVerticalTextPosition(SwingConstants.TOP);
        cb.setMargin(nomargin);
        inputModel[i]=cb.getModel();
        input.add(cb);
      }

      JPanel momentary = new JPanel();
      momentary.setLayout(new GridLayout(1, 8));
      momentary.setBorder(BorderFactory.createTitledBorder(
            BorderFactory.createEtchedBorder(), "Momentary"));
      for(int i=0; i<8; i++) {
        tb = new JButton(Integer.toString(i), MomentaryOpen);
        tb.setPressedIcon(MomentaryClosed);
        tb.setBorderPainted(false);
        tb.setFocusPainted(false);
        tb.setContentAreaFilled(false);
        tb.setHorizontalTextPosition(SwingConstants.CENTER);
        tb.setVerticalTextPosition(SwingConstants.TOP);
        tb.setMargin(nomargin);
        momentaryModel[i] = new myToggleModel();
        tb.setModel(momentaryModel[i]);
        momentary.add(tb);
      }

      JPanel buttons = new JPanel();
      stopButton = new JButton(VCRstop);
      stopButton.setToolTipText("Pause simulation");
      stopButton.setActionCommand("pause");
      stopButton.addActionListener(this);
      buttons.add(stopButton);
      frameButton = new JButton(VCRframe);
      frameButton.setToolTipText("Step one network");
      frameButton.setActionCommand("stepone");
      frameButton.addActionListener(this);
      buttons.add(frameButton);
      slowButton = new JButton(VCRslow);
      slowButton.setToolTipText("Step all networks");
      slowButton.setActionCommand("stepall");
      slowButton.addActionListener(this);
      buttons.add(slowButton);
      playButton = new JButton(VCRplay);
      playButton.setToolTipText("Continuous evaluation");
      playButton.setActionCommand("play");
      playButton.addActionListener(this);
      buttons.add(playButton);

    a.add(input);
    a.add(momentary);
    a.add(status);
    a.add(buttons);

      JPanel output = new JPanel();
      GridLayout gl = new GridLayout(4, 8);
      gl.setHgap(5);
      output.setLayout(gl);
      output.setBorder(BorderFactory.createTitledBorder(
            BorderFactory.createEtchedBorder(), "Output"));
      for(int i=0; i<32; i++) {
        outputLabel[i] = new JLabel(Integer.toString(i), OutputOff,
                                    SwingConstants.CENTER);
        outputLabel[i].setHorizontalTextPosition(SwingConstants.CENTER);
        outputLabel[i].setVerticalTextPosition(SwingConstants.TOP);
        output.add(outputLabel[i]);
      }

      JPanel flag = new JPanel();
      gl = new GridLayout(4, 8);
      gl.setHgap(5);
      flag.setLayout(gl);
      flag.setBorder(BorderFactory.createTitledBorder(
            BorderFactory.createEtchedBorder(), "Flag"));
      for(int i=0; i<32; i++) {
        flagLabel[i] = new JLabel(Integer.toString(i), OutputOff,
                                  SwingConstants.CENTER);
        flagLabel[i].setHorizontalTextPosition(SwingConstants.CENTER);
        flagLabel[i].setVerticalTextPosition(SwingConstants.TOP);
        flag.add(flagLabel[i]);
      }

    b.add(output);
    b.add(flag);

    JPanel register = new JPanel();
    register.setLayout(new BoxLayout(register, BoxLayout.Y_AXIS));
    register.setBorder(BorderFactory.createTitledBorder(
            BorderFactory.createEtchedBorder(), "Register"));

      JPanel row;
      for(int i=0; i<8; i++) {
        row = new JPanel();
        regCombo[i] = new JComboBox(new NumberListModel("Reg ", 1024));
        regCombo[i].setSelectedIndex(i);
        regLabel[i] = new JLabel("0000 0000 0000 0000  D:00000");
        regCombo[i].addItemListener(this);
        button = new JButton("Edit");
        button.setMargin(nomargin);
        button.setActionCommand("C" + i);
        button.addActionListener(this);
        row.add(regCombo[i]);
        row.add(regLabel[i]);
        row.add(button);
        register.add(row);
      }

    getContentPane().add(a);
    getContentPane().add(b);
    getContentPane().add(register);
    pack();
    setResizable(false);
    }

 /**
   * setVisible has to do a fair bit of work; when the simulation window is shown,
   * we have to hide the component bin, the network manager, and the file menu,
   * and restore them when we're done. Also, various initializations prior to the
   * start of simulation are required here as well.	 
   *
   * @param   b  visible if true, not visible is false.
   */
   
  public void setVisible(boolean b)
  {
    if(Master.isSimActive != b) {
      Master.isSimActive = b;
      if(b) {
        stateComp = Master.toolframe.isVisible();
        stateNet = Master.netList.isVisible();
        Master.toolframe.setVisible(false);
        Master.netList.setVisible(false);
        Master.components.setEnabled(false);
        Master.networks.setEnabled(false);
        Master.file.setEnabled(false);
        Master.simulate.setEnabled(false);
        stopButton.setEnabled(false);
        frameButton.setEnabled(true);
        slowButton.setEnabled(true);
        playButton.setEnabled(true);
        list = Master.net.listIterator(0);
        playing = false;
        status.setText("Simulation ready.");

        SimMemory.clear();
        while(list.hasNext())
          ((NetworkFrame)list.next()).getNetwork().reset();

        for(int i=0; i<32; i++)
          inputModel[i].setSelected(SimMemory.GetBit(SimMemory.INPUT, i));
        updateDisplay();
        super.setVisible(true);
      } else {
        super.setVisible(false);
        playing = false;
        if(runThread != null)
          runThread.interrupt();

        SimMemory.clear();
        while(list.hasNext())
          ((NetworkFrame)list.next()).getNetwork().reset();

        Master.components.setEnabled(true);
        Master.networks.setEnabled(true);
        Master.file.setEnabled(true);
        Master.simulate.setEnabled(true);
        Master.toolframe.setVisible(stateComp);
        Master.netList.setVisible(stateNet);
        Master.frame.repaint(redrawDelay);
      }
    }
  }

	/**
   * After each full evaluation of all the networks, pollInput is called to
   * scan the state of all the input switches and update the SimMemory
   * accordingly.
	 *
	 */
  public void pollInput()
  {
    for(int i=0; i<32; i++) {
      SimMemory.SetBit(SimMemory.INPUT, i, inputModel[i].isSelected());
    }
    for(int i=0; i<8; i++) {
      SimMemory.SetBit(SimMemory.MOMENTARY, i, momentaryModel[i].isPressed());
    }
  }

	/**
   * Handler for the VCR simulation control buttons.
	 *
	 * @param   e  is an <Code>ActionEvent</Code>
	 */
  public void actionPerformed(ActionEvent e)
  {
    String s = e.getActionCommand();
    int index, newval;
    if (s.charAt(0) == 'C') {
      // Change register button.
      int reg = Integer.parseInt(s.substring(1));
      String newString = (String) JOptionPane.showInputDialog(this,
                      "Enter a new (decimal) value for register " + reg,
                      "Change register", JOptionPane.PLAIN_MESSAGE,
                      null, null, Integer.toString(SimMemory.GetReg(reg)));
      if(newString != null) {
        try {
          newval = Integer.parseInt(newString);
          SimMemory.SetReg(reg, newval);
          updateDisplay();
        } catch (NumberFormatException ne) {}
      }
    } else if(s.equals("pause")) {
      playing = false;
      stopButton.setEnabled(false);
      frameButton.setEnabled(true);
      slowButton.setEnabled(true);
      playButton.setEnabled(true);
    } else if (s.equals("stepone")) {
      playing = false;
      stepNetwork();
      updateDisplay();
      Master.frame.repaint(redrawDelay);
    } else if (s.equals("stepall")) {
      playing = false;
      while(!stepNetwork());
      updateDisplay();
      Master.frame.repaint(redrawDelay);
    } else if (s.equals("play")) {
      if(!playing) {
        playing = true;
        stopButton.setEnabled(true);
        frameButton.setEnabled(false);
        slowButton.setEnabled(false);
        playButton.setEnabled(false);
        status.setText("Continuous Evaluation");
        runThread = new Thread(this);
        runThread.start();
      }
    }
  }


	/**
	 * Run implementation of the <Code>Thread</Code> abstraction.
	 *
	 */
  public void run() {
    // We shouldn't call updateDisplay directly within this Thread, since it
    // affects Swing components. So, we will invoke it within the event handler
    // thread when we need it.
    Thread redrawThread = new Thread() {
      public void run() {
        updateDisplay();
      }
    };

    int delay = evalDelay;
    if(Master.net.getSize()>0)
      delay = (evalDelay/Master.net.getSize())+minDelay;
    numEval = 0;
    startTime = (new Date()).getTime();
    while(playing) {
      try {
        do {
          Thread.sleep(delay);
        } while(!stepNetwork());
        try {
          SwingUtilities.invokeAndWait(redrawThread);
        } catch(InvocationTargetException e) {}
        Master.frame.repaint(redrawDelay);

        // There's some debugging code here to display the rate of evaluation.
        if(Master.verbose) {
          numEval++;
          if(numEval >= 5) {
            timeEval = (int) ((new Date()).getTime() - startTime);
            if(timeEval == 0)
              timeEval = 1;
            System.err.println("Rate: " + (1000*numEval)/timeEval);
            numEval = 0;
            startTime = (new Date()).getTime();
          }
        }
      } catch(InterruptedException e) {
      }
    }
  }


	/**
	 * Resets all the registers for the new <Code>Simulation</Code>
	 *
	 */
  public void resetReg() {
    for(int i=0; i<8; i++)
      regCombo[i].setSelectedIndex(i);
  }

  /**
   * Redisplay the registers when a Combobox is updated.
   *
   * @param   e  is a listener
   */
  public void itemStateChanged(ItemEvent e) {
    updateDisplay();
  }


	/**
   * returns true after all networks are evaluated
   * returns false otherwise
	 *
	 * @return     true or false depending on the evaluations.
	 */
  public boolean stepNetwork() {
    int networkNumber;
    if(Master.net.getSize() > 0) {
      if(!list.hasNext()) {
        list = Master.net.listIterator(0);
      }
      networkNumber = list.nextIndex();
      if(networkNumber == 0)
        pollInput();
      ((NetworkFrame)list.next()).getNetwork().simulate();
      if(!playing)
        status.setText("Last evaluated: " + networkNumber);
      return !list.hasNext();
    }
    return true;
  }

	/**
	 * Redraws the display depending on what needs to be redrawn.
	 *
	 */
  public void updateDisplay() {
    for(int i=0; i<32; i++) {
      outputLabel[i].setIcon(SimMemory.GetBit(SimMemory.OUTPUT, i) ? OutputOn : OutputOff);
      flagLabel[i].setIcon(SimMemory.GetBit(SimMemory.FLAG, i) ? OutputOn : OutputOff);
    }
    String regData;
    String decimal;
    int reg;
    for(int i=0; i<8; i++) {
      reg = regCombo[i].getSelectedIndex();
      regData = new String();
      for(int j=0; j<4; j++) {
        for(int k=0; k<4; k++) {
          regData = regData + (SimMemory.GetBit(reg, 15 - (j*4 + k)) ? "1" : "0");
        }
        regData = regData + " ";
      }
      decimal = "00000" + SimMemory.GetReg(reg);
      regLabel[i].setText(regData + " D:" + decimal.substring(decimal.length()-5));
    }
  }

	/**
	 * Sets the evaluation delay.
	 *
	 * @param   nv  sleep time that will be read in from the file
	 *              <Code>Master</Code> takes care of this.
	 */
  public void setEvalDelay(int nv) {
    evalDelay = nv;
  }


	/**
	 * Sets the redraw delay.
	 *
	 * @param   nv  sleep time for the redrawing the screen.
	 */
  public void setRedrawDelay(int nv) {
    redrawDelay = nv;
  }


	/**
	 * Sets the minimum dealy for evaluation.
	 *
	 * @param   nv  minimum delay read in from <Code>Master</Code>
	 */
  public void setMinDelay(int nv) {
    minDelay = nv;
  }


	/**
	 * Returns the current evaluation delay.
	 *
	 * @return     the evaluation delay for the current <Code>Simuation</Code>
	 */
  public int getEvalDelay() {
    return evalDelay;
  }


	/**
	 * Returns the current redraw delay.
	 *
	 * @return     the redraw delay for the current <Code>Simulation</Code>
	 */
  public int getRedrawDelay() {
    return redrawDelay;
  }


	/**
	 * Returns the current minimu delay.
	 *
	 * @return     the minimum delay for the current <Code>Simulation</Code>
	 */
  public int getMinDelay() {
    return minDelay;
  }

  // Model for momentary switches.
  public class myToggleModel extends JToggleButton.ToggleButtonModel
  {
    private boolean lastMesg;

    public myToggleModel() {
      super();
      lastMesg = false;
    }

    public void setArmed(boolean b) {
      super.setArmed(b);
      if(b) {
        if(!lastMesg) {
          fireActionPerformed(new ActionEvent(this, 999, "D" + actionCommand));
          lastMesg = true;
        }
      } else {
        if(lastMesg) {
          fireActionPerformed(new ActionEvent(this, 999, "U" + actionCommand));
          lastMesg = false;
        }
      }
    }
  }
}