import java.awt.*;
import java.awt.image.*;
import java.lang.*;		// includes math libraries
import java.lang.String.*;
import java.net.*;
import java.util.*;
import javax.swing.*;

/**
  * Class <CODE>SubComponent</CODE>.
  *
  * @author Mike Cammarano
  * @author Charlie So
  * @author Robert Marlowe
  *
  * Subtract implementation of the <CODE>CircuitComponent</CODE> 
  * interface.
  *
  * Class for 16-bit Subtract Function. This is utilized by 
  * <CODE>SubComponent</CODE>.
  *
  *  This function allows the user to SUBtract two 16 bit registers 
  *  or constant values.
  *
  *	Input - 
  *		SUB has one control input (to the top node), 
  *		which initiates the operation when it is ON.
  *	Output- 
  *		SUB produces one of three possible outputs.  The
  *		state of the outputs indicates the result of a 
  *		magnitude comparison between value1 and value2.
  *		SUB is often used as a comparator where the state 
  *		of the outputs identifies whether value1 is 
  *		greater than, equal to, or less than value2.
  *	Top and Middle Node Content -
  *		The top and middle nodes contain value1 and 
  *		value2, respectively.  The value in each node may 
  *		be 1 ... 999 -> 16-bit CPU.
  * 	Bottom Node Content -
  *		The register entered in the bottom node stores
  *		the absolute (unsigned) difference between 
  *		value1 and value2.
  *
  *  If the input is ON (= subtract value1 and value2).  The 
  * 	SUB instruction performs an absolute subtraction of
  *	value1 - value2 (top node - middle node) and stores the
  *	difference in a holding register in the bottom node.
  *
  */
public class SubComponent implements CircuitComponent {
  /**
    * The register used to hold this component's top node value.
    */
  private int value1Line;
  /**
    * The register used to hold this component's middle node value.
    */
  private int value2Line;
  /**
    * The register used to hold this component's bottom node value.
    */
  private int diffLine;
  /**
    * Flag indicating whether component should be highlighted. 
    */
  private boolean powered;
  /**
    * Image used to draw this component in a network when it is not 
    * powered.
    */
  private static Image pic;
  /**
    * Image used to draw this component in a network when it is  
    * powered.
    */
  private static Image picLit;
  /**
    * Cursor image used for this component during drag and drop.
    */
  private static Image cursorImage;
  /**
    * Icon used for this component in the component bin.
    */
  private static ImageIcon icon;
  /**
    * ToolTip text used for this component.
    */
  private static String tip = "SUB";
  /**
    * The simulations internal name for this component.
    */
  private static String name = "SUB";

  static {
      // This static initializer loads all the image files needed
      URL myURL;
      ImageIcon myIcon;
      myURL = ClassLoader.getSystemResource("BigSub.gif");
    	myIcon = new ImageIcon(myURL);
    	pic = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("BigSubOn.gif");
        myIcon = new ImageIcon(myURL);
        picLit = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("CursorSub.gif");
        myIcon = new ImageIcon(myURL);
        cursorImage = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("ToolSub.gif");
        icon = new ImageIcon(myURL);
  }

  /**
    * Construct a 3-rung Sub func using default value1, value2, 
    * and difference values.
    */
  public SubComponent() {
    value1Line 	= 0;	// initial registers used
    value2Line 	= 1;
    diffLine    = 2;	// SimMemory automatically initializes regs
  }

  /**
    * @return 3, the number of ladder logic rungs occupied by this 
    * component.
    */
  public int size() {
    return 3;
  }

  /**
    * Performs one evaluation of this component using the specified
    * input logic levels. If the 1st rung of the Sub is logic-low,
    * then the Sub resets. Otherwise, the current input on the
    * first rung is passed to the appropriate output.  
    *
    * Only the top rung	of the input (b[0]) is affected by the 
    * 	operation.  The top, middle, and bottom nodes of the 
    *	output boolean[] is affected. 
    *
    * @param b - array containing input logic levels of the rungs
    *            occupied by this component.  Only b[0] is observed.
    *
    * @return the output logic levels from this component.
    */
  public boolean[] step(boolean[] b) {
    	int tempValue1 = SimMemory.GetReg(value1Line);
    	int tempValue2 = SimMemory.GetReg(value2Line);
	int tempDiff  = 0;
        powered = b[0];
	boolean tempArr[];
	tempArr = new boolean[] {false, false, false};
	if(b[0] && (tempValue1 < 1000 && tempValue1 > 0) 
		&& (tempValue2 < 1000 && tempValue2 > 0)) {

		Math.abs(tempDiff = tempValue1 - tempValue2);
		SimMemory.SetReg(diffLine, tempDiff);//write 3rd Rung
 
		if     (tempValue1 >  tempValue2)tempArr[0] = true;
		else if(tempValue1 == tempValue2)tempArr[1] = true;
		else if(tempValue1 <  tempValue2)tempArr[2] = true;
	}	
	return tempArr;	// return output rung state			 
  }						 

  /**
    * Displays a modal dialog box to allow the user to edit the
    *  	parameters of this component, such as which registers are 
    * 	used. This method will initialize the state of the 
    * 	existing instance of <CODE>Node3Dialog</CODE> in 
    * 	<CODE>Master</CODE>, show it, and then update value1 and 
    *   value2 values to be used in the operation of the SUB 
    *   based on the user's selections from the dialog.
    */
  public void edit() {
    Node3Dialog a = Master.DialogNode3;
    a.setTitle("Subtract Register Setup");
    SimMemory.deleteUser(diffLine, this);
    a.setState(value1Line,value2Line, diffLine, this);
    a.setVisible(true);
    value1Line 	= a.getValue1Line();
    value2Line 	= a.getValue2Line();
    diffLine 	= a.getResultLine();
    SimMemory.addUser(diffLine, this);
  }

  /**
    * Clears any registers marked as used by this component prior to
    * its deletion.
    */
  public void clear() {
    SimMemory.deleteUser(diffLine, this);
  }

  /**
    * Resets this component, and initializes values
    * in <CODE>SimMemory</CODE>.
    */
  public void reset() {
	// Not needed in this component
  }

  /**
    * Returns the text to be used as a tool tip when this component
    * is placed in the component bin.
    *
    * @return the string to display.
    */
  public String getTip() {
    return tip;
  }

  /**
    * Returns the name of this component to be used internally in 
    * saved files and in drag and drop operations.
    *
    * @return the string to use.
    */
  public String getName() {
    return name;
  }

  /**
    * Returns the Icon to be used for this component in the component 
    * bin.
    *
    * @return the icon.
    */
  public ImageIcon getToolIcon() {
    return icon;
  }

  /**
    * Returns a one line string representing the state of this 
    * component for use in a save file. The string will be of the 
    * form: <BR>
    * Sub 
    * <I>value1-register value2-register difference-register</I>.
    *
    * @return the parameter string.
    */
  public String getSaveData() {
    return name + " " + value1Line + " " + value2Line + " " +diffLine;
  }

  /**
    * Returns whether this component must be the rightmost component 
    * on any rung.  For example, an output coil terminates a rung; 
    * a switch does not.
    *
    * @return <CODE>False</CODE> since SUB's don't terminate rungs.
    */
  public boolean terminatesRung() {
    return false;
  }

  /**
    * Returns the image to use as a mouse cursor for this component 
    * during a drag and drop operation.
    *
    * @return the cursor image.
    */
  public Image getImage() {
    return cursorImage;
  }

  /**
    * Returns a new instance of this component created using the 
    * default constructor.
    * This method is used by the <CODE>getComponent</CODE> factory 
    * method of <CODE>Elements</CODE>.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance() {
    SubComponent x = new SubComponent();
    SimMemory.addUser(x.diffLine, x);
    return x;
  }

  /**
    * Returns a new instance of this component, using the specified 
    * initial parameters.  This method is used by the 
    * <CODE>getComponent</CODE> factory method of <CODE>Elements</CODE> 
    * to instantiate new components during a file open.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance(int[] a) {	
    SubComponent x = new SubComponent();
    x.value1Line = a[0];
    x.value2Line = a[1];
    x.diffLine    = a[2];
    SimMemory.addUser(x.diffLine, x);
    return x;
  }

  /**
    * Draws this component and any associated parameters at the 
    * specified coordinates in the graphics context. The value1 value 
    * is written in the top third of the Sub, the value2 value is 
    * written in the middle third of the Sub and the
    * difference value is written in the bottom third of the Sub.
    *
    * @param g  the graphics context in which to draw.
    * @param x  the x-coordinate of the northwest corner.
    * @param y  the y-coordinate of the northwest corner.
    * @param o  the image observer to be notified of image drawing.
    */
  public void drawComponent(Graphics g,int x,int y,ImageObserver o){
    if(Master.isSimActive && powered) {
      // Draw a powered component
      g.drawImage(picLit, x, y, o);
      g.setColor(Network.darkGreen);
    } 
    else {
      // Draw a non-powered component
      g.drawImage(pic, x, y, o);
    }

    // Draw Register assignments onto component at rung
    String text;
    
    // Draw the value1 value in the top third of the component
    text = "     " + Integer.toString(value1Line);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+16);

    // Draw the value2 value in the middle third of the component
    text = "     " + Integer.toString(value2Line);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+40);

    // Draw the diff value in the bottom third of the component
    text = "     " + Integer.toString(diffLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+64);

    g.setColor(Color.black);	// in case the color got set to 
				// dark green set back to black
  }
}




/*  To compile and run this Class
	first  - make sure that the class's name is also the file name 
		 -- observe case
	second - at the command prompt in the working directory of the 
		 SDK compile the class.  
			EX. L:\work>N:jdk1.2.1\bin\javac *.java
		 In this case all file *.java will be linked, 
		 therefore take care that they are related or delete 
		 unneeded files.
	third  - run the Class by 
			EX. L:\WORK>N:jdk1.2.1\bin\java figure1511
		 where figure1511 is both the name of the class & file.

    For creating links to, for example, methods ...

	{@link #reset() <CODE>reset</CODE>.}
*/
		