import java.awt.*;
import java.awt.image.*;
import java.lang.String.*;
import java.net.*;
import java.util.*;
import javax.swing.*;

/**
  * Timer implementation of the <CODE>CircuitComponent</CODE> interface.
  *
  * @author Mike Cammarano
  * @author Charlie So
  */
public class TimerComponent implements CircuitComponent 
{
  /**
    * The register used to hold this component's preset value.
    */
  private int presetLine;
  /**
    * The register used to hold this component's accumulator value.
    */
  private int accLine;
  /**
    * The initial preset value for this component.
    */
  private int presetVal;
  /**
    * The initial accumulator value for this component.
    */
  private int accVal;
  /**
    * The time this timer became active.
    */
  private Date   startTime;
  /**
    * The time elapsed for this timer.
    */
  private long timeElapsed;
  /**
    * The number of milliseconds that must elapse between timer increments.
    */
  private int    millisPerTick;
  /**
    * Image used to draw this component in a network when it is not powered.
    */
  private static Image pic;
  /**
    * Image used to draw this component in a network when it is powered.
    */
  private static Image picLit;
  /**
    * Cursor image used for this component during drag and drop.
    */
  private static Image cursorImage;
  /**
    * Icon used for this component in the component bin.
    */
  private static ImageIcon icon;
  /**
    * ToolTip text used for this component.
    */
  private static String tip = "Timer";
  /**
    * The simulations internal name for this component.
    */
  private static String name = "Timer";

  static {
      // This static initializer loads all the image files needed
      URL myURL = ClassLoader.getSystemResource("BigTimer.gif");
    	ImageIcon myIcon = new ImageIcon(myURL);
    	pic = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("BigTimerOn.gif");
      myIcon = new ImageIcon(myURL);
      picLit = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("CursorTimer.gif");
      myIcon = new ImageIcon(myURL);
      cursorImage = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("ToolTimer.gif");
      icon = new ImageIcon(myURL);
  }

  /**
    * Construct a 1-second timer using default preset and accumulator values.
    */
  public TimerComponent() {
    presetLine = 0;
    accLine = 1;
    presetVal = 0;
    accVal = 0;
    millisPerTick = 1000;
  }

  /**
    * Returns 2, the number of ladder logic rungs occupied by this component.
    *
    * @return 2, the number of ladder logic rungs occupied by this component.
    */
  public int size() {
    return 2;
  }

  /**
    * Performs one evaluation of this component using the specified input
    * logic levels. If the 2nd rung of the counter is logic-low, then the
    * timer resets. Otherwise, the current input on the first rung is
    * compared to the last input to detect a positive transition. If the
    * timer sees a positive transition, it begins counting upwards until the
    * accumulator value reaches the preset value.
    *
    * @param b  array containing input logic levels of the rungs
    *           occupied by this component.
    *
    * @return the output logic levels from this component.
    */
  public boolean[] step(boolean[] b) {
    int tempAcc = SimMemory.GetReg(accLine);
    int tempPre = SimMemory.GetReg(presetLine);
    if(b[1]) {
      if(Master.simWindow.playing) {
        if (startTime != null) {
          Date newTime = new Date();
          timeElapsed += newTime.getTime() - startTime.getTime();
          startTime = newTime;
        } else if(b[0] || timeElapsed > 0) {
          startTime = new Date();
        }
      } else {
        startTime = null;
        timeElapsed += 100;
      }
      tempAcc = (int) (timeElapsed/millisPerTick);
      if(tempAcc > tempPre)
        tempAcc = tempPre;
    } else {
      startTime = null;
      timeElapsed = 0;
      tempAcc = 0;
    }

    SimMemory.SetReg(accLine, tempAcc);

    boolean tempArr[];
    tempArr = new boolean[2];

    tempArr[0] = (tempAcc == tempPre);
    tempArr[1] = (tempAcc <  tempPre);

    return tempArr;
  }

  /**
    * Displays a modal dialog box to allow the user to edit the parameters
    * of this component, such as which registers are used. This method will
    * initialize the state of the existing instance of <CODE>TimerDialog</CODE>
    * in <CODE>Master</CODE>, show it, and then update the accumulator, preset,
    * and rate (1 sec or .1 sec) of this timer based on the user's selections
    * in that dialog.
    */
  public void edit() {
    SimMemory.deleteUser(presetLine, this);
    SimMemory.deleteUser(accLine, this);
    Master.DialogTimer.setState(presetLine, accLine,
                                presetVal, accVal, millisPerTick, this);
    Master.DialogTimer.setVisible(true);
    accLine = Master.DialogTimer.getAccLine();
    presetLine = Master.DialogTimer.getPresetLine();
    presetVal = Master.DialogTimer.getPresetVal();
    accVal = Master.DialogTimer.getAccVal();
    millisPerTick = Master.DialogTimer.getMillisPerTick();
    SimMemory.addUser(presetLine, this);
    SimMemory.addUser(accLine, this);
  }

  /**
    * Clears any registers marked as used by this component prior to
    * its deletion.
    */
  public void clear() {
    SimMemory.deleteUser(presetLine, this);
    SimMemory.deleteUser(accLine, this);
  }

  /**
    * Resets this component, and initializes the preset and accumulator values
    * in <CODE>SimMemory</CODE>.
    */
  public void reset() {
    SimMemory.SetReg(presetLine, presetVal);
    SimMemory.SetReg(accLine, accVal);
    startTime = null;
    timeElapsed = accVal*millisPerTick;
  }

  /**
    * Returns the text to be used as a tool tip when this component
    * is placed in the component bin.
    *
    * @return the string to display.
    */
  public String getTip() {
    return tip;
  }

  /**
    * Returns the name of this component to be used internally in saved files
    * and in drag and drop operations.
    *
    * @return the string to use.
    */
  public String getName() {
    return name;
  }

  /**
    * Returns the Icon to be used for this component in the component bin.
    *
    * @return the icon.
    */
  public ImageIcon getToolIcon() {
    return icon;
  }

  /**
    * Returns a one line string representing the state of this component for use in
    * a save file. The string will be of the form:
    * <BR>
    * Timer <I>preset-register accumulator-register preset-value accumulator-value
               milliseconds-per-tick</I>.
    *
    * @return the parameter string.
    */
  public String getSaveData() {
    return name + " " + presetLine + " " + accLine + " "
                + presetVal + " " + accVal + " " + millisPerTick;
  }

  /**
    * Returns whether this component must be the rightmost component on any rung.
    * For example, an output coil terminates a rung; a switch does not.
    *
    * @return <CODE>False</CODE> since timers don't terminate rungs.
    */
  public boolean terminatesRung() {
    return false;
  }

  /**
    * Returns the image to use as a mouse cursor for this component during
    * a drag and drop operation.
    *
    * @return the cursor image.
    */
  public Image getImage() {
    return cursorImage;
  }

  /**
    * Returns a new instance of this component created using the default constructor.
    * This method is used by the <CODE>getComponent</CODE> factory method of
    * <CODE>Elements</CODE>.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance() {
    TimerComponent x = new TimerComponent();
    SimMemory.addUser(x.presetLine, x);
    SimMemory.addUser(x.accLine, x);
    return x;
  }

  /**
    * Returns a new instance of this component, using the specified initial parameters.
    * This method is used by the <CODE>getComponent</CODE> factory method of
    * <CODE>Elements</CODE> to instantiate new components during a file open.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance(int[] a) {
    TimerComponent x = new TimerComponent();
    x.presetLine = a[0];
    x.accLine = a[1];
    x.presetVal = a[2];
    x.accVal = a[3];
    x.millisPerTick = a[4];
    SimMemory.addUser(x.presetLine, x);
    SimMemory.addUser(x.accLine, x);
    return x;
  }

  /**
    * Draws this component and any associated parameters at the specified coordinates
    * in the graphics context. The preset value is written in the top half of the
    * timer, and the initial accumulator value is written in the bottom half.
    *
    * @param g  the graphics context in which to draw.
    * @param x  the x-coordinate of the northwest corner.
    * @param y  the y-coordinate of the northwest corner.
    * @param o  the image observer to be notified of image drawing.
    */
  public void drawComponent(Graphics g, int x, int y, ImageObserver o) {
    if(Master.isSimActive && startTime != null) {
      // Draw a powered component
      g.drawImage(picLit, x, y, o);
      g.setColor(Network.darkGreen);
    } else {
      // Draw a non-powered component
      g.drawImage(pic, x, y, o);
    }

    String text = "";

    // Draw the timer delay
    if(millisPerTick == 1000)
      text = "1.0";
    else if(millisPerTick == 100)
      text = "0.1";
    g.drawString(text, x+26, y+27);

    // Draw the preset value in the top half of the component
    text = "     " + Integer.toString(presetVal);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+16);

    // Draw the accumulator value in the bottom half of the component
    text = "     " + Integer.toString(accVal);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+40);
    g.setColor(Color.black);
  }
}
