/*  Class TimerDialog.
    Dialog box for Timers. This is utilized by TimerComponent

    This modal Dialog allows the user to choose between 1sec and .1sec
    timers and to specify two registers to use as preset and accumulator,
    along with their initial values.

    Created by: Mike Cammarano and Charlie So.
*/

import java.awt.*;
import java.awt.event.*;
import java.lang.String.*;
import javax.swing.*;
import javax.swing.border.*;

/**
  * Dialog box for Timers. This is utilized by TimerComponent as the
  * pop up dialog in response to its <CODE>edit</CODE> method. This modal Dialog
  * allows the user to choose between 1 second and .1 second timers and
  * to specify two registers to use as preset and accumulator, along
  * with their initial values.
  *
  * @author Mike Cammarano
  * @author Charlie So
  */
public class TimerDialog extends JDialog implements ActionListener
{
  private JComboBox         presetCombo;
  private RegisterListModel presetComboModel;
  private JTextField        presetText;
  private JButton           presetSet;
  private JComboBox         accCombo;
  private RegisterListModel accComboModel;
  private JTextField        accText;
  private JButton           accSet;
  private JRadioButton      rbA;
  private JRadioButton      rbB;

  private ButtonGroup rbg = new ButtonGroup();

  private int millisPerTick;

  private int presetLine, accLine, presetVal, accVal;

  /**
    * Construct a new instance of the dialog.
    */
  public TimerDialog(Frame owner) {
      super(owner, "Timer", true);
 
    	getContentPane().setLayout(new BoxLayout(getContentPane(), BoxLayout.Y_AXIS));

      JPanel radioPane = new JPanel();
      JPanel mainPane = new JPanel();

      rbA = new JRadioButton("1 sec");
      ButtonModel bm = rbA.getModel();
      bm.setActionCommand("1000");
      rbg.add(rbA);
      radioPane.add(rbA);

      rbB = new JRadioButton(".1 sec");
      bm = rbB.getModel();
      bm.setActionCommand("100");
      rbg.add(rbB);
      radioPane.add(rbB);

      mainPane.setLayout(new GridLayout(1, 2));
      JPanel left = new JPanel();
      left.setLayout(new GridLayout(2,1));
      left.setBorder(BorderFactory.createTitledBorder(
            BorderFactory.createEtchedBorder(), "Preset"));
      presetCombo = new JComboBox(presetComboModel = new RegisterListModel("Reg ", 1024));
      left.add(presetCombo);
      presetText = new JTextField(8);
      left.add(presetText);

      JPanel right = new JPanel();
      right.setLayout(new GridLayout(2,1));
      right.setBorder(BorderFactory.createTitledBorder(
            BorderFactory.createEtchedBorder(), "Accumulator"));
      accCombo = new JComboBox(accComboModel = new RegisterListModel("Reg ", 1024));
      right.add(accCombo);
      accText = new JTextField(8);
      right.add(accText);

      mainPane.add(left);
      mainPane.add(right);

      getContentPane().add(radioPane);
      getContentPane().add(mainPane);

    //  Build button panel, containing OK and Cancel buttons.				
    JPanel buttons = new JPanel();
    JButton b;
    b = new JButton("OK");
    b.addActionListener(this);
    buttons.add(b);
    b = new JButton("Cancel");
    b.addActionListener(this);
    buttons.add(b);

    getContentPane().add(buttons);
    pack();
    setSize(220, 180);
    setResizable(false);
  }

  /**
    * Process events on the OK and Cancel buttons.
    *
    * @see ActionListener
    */
  public void actionPerformed (ActionEvent e) 
  	{
        String s = e.getActionCommand();
        if(s.equals("OK")) {
          updateData();
          setVisible(false);
        } else if(s.equals("Cancel")) {
          setVisible(false);
        }
      }

  /**
    * Prior to showing this Dialog, the setState function is called to set
    * all of the radio button and combo box selections to correspond to the
    * current settings of the component being edited.
    *
    * @param newpresetLine  initially selected preset register.
    * @param newaccLine     initially selected accumulator register.
    * @param newpresetVal   initial value for preset.
    * @param newaccVal      initial value for accumulator.
    * @param millis         initial number of milliseconds per clock tick (100 or 1000).
    * @param c              the component this dialog will represent.
    */
  public void setState(int newpresetLine, int newaccLine,
                       int newpresetVal, int newaccVal, int millis, CircuitComponent c) {
    presetComboModel.setComponent(c);
    accComboModel.setComponent(c);
    presetLine = newpresetLine;
    accLine = newaccLine;
    presetVal = newpresetVal;
    accVal = newaccVal;
    millisPerTick = millis;
    rbg.setSelected(rbA.getModel(), true);
    if(millis == 100)
      rbg.setSelected(rbB.getModel(), true);
    presetCombo.setSelectedIndex(presetLine);
    presetText.setText(Integer.toString(newpresetVal));
    SimMemory.SetReg(presetLine, presetVal);
    accCombo.setSelectedIndex(accLine);
    accText.setText(Integer.toString(newaccVal));
    SimMemory.SetReg(accLine, accVal);
  }

  /**
    * Returns the preset register selected in this dialog.
    *
    * @return the preset register selected in this dialog.
    */
  public int getPresetLine() {
    return presetLine;
  }

  /**
    * Returns the accumulator register selected in this dialog.
    *
    * @return the accumulator register selected in this dialog.
    */
  public int getAccLine() {
    return accLine;
  }

  /**
    * Returns the preset value entered in this dialog.
    *
    * @return the preset value entered in this dialog.
    */
  public int getPresetVal() {
    return presetVal;
  }

  /**
    * Returns the accumulator value entered in this dialog.
    *
    * @return the accumulator value entered in this dialog.
    */
  public int getAccVal() {
    return accVal;
  }

  /**
    * Returns the number of milliseconds per clock tick (100 or 1000) entered
    * in this dialog.
    *
    * @return the number of milliseconds per clock tick.
    */
  public int getMillisPerTick() {
    return millisPerTick;
  }

  /**
    * If the user selects the OK button, this function is called to capture
    * the states of the controls and prepare all the accessor methods to
    * return the updated values. If this function is not called (i.e. if the
    * user chooses Cancel), then the accessor methods will simply return the
    * values that were initially specified by the <CODE>setState</CODE> method.
    */
  public void updateData() {
      String s = rbg.getSelection().getActionCommand();
      millisPerTick = Integer.parseInt(s);
      presetLine = presetCombo.getSelectedIndex();
      accLine = accCombo.getSelectedIndex();
      presetVal = Integer.parseInt(presetText.getText());
      accVal = Integer.parseInt(accText.getText());
      SimMemory.SetReg(presetLine, presetVal);
      SimMemory.SetReg(accLine, accVal);
  }
}
