import java.awt.*;
import java.awt.image.*;
import java.lang.*;		// includes math libraries
import java.lang.String.*;
import java.net.*;
import java.util.*;
import javax.swing.*;

/**
  * Class <CODE>TtoRComponent</CODE>.
  *
  * @author Robert Marlowe
  * @author Mike Cammarano
  * @author Charlie So
  *
  * TtoR implementation of the <CODE>CircuitComponent</CODE> 
  * interface.
  *
  * Class for 16-bit Table to Register Function. This is utilized by 
  * <CODE>TtoRComponent</CODE>.
  *
  *  The T->R instruction copies the bit pattern of a register or 16 
  *	contiguous discretes in a table to a specific holding register.
  *	It can accomodate the transfer of one register per scan.  It has
  * 	three control inputs and produces two possible outputs.
  *
  *	Inputs - 
  *		T->R has three control inputs.  The input to the top 
  *		node initiates the DX move operation.  
  *
  *		When the input to the middle node goes ON, the current 
  *		value stored in the pointer register is frozen while the 
  *		DX operation continues.  This causes the same table data 
  *		to be written to the destination register on each scan.
  *		
  *		When the input to the bottom node goes ON, the value in 
  *		the pointer is reset to zero.  This causes the next DX 
  *		move operation to copy the first destination register in 
  *		the table.
  *	Outputs- 
  *		T->R can produce two possible outputs, from the top and 
  *		middle nodes.  The state of the output from the top node
  *		echoes the state of the top input.  The output from the 
  *		middle node goes ON when the value in the pointer 
  *		register equals the specified table length.  At	this 
  *		point, the instruction cannot increment any further.
  *	Top Node Content -
  *		The top node references the first register of discrete 
  *		reference in the source table.  A register or string of
  *		contiguous discretes from this table will be copied in
  *		a scan in a table-to-register operation. 
  *	Middle Node Content -
  *		The register entered in the middle node is the pointer 
  *		to the destination where the source data will be 
  *		copied.  The destination register is the next contiguous
  *		register after the pointer - i.e., if the middle node
  *		displays a pointer of 40100, then the destination 
  *		register for the T->R copy is 40101.
  *
  *		The value stored in the pointer register indicates which
  *		register in the source table will be copied to the 
  *		destination register in the current scan.  A value of 
  *		zero in the pointer indicates that the bit pattern in 
  *		the first register of the source table will be copied to
  *		the destination; a value of 1 in the pointer register 
  *		indicates that the bit pattern in the second register of 
  *		the source table will be copied to the destination
  *		register; etc.
  *	Botton Node Content -
  *		The integer value entered in the bottom node specifies 
  *		the length of the source table - i.e., the number of 
  *		registers that may be copied.  The table length value 
  *		can range from 1 ... 1024.
  *
  */
public class TtoRComponent implements CircuitComponent {
  /**
    * The register used to hold this component's top node value.
    */
  private int sourceLine;
  /**
    * The register used to hold this component's middle node value.
    */
  private int pointerLine;
  /**
    * The register used to hold this component's bottom node value.
    */
  private int lengthLine;
  /**
    * Flag indicating whether component should be highlighted. 
    */
  private boolean powered;
  /**
    * Image used to draw this component in a network when it is not 
    * powered.
    */
  private static Image pic;
  /**
    * Image used to draw this component in a network when it is  
    * powered.
    */
  private static Image picLit;
  /**
    * Cursor image used for this component during drag and drop.
    */
  private static Image cursorImage;
  /**
    * Icon used for this component in the component bin.
    */
  private static ImageIcon icon;
  /**
    * ToolTip text used for this component.
    */
  private static String tip = "TtoR";
  /**
    * The simulations internal name for this component.
    */
  private static String name = "TtoR";

  static {
      // This static initializer loads all the image files needed
      URL myURL;
      ImageIcon myIcon;
      myURL = ClassLoader.getSystemResource("BigTtoR.gif");
    	myIcon = new ImageIcon(myURL);
    	pic = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("BigTtoROn.gif");
        myIcon = new ImageIcon(myURL);
        picLit = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("CursorTtoR.gif");
        myIcon = new ImageIcon(myURL);
        cursorImage = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("ToolTtoR.gif");
        icon = new ImageIcon(myURL);
  }

  /**
    * Construct a 3-rung TtoR function using default source, 
    * pointer, and length values.
    */
  public TtoRComponent() {
    sourceLine  = 0;  // initial registers used
    pointerLine = 1;
    lengthLine  = 2;  // SimMemory auto initializes regs
  }

  /**
    * @return 3, the number of ladder logic rungs occupied by this 
    * component.
    */
  public int size() {
    return 3;
  }

  /**
    * Performs one evaluation of this component using the specified
    * input logic levels. If the 1st rung of the TtoR is logic-low,
    * then the TtoR resets. Otherwise, the operation commences and 
    * the current state of the first rung is passed to the output.  
    *
    * The top, middle, and bottom rungs of the input (b[0],b[1],b[2]) 
    * are read and the top and middle rungs of the output 
    * (boolean[0], boolean[1]) is affected by the operation.
    *
    * @param b - array containing input logic levels of the rungs
    *            occupied by this component. 
    *
    * @return the output logic levels from this component.
    */
  public boolean[] step(boolean[] b) {
     powered = b[0];
     boolean tempArr[];
     tempArr = new boolean[] {false, false, false};
     if(b[2]){				  // if bottom rung is ON
        SimMemory.SetReg(pointerLine, 0); // reset the pointer value
     }					  // to zero
     int cell 	= SimMemory.GetReg(pointerLine);//cell-> source table
     int length = SimMemory.GetReg(lengthLine); //table length
     if(b[0]){			// if top rung is ON
	tempArr[0] = true;	// pass input state to out top rung
	if(cell == length){	// if pointer value == table length
	   tempArr[1] = true;	// set 2nd output rung true
	}
	if(cell <= (length-1)) {
	   // move source into specific cell in table
	   int data = SimMemory.GetReg(sourceLine+cell);
	   SimMemory.SetReg(pointerLine+1, data);
     	   if(!b[1]){	// increment pointer value
		SimMemory.SetReg(pointerLine, cell+1);
     	   }
	}					     
     }
     return tempArr;			
  }					

  /**
    * Displays a modal dialog box to allow the user to edit the
    *  	parameters of this component, such as which registers are 
    * 	used. This method will initialize the state of the 
    * 	existing instance of <CODE>Node3Dialog</CODE> in 
    * 	<CODE>Master</CODE>, show it, and then update source and 
    *   destination registers to be used in the operation of the 
    *	AND based on the user's selections from the dialog.
    */
  public void edit() {
    Node3Dialog a = Master.DialogNode3;
    a.setTitle("TtoR Register Setup");
    for(int i=pointerLine; i < (SimMemory.GetReg(lengthLine)); i++){
    	SimMemory.deleteUser(i, this);
    }
    a.setState(sourceLine,pointerLine, lengthLine, this);
    a.setVisible(true);
    sourceLine 		= a.getValue1Line();
    pointerLine 	= a.getValue2Line();
    lengthLine 		= a.getResultLine();
    for(int i=pointerLine; i < (SimMemory.GetReg(lengthLine)); i++){
    	SimMemory.addUser(i, this);
    }
  }

  /**
    * Clears any registers marked as used by this component prior to
    * its deletion.
    */
  public void clear() {
    for(int i=pointerLine; i < (SimMemory.GetReg(lengthLine)); i++){
    	SimMemory.deleteUser(i, this);
    }
  }

  /**
    * Resets this component, and initializes values
    * in <CODE>SimMemory</CODE>.
    */
  public void reset() {
	// Not needed in this component
  }

  /**
    * Returns the text to be used as a tool tip when this component
    * is placed in the component bin.
    *
    * @return the string to display.
    */
  public String getTip() {
    return tip;
  }

  /**
    * Returns the name of this component to be used internally in 
    * saved files and in drag and drop operations.
    *
    * @return the string to use.
    */
  public String getName() {
    return name;
  }

  /**
    * Returns the Icon to be used for this component in the component 
    * bin.
    *
    * @return the icon.
    */
  public ImageIcon getToolIcon() {
    return icon;
  }

  /**
    * Returns a one line string representing the state of this 
    * component for use in a save file. The string will be of the 
    * form: <BR>
    * TtoR 
    * <I>source-register pointer-register length-register
    * </I>.
    *
    * @return the parameter string.
    */
  public String getSaveData() {
    return name + " " + sourceLine + " " + pointerLine 
		+ " " + lengthLine;
  }

  /**
    * Returns whether this component must be the rightmost component 
    * on any rung.  For example, an output coil terminates a rung; 
    * a switch does not.
    *
    * @return <CODE>False</CODE> since TtoRs don't terminate rungs.
    */
  public boolean terminatesRung() {
    return false;
  }

  /**
    * Returns the image to use as a mouse cursor for this component 
    * during a drag and drop operation.
    *
    * @return the cursor image.
    */
  public Image getImage() {
    return cursorImage;
  }

  /**
    * Returns a new instance of this component created using the 
    * default constructor.
    * This method is used by the <CODE>getComponent</CODE> factory 
    * method of <CODE>Elements</CODE>.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance() {
    TtoRComponent x = new TtoRComponent();
    for(int i=x.pointerLine; i < (SimMemory.GetReg(x.lengthLine)); i++){
    	SimMemory.addUser(i, x);
    }
    return x;
  }

  /**
    * Returns a new instance of this component, using the specified 
    * initial parameters.  This method is used by the 
    * <CODE>getComponent</CODE> factory method of <CODE>Elements</CODE> 
    * to instantiate new components during a file open.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance(int[] a) {	
    TtoRComponent x = new TtoRComponent();
    x.sourceLine 	= a[0];
    x.pointerLine 	= a[1];
    x.lengthLine    	= a[2];
    for(int i=x.pointerLine; i < (SimMemory.GetReg(x.lengthLine)); i++){
    	SimMemory.addUser(i, x);
    }
    return x;
  }

  /**
    * Draws this component and any associated parameters at the 
    * specified coordinates in the graphics context. The source 
    * register is written in the top third of the TtoR, the 
    * pointer register is written in the middle third of the 
    * TtoR and the length register is written in the bottom third 
    * of the TtoR.
    *
    * @param g  the graphics context in which to draw.
    * @param x  the x-coordinate of the northwest corner.
    * @param y  the y-coordinate of the northwest corner.
    * @param o  the image observer to be notified of image drawing.
    */
  public void drawComponent(Graphics g,int x,int y,ImageObserver o){
    if(Master.isSimActive && powered) {
      // Draw a powered component
      g.drawImage(picLit, x, y, o);
      g.setColor(Network.darkGreen);
    } 
    else {
      // Draw a non-powered component
      g.drawImage(pic, x, y, o);
    }

    // Draw Register assignments onto component at rung
    String text;
    
    // Draw the source register in the top third of the component
    text = "     " + Integer.toString(sourceLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+16);

    // Draw the destination register in the middle third of the comp
    text = "     " + Integer.toString(pointerLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+40);

    // Draw the length register in the bottom third of the component
    text = "     " + Integer.toString(lengthLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+64);

    g.setColor(Color.black);	// in case the color got set to 
  } 				// dark green set back to black
}




/*  To compile and run this Class
	first  - make sure that the class's name is also the file name 
		 -- observe case
	second - at the command prompt in the working directory of the 
		 SDK compile the class.  
			EX. L:\work>N:jdk1.2.1\bin\javac *.java
		 In this case all file *.java will be linked, 
		 therefore take care that they are related or delete 
		 unneeded files.
	third  - run the Class by 
			EX. L:\WORK>N:jdk1.2.1\bin\java figure1511
		 where figure1511 is both the name of the class & file.

    For creating links to, for example, methods ...

	{@link #reset() <CODE>reset</CODE>.}
*/
		