import java.awt.*;
import java.awt.image.*;
import java.lang.*;		// includes math libraries
import java.lang.String.*;
import java.net.*;
import java.util.*;
import javax.swing.*;

/**
  * Class <CODE>XORComponent</CODE>.
  *
  * @author Mike Cammarano
  * @author Charlie So
  * @author Robert Marlowe
  *
  * XOR implementation of the <CODE>CircuitComponent</CODE> 
  * interface.
  *
  * Class for 16-bit XOR Function. This is utilized by 
  * <CODE>XORComponent</CODE>.
  *
  *  The XOR instruction performs a Boolean XOR operation on the bit 
  * 	patterns in the source and destination matrices.  The XORed 
  *	bit pattern is then posted in the destination matrix, 
  *	overwriting its previous contents.
  *
  *	Input - 
  *		XOR has one control input (to the top node), 
  *             which initiates the logical operation.
  *	Output- 
  *		XOR produces one output (from its top node), which
  *		echoes the state of the top input.
  *	Top Node Content -
  *		The entry in the top node is the first reference in 
  *		the source matrix.
  *	Middle Node Content -
  *		The entry in the middle node is the first reference 
  *		in the destination matrix.
  *	Botton Node Content -
  *		The integer intered in the bottom node specifies the 
  *		matrix length - i.e., the number of registers or 
  *		16-bit words in the two matrixes.  The matrix length 
  *		can be in the range 1 ... 1024.  Ex. A length of 2 
  *		indicates that 32 bits in each matrix will be XORed.
  *
  *  If the input is ON (= XOR source and destination).  The XOR  
  *	XORs source (its top node) to destination (its middle node) 
  *	and stores the result in the destination (the middle node)
  *	overwriting the previous value.
  *
  */
public class XORComponent implements CircuitComponent {
  /**
    * The register used to hold this component's top node value.
    */
  private int sourceLine;
  /**
    * The register used to hold this component's middle node value.
    */
  private int destinationLine;
  /**
    * The register used to hold this component's bottom node value.
    */
  private int lengthLine;
  /**
    * Flag indicating whether component should be highlighted. 
    */
  private boolean powered;
  /**
    * Image used to draw this component in a network when it is not 
    * powered.
    */
  private static Image pic;
  /**
    * Image used to draw this component in a network when it is  
    * powered.
    */
  private static Image picLit;
  /**
    * Cursor image used for this component during drag and drop.
    */
  private static Image cursorImage;
  /**
    * Icon used for this component in the component bin.
    */
  private static ImageIcon icon;
  /**
    * ToolTip text used for this component.
    */
  private static String tip = "XOR";
  /**
    * The simulations internal name for this component.
    */
  private static String name = "XOR";

  static {
      // This static initializer loads all the image files needed
      URL myURL;
      ImageIcon myIcon;
      myURL = ClassLoader.getSystemResource("BigXOR.gif");
    	myIcon = new ImageIcon(myURL);
    	pic = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("BigXOROn.gif");
        myIcon = new ImageIcon(myURL);
        picLit = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("CursorXOR.gif");
        myIcon = new ImageIcon(myURL);
        cursorImage = myIcon.getImage();
      myURL = ClassLoader.getSystemResource("ToolXOR.gif");
        icon = new ImageIcon(myURL);
  }

  /**
    * Construct a 3-rung XOR func using default source, destination, 
    * and iteration values.
    */
  public XORComponent() {
    sourceLine 	    = 0;  // initial registers used
    destinationLine = 1;
    lengthLine      = 2;  // SimMemory auto initializes regs
  }

  /**
    * @return 3, the number of ladder logic rungs occupied by this 
    * component.
    */
  public int size() {
    return 3;
  }

  /**
    * Performs one evaluation of this component using the specified
    * input logic levels. If the 1st rung of the XOR is logic-low,
    * then the XOR resets. Otherwise, the current state of the
    * first rung is passed to the output.  
    *
    * Only the top rung	of the input (b[0]) and the top rung of the 
    *	output boolean[0] is affected by the operation.
    *
    * @param b - array containing input logic levels of the rungs
    *            occupied by this component.  Only b[0] is observed.
    *
    * @return the output logic levels from this component.
    */
  public boolean[] step(boolean[] b) {
     int tempLength = SimMemory.GetReg(lengthLine);
     powered = b[0];
     boolean tempArr[];
     tempArr = new boolean[] {false, false, false};
     if(b[0]){
	tempArr[0] = true;
	for(int i=0; i < tempLength; i++) {
	    int tempSource 	= SimMemory.GetReg(sourceLine+i);
	    int tempDestination = SimMemory.GetReg(destinationLine+i);
	    tempDestination     = tempSource ^ tempDestination;
	    SimMemory.SetReg(destinationLine+i, tempDestination);
	}
     }
     return tempArr;			
  }					

  /**
    * Displays a modal dialog box to allow the user to edit the
    *  	parameters of this component, such as which registers are 
    * 	used. This method will initialize the state of the 
    * 	existing instance of <CODE>Node3Dialog</CODE> in 
    * 	<CODE>Master</CODE>, show it, and then update source and 
    *   destination registers to be used in the operation of the 
    *	XOR based on the user's selections from the dialog.
    */
  public void edit() {
    Node3Dialog a = Master.DialogNode3;
    a.setTitle("XOR Register Setup");
    for(int i=destinationLine; i < (SimMemory.GetReg(lengthLine)); i++){
    	SimMemory.deleteUser(i, this);
    }
    a.setState(sourceLine, destinationLine, lengthLine, this);
    a.setVisible(true);
    sourceLine 		= a.getValue1Line();
    destinationLine 	= a.getValue2Line();
    lengthLine 		= a.getResultLine();
    for(int i=destinationLine; i < (SimMemory.GetReg(lengthLine)); i++){
    	SimMemory.addUser(i, this);
    }
  }

  /**
    * Clears any registers marked as used by this component prior to
    * its deletion.
    */
  public void clear() {
    for(int i=destinationLine; i < (SimMemory.GetReg(lengthLine)); i++){
    	SimMemory.deleteUser(i, this);
    }
  }

  /**
    * Resets this component, and initializes values
    * in <CODE>SimMemory</CODE>.
    */
  public void reset() {
	// Not needed in this component
  }

  /**
    * Returns the text to be used as a tool tip when this component
    * is placed in the component bin.
    *
    * @return the string to display.
    */
  public String getTip() {
    return tip;
  }

  /**
    * Returns the name of this component to be used internally in 
    * saved files and in drag and drop operations.
    *
    * @return the string to use.
    */
  public String getName() {
    return name;
  }

  /**
    * Returns the Icon to be used for this component in the component 
    * bin.
    *
    * @return the icon.
    */
  public ImageIcon getToolIcon() {
    return icon;
  }

  /**
    * Returns a one line string representing the state of this 
    * component for use in a save file. The string will be of the 
    * form: <BR>
    * XOR 
    * <I>source-register destination-register length-register
    * </I>.
    *
    * @return the parameter string.
    */
  public String getSaveData() {
    return name + " " + sourceLine + " " + destinationLine 
		+ " " + lengthLine;
  }

  /**
    * Returns whether this component must be the rightmost component 
    * on any rung.  For example, an output coil terminates a rung; 
    * a switch does not.
    *
    * @return <CODE>False</CODE> since XORs don't terminate rungs.
    */
  public boolean terminatesRung() {
    return false;
  }

  /**
    * Returns the image to use as a mouse cursor for this component 
    * during a drag and drop operation.
    *
    * @return the cursor image.
    */
  public Image getImage() {
    return cursorImage;
  }

  /**
    * Returns a new instance of this component created using the 
    * default constructor.
    * This method is used by the <CODE>getComponent</CODE> factory 
    * method of <CODE>Elements</CODE>.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance() {
    XORComponent x = new XORComponent();
    for(int i=x.destinationLine; i < (SimMemory.GetReg(x.lengthLine)); i++){
    	SimMemory.addUser(i, x);
    }
    return x;
  }

  /**
    * Returns a new instance of this component, using the specified 
    * initial parameters.  This method is used by the 
    * <CODE>getComponent</CODE> factory method of 
    * <CODE>Elements</CODE> to instantiate new components during a 
    * file open.
    *
    * @return the new <CODE>CircuitComponent</CODE> instance.
    */
  public CircuitComponent getInstance(int[] a) {	
    XORComponent x = new XORComponent();
    x.sourceLine 	= a[0];
    x.destinationLine 	= a[1];
    x.lengthLine    	= a[2];
    for(int i=x.destinationLine; i < (SimMemory.GetReg(x.lengthLine)); i++){
    	SimMemory.addUser(i, x);
    }
    return x;
  }

  /**
    * Draws this component and any associated parameters at the 
    * specified coordinates in the graphics context. The source 
    * register is written in the top third of the XOR, the 
    * destination register is written in the middle third of the XOR 
    * and the length register is written in the bottom third of the 
    * XOR.
    *
    * @param g  the graphics context in which to draw.
    * @param x  the x-coordinate of the northwest corner.
    * @param y  the y-coordinate of the northwest corner.
    * @param o  the image observer to be notified of image drawing.
    */
  public void drawComponent(Graphics g,int x,int y,ImageObserver o){
    if(Master.isSimActive && powered) {
      // Draw a powered component
      g.drawImage(picLit, x, y, o);
      g.setColor(Network.darkGreen);
    } 
    else {
      // Draw a non-powered component
      g.drawImage(pic, x, y, o);
    }

    // Draw Register assignments onto component at rung
    String text;
    
    // Draw the source register in the top third of the component
    text = "     " + Integer.toString(sourceLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+16);

    // Draw the destination register in the middle third of the comp
    text = "     " + Integer.toString(destinationLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+40);

    // Draw the length register in the bottom third of the component
    text = "     " + Integer.toString(lengthLine);
    text = text.substring(text.length()-5);
    g.drawString(text, x+10, y+64);

    g.setColor(Color.black);	// in case the color got set to 
  } 				// dark green set back to black
}




/*  To compile and run this Class
	first  - make sure that the class's name is also the file name 
		 -- observe case
	second - at the command prompt in the working directory of the 
		 SDK compile the class.  
			EX. L:\work>N:jdk1.2.1\bin\javac *.java
		 In this case all file *.java will be linked, 
		 therefore take care that they are related or delete 
		 unneeded files.
	third  - run the Class by 
			EX. L:\WORK>N:jdk1.2.1\bin\java figure1511
		 where figure1511 is both the name of the class & file.

    For creating links to, for example, methods ...

	{@link #reset() <CODE>reset</CODE>.}
*/
		