//############################################################
// Bbox.h
// Kari Pulli
// 3/17/96
// A 3D bounding box for Pnt3
//############################################################

#ifndef _Bbox_h
#define _Bbox_h

#include "Pnt3.h"

class Bbox {
private:
  Pnt3 lo,hi;
  Pnt3 c;      // center
  int      cdone;  // has the center been calculated

  void calcCenter(void)
    { 
      if (!cdone) { 
	cdone=1;
	c = lo; c += hi; c *= .5;
      }
    }
  
public:
  Bbox(void) { clear(); }
  Bbox(Pnt3 const &_lo, Pnt3 const &_hi)
    : lo(_lo), hi(_hi), cdone(0) {}
  ~Bbox(void) {}

  void clear(void)
    {
      hi.set(-1.e33,-1.e33,-1.e33);
      lo.set( 1.e33, 1.e33, 1.e33);
      cdone = 0;
    }

  void add(double x, double y, double z)
    {
      if (x > hi[0]) hi[0] = x;
      if (x < lo[0]) lo[0] = x;
      if (y > hi[1]) hi[1] = y;
      if (y < lo[1]) lo[1] = y;
      if (z > hi[2]) hi[2] = z;
      if (z < lo[2]) lo[2] = z;
      cdone = 0;
    }
  void add(const double *a)
    { add(a[0], a[1], a[2]); }
  void add(const float *a)
    { add(a[0], a[1], a[2]); }


  const Pnt3 &max(void) const  { return hi; }
  const Pnt3 &min(void) const  { return lo; }
  const Pnt3 &center(void)     { calcCenter(); return c; }
  double maxDim(void)
    {
      double ans = 0.0, tmp;
      for (int i=0; i<3; i++)
	if ((tmp=hi[i]-lo[i]) > ans) ans = tmp;
      return ans;
    }

  double size(void) // assume the box is a cube
    { return hi[0]-lo[0]; }

  int outside(Pnt3 &p, double d = 0.0)
    {
      return (p[0] > hi[0]+d || p[0] < lo[0]-d ||
	      p[1] > hi[1]+d || p[1] < lo[1]-d ||
	      p[2] > hi[2]+d || p[2] < lo[2]-d);
    }

  void makeCube(double scale)
    {
      // around the center, expand all dimensions equal
      // to the largest one, then scale
      calcCenter();
      double   len = maxDim()*.5*scale;
      Pnt3 tmp(len,len,len);
      hi = c+tmp;
      lo = c-tmp;
    }

  Bbox &octant(int i)
    {
      // i in [0,7], e.g. 6 == x hi, y hi, z lo
      calcCenter();
      Bbox ret(*this);
      if (i&4) ret.lo[0] = c[0];
      else     ret.hi[0] = c[0];
      if (i&2) ret.lo[1] = c[1];
      else     ret.hi[1] = c[1];
      if (i&1) ret.lo[2] = c[2];
      else     ret.hi[2] = c[2];
      ret.cdone = 0;
      return ret;
    }

  int inOctant(Pnt3 &p) 
    {
      calcCenter();
      return 4*(p[0]>=c[0]) + 2*(p[1]>=c[1]) + (p[2]>=c[2]);
    }

  Pnt3 corner(int i)
    {
      return Pnt3((i&4) ? hi[0] : lo[0],
		  (i&2) ? hi[1] : lo[1],
		  (i&1) ? hi[2] : lo[2]);
    }
};

#endif
