//############################################################
// DepthMap.h
// Kari Pulli
// 3/5/96
//############################################################

#ifndef _DepthMap_h
#define _DepthMap_h

#include <stdlib.h>
#include "Pnt3.h"
#include "Bbox.h"
#include "vector.h"

class DepthMap {
protected:
  int m_height, m_width, m_size;
  int *index;             // m_size indices to xyz, special if < 0
  vector<Pnt3> xyz;       // the 3D points
  Bbox bb;

  int& at(int u, int v) const   { return index[v*m_width+u]; }
    // at(u,v) gives the index of the 3D point at (u,v)
  int good(int u, int v)        { return (at(u,v) >= 0); }
  int good(int i)               { return (index[i] >= 0); }

  void getGridPoint(int u, int v, Pnt3 &p)
    {
      int ind = at(u,v);
      if      (ind >= 0)        p[2] = xyz[ind][2];
      else                      p[2] =  1.e33;
      p[0] = u; p[1] = v;
    }

  void fillHole(int, int); // called by close()

public:
  enum { GOOD = 1, BAD = -1 };
  enum Format { RAW, UW, RAWRLE };

  DepthMap(int w = 640, int h = 480);
  ~DepthMap();

  int type(int u, int v)
    { int t=at(u,v); return (t>=0 ?GOOD :t); }

  int inside(int u, int v) const
    { return (u>=0 && v>=0 && u<m_width && v<m_height); }

  int    height (void) const   { return m_height; }
  int    width  (void) const   { return m_width; }
  int    size   (void) const   { return xyz.size(); }
  double xDim   (void) const   { return bb.max()[0]-bb.min()[0]; }
  double yDim   (void) const   { return bb.max()[1]-bb.min()[1]; }
  double zDim   (void) const   { return bb.max()[2]-bb.min()[2]; }
  const Pnt3 &center(void)     { return bb.center(); }

  int read(const char *file, Format format = RAWRLE);
    // read file (*.xyz)
  int write(const char *file, Format format = RAWRLE, int step = 1);
    // writes a file, if format == UW, subsample by step

  int points(void) const
    { return xyz.size(); }

  int point(int u, int v, Pnt3 &p) const
    { if (!inside(u,v)) return 0;
      int i = at(u,v);
      if (i<0) return 0;
      p = xyz[abs(i)]; 
      return 1; 
    }

  void setPoint(int u, int v, const Pnt3 &p)
    { 
      if (inside(u,v)) {
        if (!good(u,v)) { at(u,v)=xyz.size(); xyz.push_back(p); }
        else xyz[at(u,v)] = p; 
      }
    }
  void updateBoundingBox();

  void removePoint(int u, int v)
    { if (inside(u,v)) at(u,v) = BAD; }
};

#endif

