//###############################################################
// Pnt3.h
// Kari Pulli
// 11/08/1996
//###############################################################

#ifndef _pnt3_h
#define _pnt3_h

#include<math.h>
#include<assert.h>
#include<iostream.h>

class Pnt3 {
protected:
  float v[3];
public: 
  Pnt3(float a=0.0, float b=0.0, float c=0.0) 
    { v[0] = a, v[1] = b, v[2] = c;}
  Pnt3(float *a)      { v[0] = a[0], v[1] = a[1], v[2] = a[2];}
  Pnt3(double *a)     { v[0] = a[0], v[1] = a[1], v[2] = a[2];}

  void set(float a=0.0, float b=0.0, float c=0.0) 
    { v[0] = a, v[1] = b, v[2] = c;}
  void set(float *a)  { v[0] = a[0], v[1] = a[1], v[2] = a[2];}
  void set(double *a) { v[0] = a[0], v[1] = a[1], v[2] = a[2];}
  
  Pnt3  operator-()            { return Pnt3(-v[0],-v[1],-v[2]); }
  Pnt3& operator+=(const Pnt3 &);
  Pnt3& operator-=(const Pnt3 &);
  Pnt3& operator*=(const float &);
  Pnt3& operator/=(const float &);
  int   operator==(const Pnt3 &);
  int   operator!=(const Pnt3 &);
  operator const float *(void) const   { return v; }
  operator       float *(void)         { return v; }
  operator const char  *(void) const   { return (char *)v; }
  operator       char  *(void)         { return (char *)v; }
  float& operator[](int i)             { return v[i]; }
  const float& operator[](int i) const { return v[i]; }
#ifdef IOSTREAMH
  friend ostream& operator<<(ostream &out, const Pnt3 &a);
  friend istream& operator>>(istream &in, Pnt3 &a);
#endif
  // operators that are not part of class: +,-,*,/

  float         norm(void); 
  float         norm2(void); 
  Pnt3 &        normalize(void);
  friend float  dist(const Pnt3 &, const Pnt3 &);
  friend float  dist2(const Pnt3 &, const Pnt3 &);
  friend float  dist_2d(const Pnt3 &, const Pnt3 &);
  friend float  dist2_2d(const Pnt3 &, const Pnt3 &);

  friend float  dot(const Pnt3 &a, const Pnt3 &b);
  friend Pnt3   cross(const Pnt3 &a, const Pnt3 &b);
  friend Pnt3   cross(const Pnt3 &a, const Pnt3 &b,
		      const Pnt3 &c);
  friend Pnt3   normal(const Pnt3 &, const Pnt3 &, const Pnt3 &);
  friend float  det(const Pnt3 &, const Pnt3 &, const Pnt3 &);
  friend void   line_plane_X(const Pnt3& p, const Pnt3& dir,
			     const Pnt3& t1, const Pnt3& t2,
			     const Pnt3& t3,
			     Pnt3 &x, float &dist);
  friend void   line_plane_X(const Pnt3& p, const Pnt3& dir,
			     const Pnt3& nrm, float d,
			     Pnt3 &x, float &dist);
  friend void   bary(const Pnt3& p,  const Pnt3& t1,
		     const Pnt3& t2, const Pnt3& t3,
		     float &b1, float &b2, float &b3);
  friend void   bary(const Pnt3& p,  const Pnt3& dir,
		     const Pnt3& t1, const Pnt3& t2, 
		     const Pnt3& t3,
		     float &b1, float &b2, float &b3);
  friend int    above_plane(const Pnt3& p, const Pnt3& a, 
			    const Pnt3& b, const Pnt3& c);
  // rigid transformations only (rot + trans)
  Pnt3 &        xform(float m[16]);// OpenGL matrix: p . M = p'
  Pnt3 &        xform(double m[16]);// OpenGL matrix: p . M = p'
  Pnt3 &        xform(float r[3][3], float  t[3]);
  Pnt3 &        xform(float r[3][3], double t[3]);
  Pnt3 &        invxform(float m[16]);// OpenGL matrix: p . M = p'
  Pnt3 &        invxform(double m[16]);// OpenGL matrix: p . M = p'
  Pnt3 &        invxform(float r[3][3], float  t[3]);
  Pnt3 &        invxform(float r[3][3], double t[3]);

  Pnt3 & setXformed(const Pnt3 &p, float r[3][3], float t[3]);
  Pnt3 & setRotated(const Pnt3 &p, float r[3][3]);
};


inline Pnt3& 
Pnt3::operator+=(const Pnt3 &a)
{
  v[0] += a.v[0]; v[1] += a.v[1]; v[2] += a.v[2];
  return *this;
}

inline Pnt3& 
Pnt3::operator-=(const Pnt3 &a)
{
  v[0] -= a.v[0]; v[1] -= a.v[1]; v[2] -= a.v[2];
  return *this;
}

inline Pnt3& 
Pnt3::operator*=(const float &a)
{
  v[0] *= a; v[1] *= a; v[2] *= a;
  return *this;
}

inline Pnt3& 
Pnt3::operator/=(const float &a)
{
  v[0] /= a; v[1] /= a; v[2] /= a;
  return *this;
}

inline int
Pnt3::operator==(const Pnt3 &a)
{
  return (a.v[0]==v[0] && a.v[1]==v[1] && a.v[2]==v[2]);
}

inline int
Pnt3::operator!=(const Pnt3 &a)
{
  return (a.v[0]!=v[0] || a.v[1]!=v[1] || a.v[2]!=v[2]);
}

inline Pnt3&
operator+(const Pnt3 &a, const Pnt3 &b)
{
  Pnt3 tmp = a; 
  return (tmp += b);
}

inline Pnt3&
operator-(const Pnt3 &a, const Pnt3 &b)
{
  Pnt3 tmp = a; 
  return (tmp -= b);
}

inline Pnt3&
operator*(const Pnt3 &a, const float &b)
{
  Pnt3 tmp = a; 
  return (tmp *= b);
}

inline Pnt3&
operator*(const float &a, const Pnt3 &b)
{
  Pnt3 tmp = b; 
  return (tmp *= a);
}

inline Pnt3&
operator/(const Pnt3 &a, const float &b)
{
  Pnt3 tmp = a; 
  return (tmp /= b);
}

#ifdef IOSTREAMH
inline ostream& 
operator<<(ostream &out, const Pnt3 &a)
{ 
  return out << "["<< a.v[0] <<" "<< a.v[1] <<" "<< a.v[2] << "]";
}

inline istream& 
operator>>(istream &in, Pnt3 &a)
{
  return in >> "[" >> a.v[0] >> a.v[1] >> a.v[2] >> "]";
}
#endif

inline float 
Pnt3::norm() 
{ 
  return sqrtf(v[0]*v[0] + v[1]*v[1] + v[2]*v[2]);
}

inline float 
Pnt3::norm2() 
{ 
  return v[0]*v[0] + v[1]*v[1] + v[2]*v[2];
}

inline Pnt3 &
Pnt3::normalize(void)
{
  float a = norm();
  if (a != 0) { v[0] /= a; v[1] /= a; v[2] /= a; }
  return *this;
}

inline float
dist2(const Pnt3 &a, const Pnt3 &b)
{
  float x = a.v[0]-b.v[0];
  float y = a.v[1]-b.v[1];
  float z = a.v[2]-b.v[2];
  return x*x + y*y + z*z;
}

inline float
dist(const Pnt3 &a, const Pnt3 &b)
{
  return sqrtf(dist2(a,b));
}

inline float
dist2_2d(const Pnt3 &a, const Pnt3 &b)
{
  float x = a.v[0]-b.v[0];
  float y = a.v[1]-b.v[1];
  return x*x + y*y;
}

inline float
dist_2d(const Pnt3 &a, const Pnt3 &b)
{
  return sqrtf(dist2_2d(a,b));
}

inline float 
dot(const Pnt3 &a, const Pnt3 &b)
{ 
  return (a.v[0]*b.v[0] + a.v[1]*b.v[1] + a.v[2]*b.v[2]);
}

inline Pnt3 
cross(const Pnt3 &a, const Pnt3 &b)
{ 
  return Pnt3(a.v[1]*b.v[2] - a.v[2]*b.v[1],
	      a.v[2]*b.v[0] - a.v[0]*b.v[2],
	      a.v[0]*b.v[1] - a.v[1]*b.v[0]);
}

// two vectors, a and b, starting from c
inline Pnt3
cross(const Pnt3 &a, const Pnt3 &b, const Pnt3 &c)
{
  float a0 = a.v[0]-c.v[0];
  float a1 = a.v[1]-c.v[1];
  float a2 = a.v[2]-c.v[2];
  float b0 = b.v[0]-c.v[0];
  float b1 = b.v[1]-c.v[1];
  float b2 = b.v[2]-c.v[2];
  return Pnt3(a1*b2 - a2*b1, a2*b0 - a0*b2, a0*b1 - a1*b0);
}

// get a normal from triangle ABC, points given in ccw order
inline Pnt3 
normal(const Pnt3 &a, const Pnt3 &b, const Pnt3 &c)
{ 
//  Pnt3 bc = c-b;
//  Pnt3 ba = a-b;
//  return (cross(bc, ba)).normalize();
  return cross(a,b,c).normalize();
}

// determinant of 3 points
inline float  
det(const Pnt3 &a, const Pnt3 &b, const Pnt3 &c)
{
  return a[0]*(b[1]*c[2]-b[2]*c[1]) 
    -    a[1]*(b[0]*c[2]-b[2]*c[0])
    +    a[2]*(b[0]*c[1]-b[1]*c[0]);
}

// calculate the intersection of a line going through p
// to direction dir with a plane spanned by t1,t2,t3
// (modified from Graphics Gems, p.299)
inline void
line_plane_X(const Pnt3& p, const Pnt3& dir,
	     const Pnt3& t1, const Pnt3& t2, const Pnt3& t3, 
	     Pnt3 &x, float &dist)
{
  // note: normal doesn't need to be unit vector
  Pnt3  nrm = cross(t1,t2,t3);
  float tmp = dot(nrm,dir);
  if (tmp == 0.0) {
    cerr << "Cannot intersect plane with a parallel line" << endl;
    return;
  }
  // d  = -dot(nrm,t1)
  // t  = - (d + dot(p,nrm))/dot(dir,nrm)
  // is = p + dir * t
  x = dir;
  dist = (dot(nrm,t1)-dot(nrm,p))/tmp;
  x *= dist;
  x += p;
  if (dist < 0.0) dist = -dist;
}

inline void   
line_plane_X(const Pnt3& p, const Pnt3& dir,
	    const Pnt3& nrm, float d, Pnt3 &x, float &dist)
{
  float tmp = dot(nrm,dir);
  if (tmp == 0.0) {
    cerr << "Cannot intersect plane with a parallel line" << endl;
    return;
  }
  x = dir;
  dist = -(d+dot(nrm,p))/tmp;
  x *= dist;
  x += p;
  if (dist < 0.0) dist = -dist;
}

// calculate barycentric coordinates of the point p
// on triangle t1 t2 t3
inline void 
bary(const Pnt3& p, 
     const Pnt3& t1, const Pnt3& t2, const Pnt3& t3,
     float &b1, float &b2, float &b3)
{
  float fact = 1.0 / det(t1,t2,t3);
  b1 = det(t2,t3,p) * fact;
  b2 = det(t3,t1,p) * fact;
  b3 = det(t1,t2,p) * fact;
}

// calculate barycentric coordinates for the intersection of
// a line starting from p, going to direction dir, and the plane
// of the triangle t1 t2 t3
inline void 
bary(const Pnt3& p, const Pnt3& dir,
     const Pnt3& t1, const Pnt3& t2, const Pnt3& t3,
     float &b1, float &b2, float &b3)
{
  Pnt3 x; float d;
  line_plane_X(p, dir, t1, t2, t3, x, d);
  float fact = 1.0 / det(t1,t2,t3);
  b1 = det(t2,t3,x) * fact;
  b2 = det(t3,t1,x) * fact;
  b3 = det(t1,t2,x) * fact;
}

// is p above the plane spanned by triangle abc (ccw order)?
inline int    
above_plane(const Pnt3& p, const Pnt3& a, 
	    const Pnt3& b, const Pnt3& c)
{
  Pnt3 nrm = cross(a,b,c);
  return dot(p,nrm) > dot(a,nrm);
}

inline Pnt3 &
Pnt3::xform(float m[16])
{
  float x = m[0]*v[0] + m[4]*v[1] + m[8] *v[2] + m[12];
  float y = m[1]*v[0] + m[5]*v[1] + m[9] *v[2] + m[13];
  v[2]    = m[2]*v[0] + m[6]*v[1] + m[10]*v[2] + m[14];
  v[0] = x; v[1] = y;
  return *this;
}

inline Pnt3 &
Pnt3::xform(double m[16])
{
  float x = m[0]*v[0] + m[4]*v[1] + m[8] *v[2] + m[12];
  float y = m[1]*v[0] + m[5]*v[1] + m[9] *v[2] + m[13];
  v[2]    = m[2]*v[0] + m[6]*v[1] + m[10]*v[2] + m[14];
  v[0] = x; v[1] = y;
  return *this;
}

inline Pnt3 &
Pnt3::xform(float r[3][3], float t[3])
{
  float x = t[0] + r[0][0]*v[0] + r[0][1]*v[1] + r[0][2]*v[2];
  float y = t[1] + r[1][0]*v[0] + r[1][1]*v[1] + r[1][2]*v[2];
  v[2]    = t[2] + r[2][0]*v[0] + r[2][1]*v[1] + r[2][2]*v[2];
  v[0] = x; v[1] = y;
  return *this;
}

inline Pnt3 &
Pnt3::xform(float r[3][3], double t[3])
{
  float x = t[0] + r[0][0]*v[0] + r[0][1]*v[1] + r[0][2]*v[2];
  float y = t[1] + r[1][0]*v[0] + r[1][1]*v[1] + r[1][2]*v[2];
  v[2]    = t[2] + r[2][0]*v[0] + r[2][1]*v[1] + r[2][2]*v[2];
  v[0] = x; v[1] = y;
  return *this;
}

inline Pnt3 &
Pnt3::invxform(float m[16])
{
  float tx = v[0] - m[12];
  float ty = v[1] - m[13];
  float tz = v[2] - m[14];
  v[0] = m[0]*tx + m[1]*ty + m[2]*tz;
  v[1] = m[4]*tx + m[5]*ty + m[6]*tz;
  v[2] = m[8]*tx + m[9]*ty + m[10]*tz;
  return *this;
}

inline Pnt3 &
Pnt3::invxform(double m[16])
{
  double tx = v[0] - m[12];
  double ty = v[1] - m[13];
  double tz = v[2] - m[14];
  v[0] = m[0]*tx + m[1]*ty + m[2]*tz;
  v[1] = m[4]*tx + m[5]*ty + m[6]*tz;
  v[2] = m[8]*tx + m[9]*ty + m[10]*tz;
  return *this;
}

inline Pnt3 &
Pnt3::invxform(float r[3][3], float t[3])
{
  float tx = v[0] - t[0];
  float ty = v[1] - t[1];
  float tz = v[2] - t[2];
  v[0] = r[0][0]*tx + r[1][0]*ty + r[2][0]*tz;
  v[1] = r[0][1]*tx + r[1][1]*ty + r[2][1]*tz;
  v[2] = r[0][2]*tx + r[1][2]*ty + r[2][2]*tz;
  return *this;
}

inline Pnt3 &
Pnt3::invxform(float r[3][3], double t[3])
{
  float tx = v[0] - t[0];
  float ty = v[1] - t[1];
  float tz = v[2] - t[2];
  v[0] = r[0][0]*tx + r[1][0]*ty + r[2][0]*tz;
  v[1] = r[0][1]*tx + r[1][1]*ty + r[2][1]*tz;
  v[2] = r[0][2]*tx + r[1][2]*ty + r[2][2]*tz;
  return *this;
}

inline Pnt3 &
Pnt3::setXformed(const Pnt3 &p, float r[3][3], float t[3])
{
  v[0] = r[0][0]*p[0] + r[0][1]*p[1] + r[0][2]*p[2] + t[0];
  v[1] = r[1][0]*p[0] + r[1][1]*p[1] + r[1][2]*p[2] + t[1];
  v[2] = r[2][0]*p[0] + r[2][1]*p[1] + r[2][2]*p[2] + t[2];
  return *this;
}

inline Pnt3 &
Pnt3::setRotated(const Pnt3 &p, float r[3][3])
{
  v[0] = r[0][0]*p[0] + r[0][1]*p[1] + r[0][2]*p[2];
  v[1] = r[1][0]*p[0] + r[1][1]*p[1] + r[1][2]*p[2];
  v[2] = r[2][0]*p[0] + r[2][1]*p[1] + r[2][2]*p[2];
  return *this;
}

class Vec3 : public Pnt3 {
public:
  Vec3(float a=0.0, float b=0.0, float c=0.0) 
    { v[0] = a, v[1] = b, v[2] = c; normalize(); }
  Vec3(const Pnt3 &p)
    { v[0] = p[0]; v[1] = p[1]; v[2] = p[2]; normalize(); }
  Vec3 &xform(float m[16]); // OpenGL model matrix
  Vec3 &xform(float r[3][3]);
};

inline Vec3 &
Vec3::xform(float m[16])
{
  float x = m[0]*v[0] + m[4]*v[1] + m[8] *v[2];
  float y = m[1]*v[0] + m[5]*v[1] + m[9] *v[2];
  v[2]    = m[2]*v[0] + m[6]*v[1] + m[10]*v[2];
  v[0] = x; v[1] = y;
  return *this;
}

inline Vec3 &
Vec3::xform(float r[3][3])
{
  float x = r[0][0]*v[0] + r[0][1]*v[1] + r[0][2]*v[2];
  float y = r[1][0]*v[0] + r[1][1]*v[1] + r[1][2]*v[2];
  v[2]    = r[2][0]*v[0] + r[2][1]*v[1] + r[2][2]*v[2];
  v[0] = x; v[1] = y;
  return *this;
}

#endif











