//----------------------------------------------------------------
//
//  geometry.cc
//
//----------------------------------------------------------------

#include "geometry.h"
#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <GL/gl.h>
#include <Inventor/SoInput.h>
#include <Inventor/SoDB.h>
#include <Inventor/actions/SoAction.h>
#include <Inventor/actions/SoGLRenderAction.h>
#include <Inventor/elements/SoCacheElement.h>
#include <Inventor/nodes/SoCallback.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoPointSet.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/nodes/SoTextureCoordinate2.h>
#include <Inventor/nodes/SoTextureCoordinateBinding.h>

#include "DepthMap.h"

#include "vector.h"
#include "map.h"

void
Geometry::figureOutGraphType(void)
{
  int n = root->getNumChildren();
  if (n<4) { graphType = GENERAL; return; }
  if (!root->getChild(0)->isOfType
      (SoMaterialBinding::getClassTypeId()))
    { graphType = GENERAL; cout << "aa" << endl; return; }
  if (!root->getChild(1)->isOfType
      (SoMaterial::getClassTypeId()))
    { graphType = GENERAL; cout << "bb" << endl; return; }
  if (!root->getChild(2)->isOfType
      (SoCoordinate3::getClassTypeId()))
    { graphType = GENERAL; cout << "cc" << endl; return; }
  mb    = (SoMaterialBinding *)root->getChild(0);
  mat   = (SoMaterial *)root->getChild(1);
  coord = (SoCoordinate3 *)root->getChild(2);
  if (root->getChild(3)->isOfType(SoPointSet::getClassTypeId())) {
    graphType = POINTS;
    pSet = (SoPointSet *)root->getChild(3);
  } else if (root->getChild(3)->isOfType
	     (SoIndexedFaceSet::getClassTypeId())) {
    graphType = MESH;
    fSet = (SoIndexedFaceSet *)root->getChild(3);
  } else graphType = GENERAL;
}


void
Geometry::initPoints(void)
{
  root->removeAllChildren();
  root->addChild(mb);
  root->addChild(mat);
  root->addChild(coord);
  root->addChild(pSet);
  mb->value = SoMaterialBinding::OVERALL; // default, may change
  mat->diffuseColor.setValue(.8, .8, .8);
  coord->point.deleteValues(0);
  pSet->startIndex = 0;
  pSet->numPoints = SO_POINT_SET_USE_REST_OF_POINTS;
  graphType = POINTS;
}


void
Geometry::initMesh(void)
{
  root->removeAllChildren();
  root->addChild(mb);
  root->addChild(mat);
  root->addChild(coord);
  root->addChild(fSet);
  root->addChild(tSet);
  mb->value = SoMaterialBinding::OVERALL; // default, may change
  mat->diffuseColor.setValue(.8, .8, .8);
  coord->point.deleteValues(0);
  fSet->coordIndex.deleteValues(0);
  tSet->coordIndex.deleteValues(0);
  graphType = MESH;
}


Geometry::Geometry() 
: no_colors(0)
{
  root = new SoSeparator;
  root->ref();

  mb    = new SoMaterialBinding;
  mat   = new SoMaterial;
  coord = new SoCoordinate3;

  pSet = new SoPointSet;
  fSet = new SoIndexedFaceSet;
  tSet = new SoIndexedTriangleStripSet;
  graphType = NOTHING;

  mb->ref();
  mat->ref();
  coord->ref();
  pSet->ref();
  fSet->ref();
  tSet->ref();
}

Geometry::~Geometry()
{
  root->unref();
  mb->unref();
  mat->unref();
  coord->unref();
  pSet->unref();
  fSet->unref();
}

static int dListInd;


static void 
drawLinesGL(void *, SoAction *action)
{
  if (action->isOfType(SoGLRenderAction::getClassTypeId())) {
    glCallList(dListInd);
  }
}


void
Geometry::readEdgeData(ifstream &from)
{
  // do the drawing in GL for speed
  dListInd = glGenLists(1);
  SoCallback *glCB = new SoCallback;
  glCB->setCallback(drawLinesGL);

  glNewList(dListInd, GL_COMPILE);
  // now read the file and put the lines into a display list
  char string[10];
  float v[6];
  int cnt=0;

  Bbox bb;

  glDisable(GL_LIGHTING);
  glBegin(GL_LINES);
  while (1) {
    if (!(from >> string)) break;
    if (string[0] == 'i') {
      from >> v[0] >> v[1] >> v[2] >> v[3] >> v[4] >> v[5];
      bb.add(v); bb.add(&v[3]);
      glColor3f(1., 0., 0.);
      glVertex3fv(v);
      glVertex3fv(&v[3]);
    } else if (string[0] == 'o') {
      from >> v[0] >> v[1] >> v[2] >> v[3] >> v[4] >> v[5];
      bb.add(v); bb.add(&v[3]);
      glColor3f(0., 1., 0.);
      glVertex3fv(v);
      glVertex3fv(&v[3]);
    } else if (string[0] == 't') {
      from >> v[0] >> v[1] >> v[2] >> v[3] >> v[4] >> v[5];
      bb.add(v); bb.add(&v[3]);
      glColor3f(1., 0., 0.);
      glVertex3fv(v);
      glColor3f(0., 1., 0.);
      glVertex3fv(&v[3]);
    } else if (string[0] == 'n') {
      from >> v[0] >> v[1] >> v[2];
      bb.add(v);
      glColor3f(1., 0., 1.);
      glVertex3fv(v);
      glVertex3fv(v);
    }
    if (++cnt % 10000 == 0) cout << cnt << endl;
  }
  glEnd();
  glEnable(GL_LIGHTING);
  glEndList();

  {
    SoSeparator *root2 = new SoSeparator;
    root->addChild(root2);

    // draw in invisible points for bounding box
    SoDrawStyle *_ds = new SoDrawStyle;
    _ds->style = SoDrawStyle::INVISIBLE;
    root2->addChild(_ds);
    int i,j,k,cnt=0;
    SoCoordinate3 *bbox = new SoCoordinate3;
    for (i=0; i<2; i++) 
      for (j=0; j<2; j++) 
	for (k=0; k<2; k++) 
	  bbox->point.set1Value(cnt++, 
				2*(i?bb.max()[0]:bb.min()[0]),
				2*(j?bb.max()[1]:bb.min()[1]),
				2*(k?bb.max()[2]:bb.min()[2]));
    root2->addChild(bbox);
    SoPointSet *pset = new SoPointSet;
    pset->startIndex = 0;
    pset->numPoints = SO_POINT_SET_USE_REST_OF_POINTS;
    root2->addChild(pset);
  }

  root->addChild(glCB);
  cout << "Done" << endl;
}


int
Geometry::readFile(char *fileName)
{
  // check if it's Inventor scenegraph
  int len = strlen(fileName);
  if (strcmp(".iv", &fileName[len-3]) == 0) {
    SoInput in;
    if (!in.openFile(fileName)) {
      cerr << "Cannot open " << fileName << endl;
      return 0;
    }
    SoSeparator *sg = SoDB::readAll(&in);
      if (sg == NULL) {
	cerr << "Problem reading file " << fileName << endl;
	return 0;
      }
    in.closeFile();

    root->removeAllChildren();
    for (int i=0; i<sg->getNumChildren(); i++)
      root->addChild(sg->getChild(i));
    figureOutGraphType();
    return 1;
  }

  // check if it's color image
  if (strcmp(".rgb", &fileName[len-4]) == 0) {
    img.read(fileName, Image::SGI, 640, 480);
    return 1;
  }

  // check if it's depthmap
  if (len > 7 && strcmp(".rlexyz", &fileName[len-7]) == 0) {
    DepthMap dm;
    dm.read(fileName);
    Pnt3 p;
    int cnt = 0;
    
    if (graphType != POINTS) initPoints();

    if (img.width() > 0) {
      if (!no_colors) mb->value = SoMaterialBinding::PER_VERTEX;
      for (int v=0; v<dm.height(); v++)
	for (int u=0; u<dm.width(); u++)
	  if (dm.point(u,v,p)) {
	    mat->diffuseColor.set1Value(cnt, 
					img.getR(u,v)/255.0,
					img.getG(u,v)/255.0,
					img.getB(u,v)/255.0);
	    coord->point.set1Value(cnt++, p);
	  }
    } else {
      for (int v=0; v<dm.height(); v++)
	for (int u=0; u<dm.width(); u++)
	  if (dm.point(u,v,p)) 
	    coord->point.set1Value(cnt++, p);
    }
    
    return 1;
  }  

  // else just open the file
  ifstream from(fileName);
  if (!from) {
    cerr << "Opening file " << fileName << " failed" << endl;
    return 0;
  }
  
  // decide the file contents
  // based on whether we find first a single p or
  // string Vertex in the file...

  int   i, cnt = 0, fi = 0;
  float val[3], currCol[3];
  char  ch, buffer[256];
  int   line = 1;
  map<int, int, less<int> > vertexMap;

  graphType = NOTHING;
  while (from >> ch) {
    switch (ch) {
    case 'p' :
    case 'd' :
      if (graphType != POINTS) initPoints();
      if (ch == 'p') {
	// read a point
	from >> val[0] >> val[1] >> val[2];
	coord->point.set1Value(cnt, val);
	if (mb->value.getValue() == SoMaterialBinding::PER_VERTEX)
	  mat->diffuseColor.set1Value(cnt, currCol);
	cnt++;
      } else if (ch == 'd') {
	// read a (diffuse) color
	from >> currCol[0] >> currCol[1] >> currCol[2];
	if (!no_colors) mb->value = SoMaterialBinding::PER_VERTEX;
      } else {
	cerr << "Can't be here " << __FILE__ << __LINE__ << endl;
      }
      break;
    case 'V' :
      if (graphType != MESH) initMesh();

      from.get(buffer,256,'\n');
      if (from.get(ch) && ch != '\n') {
	cerr << "Input file line " << line << " too long" << endl;
	return 0;
      }
      line++;

      sscanf(buffer, "ertex %d %f %f %f",
	     &i,&val[0],&val[1],&val[2]);

      vertexMap[i] = cnt;

      coord->point.set1Value(cnt, val);
      {
	char *tmp = strchr(buffer, '[');
	if (tmp) {
	  if (no_colors) {
	    double dummy[3];
	    sscanf(tmp, "[%f %f %f]",
		   &dummy[0],&dummy[1],&dummy[2]);
	  } else {
	    mb->value = SoMaterialBinding::PER_VERTEX_INDEXED;
	    sscanf(tmp, "[%f %f %f]",
		   &currCol[0],&currCol[1],&currCol[2]);
	    mat->diffuseColor.set1Value(cnt, currCol);
	  }
	}
      }
      cnt++;
      break;
    case 'F' :
      assert(graphType == MESH);

      from.get(buffer,256,'\n');
      if (from.get(ch) && ch != '\n') {
	cerr << "Input file line " << line << " too long" << endl;
	return 0;
      }
      line++;
      {
	int a,b,c;
	sscanf(buffer, "ace %i %d %d %d", &i,&a,&b,&c);
	a = vertexMap[a];
	b = vertexMap[b];
	c = vertexMap[c];
	fSet->coordIndex.set1Value(fi++, a);
	fSet->coordIndex.set1Value(fi++, b);
	fSet->coordIndex.set1Value(fi++, c);
	fSet->coordIndex.set1Value(fi++, SO_END_FACE_INDEX);
      }
      break;
    case 'i' : // in
    case 'o' : // out
    case 't' : // through
    case 'n' : // no
      // this case is for visualizing the error image consisting
      // of points and their projections to the mesh
      from.putback(ch);
      readEdgeData(from);
      return 1;
    default:
      // skip to the end of line
      do { from.get(ch); } while (ch != '\n' && !from.eof());
      line++;
    }
  }      
  return 1;
}


void
Geometry::writeFile(char *fileName, int UWformat)
{
  if (UWformat) {
    int colors = (mb->value.getValue() != SoMaterialBinding::OVERALL);
    ofstream to(fileName);
    if (graphType == POINTS) {
      for (int i=0; i<coord->point.getNum(); i++) {
	if (colors) {
	  to << "d " << mat->diffuseColor[i][0] << " " 
	    << mat->diffuseColor[i][1] << " " 
	      << mat->diffuseColor[i][2] << endl;
	}
	to << "p " << coord->point[i][0] << " " 
	  << coord->point[i][1] << " " 
	    << coord->point[i][2] << endl;
      }
    } else if (graphType == MESH) {
      for (int i=0; i<coord->point.getNum(); i++) {
	to << "Vertex " << i+1 << " ";
	to << coord->point[i][0] << " " 
	  << coord->point[i][1] << " " 
	    << coord->point[i][2] << " ";
	if (colors) {
	  to << "[";
	  to << mat->diffuseColor[i][0] << " " 
	    << mat->diffuseColor[i][1] << " " 
	      << mat->diffuseColor[i][2] << " ";
	  to << "]";
	}
	to << endl;
      }
      for (i=0; i<fSet->coordIndex.getNum(); i+=4) {
	to << "Face " << i+1 << " ";
	to << fSet->coordIndex[i]+1 << " " 
	  << fSet->coordIndex[i+1]+1 << " " 
	    << fSet->coordIndex[i+2]+1 << endl;
      }
    } else if (graphType == GENERAL) {
      cerr << "Don't know how to convert this to UW format" << endl;
    }
  } else {
    // just dump the Inventor scene graph
    SoWriteAction myAction;
    myAction.getOutput()->openFile(fileName);
    myAction.getOutput()->setBinary(TRUE);
    myAction.apply(root);
    myAction.getOutput()->closeFile();
  }
}
