//############################################################
// Context.C
// Kari Pulli
// 12/08/95
// The graphics context class for a Motif GUI library
//############################################################

#include "Context.h"
#include "PixelMap.h"

const int REQ_HEIGHT = 16;
const int REQ_WIDTH  = 16;

uint Context::tileWidth   = 0;
uint Context::tileHeight  = 0;
uint Context::stipWidth   = 0;
uint Context::stipHeight  = 0;

void Context::classConstruct(void)
{
  // obtain optimum tile/stipple dimensions
  XQueryBestTile    (display, root, REQ_WIDTH, REQ_HEIGHT,
		     &tileWidth, &tileHeight);
  XQueryBestStipple (display, root, REQ_WIDTH, REQ_HEIGHT,
		     &stipWidth, &stipHeight);
}

Context::Context(const Color &fcolor, const Color &bcolor)
: GraphicalEnv(), fore(fcolor), back(bcolor)
{
  static Boolean classInit = FALSE;
  if (!classInit) {
    classConstruct();
    classInit = TRUE;
  }
  
  XGCValues vals;
  vals.foreground = (ulong) fore;
  vals.background = (ulong) back;
  gc              = XCreateGC(display, root, 
			      GCForeground | GCBackground,
			      &vals);
  XFlush(display);
}

Context::~Context(void)
{
  XFreeGC(display, gc);
}

void
Context::setForeground(const Color &color)
{
  XSetForeground(display, gc, (ulong) (fore = color));
}

void
Context::setBackground(const Color &color)
{
  XSetBackground(display, gc, (ulong) (back = color));
}

void
Context::toggleReverse(void)
{
  Color tcol = fore;
  XSetForeground(display, gc, (ulong) (fore = back));
  XSetBackground(display, gc, (ulong) (back = tcol));
}

static const int FunctionLookup[] = {
  GXclear, GXset, GXand, GXnand, GXcopy,
  GXnoop, GXxor, GXor, GXnor, GXinvert
  };

Boolean 
Context::setFunction(const DrawFunction func)
{
  if (func >= DRAW_FUNC_INVERT) return FALSE;
  XSetFunction(display, gc, FunctionLookup[func]);
  return TRUE;
}

static const int LineStyleLookup[] = {
  LineSolid, LineOnOffDash, LineDoubleDash
  };

Boolean
Context::setLineStyle(const LineStyle ls)
{
  if (ls > LINE_STYLE_DOUBLE_DASH) return FALSE;
  XGCValues vals;
  vals.line_style = LineStyleLookup[ls];
  XChangeGC(display, gc, GCLineStyle, &vals);
  return TRUE;
}

const uint Context::LW1_FAST    = 0;
const uint Context::LW1_PRECISE = 1;

void 
Context::setLineWidth(const uint lw)
{
  XGCValues vals;
  vals.line_width = lw;
  XChangeGC(display, gc, GCLineWidth, &vals);
}

Boolean 
Context::setFillStyle(const FillStyle  fs,
		      PixelMap *       pmap,
		      int              x,
		      int              y)
{
  if (fs != FILL_SOLID && pmap == NULL) return FALSE;

  XGCValues vals;
  ulong     mask = GCFillStyle;

  switch (fs) {
  case FILL_SOLID:
    vals.fill_style = FillSolid;
    break;
  case FILL_TILE:
    {
      if (pmap->getWidth()  != tileWidth ||
	  pmap->getHeight() != tileHeight) return FALSE;

      mask |= GCTileStipXOrigin | GCTileStipYOrigin | GCTile;
      vals.fill_style  = FillTiled;
      vals.tile        = (Pixmap) (*pmap);
      vals.ts_x_origin = x;
      vals.ts_y_origin = y;
    } 
    break;
  case FILL_STIPPLE:
  case FILL_STIPPLE_OPAQUE:
    {
      if (pmap->getWidth()  != tileWidth ||
	  pmap->getHeight() != tileHeight) return FALSE;

      mask |= GCTileStipXOrigin | GCTileStipYOrigin | GCStipple;
      vals.fill_style  = (fs == FILL_STIPPLE ?
			  FillStippled :
			  FillOpaqueStippled);			  
      vals.stipple     = (Pixmap) (*pmap);
      vals.ts_x_origin = x;
      vals.ts_y_origin = y;
    } 
    break;
  default:
    return FALSE;
  }
  
  XChangeGC(display, gc, mask, &vals);
  return TRUE;
}
