//############################################################
// CustomDialog.C
// Kari Pulli
// 12/21/95
// A class for a Motif GUI library
//############################################################

extern "C" {
#include <Xm/ScrolledW.h>
}
#include "CustomDialog.h"
#include "CString.h"
#include "Separator.h"
#include "Form.h"

#define ButtonLoop(i) for (i=0; i<NUM_BUTTONS; i++)
#define TIGHTNESS 15

CustomDialog::CustomDialog(const CString          &name,
			   const GraphicalObject  *parent)
: Dialog(XmCreateFormDialog, name, parent)
{
  int i;
  ButtonLoop(i) {
    button[i]   = NULL;
    activate[i] = NULL;
    udata[i]    = NULL;
  }
  actionArea    = NULL;
  separator     = NULL;
  controlArea   = NULL;

  // hidden by default
//  hideButton(APPLY);
//  hideButton(HELP);
}

CustomDialog::~CustomDialog(void)
{
  int i;
  // destroy others but controlArea
  ButtonLoop(i) if (button[i]) delete button[i];
  if (actionArea) delete actionArea;
  if (separator)  delete separator;
}

void 
CustomDialog::customize(GraphicalObject &ctrlArea,
			Boolean          createOK,
			Boolean          createCancel,
			Boolean          createApply,
			Boolean          createHelp)
{
  int i;
  // get rid of old customization
  ButtonLoop(i) if (button[i]) delete button[i];
  if (actionArea) delete actionArea;
  if (separator)  delete separator;

  controlArea = &ctrlArea;
  separator = new Separator("sep", this);

  Boolean cflag[Dialog::UNKNOWN];
  int bCnt = 0;

  if (cflag[Dialog::OK]     = createOK)     bCnt++;
  if (cflag[Dialog::CANCEL] = createCancel) bCnt++;
  if (cflag[Dialog::APPLY]  = createApply)  bCnt++;
  if (cflag[Dialog::HELP]   = createHelp)   bCnt++;

  actionArea = new Form("actionArea", this, TIGHTNESS*bCnt+1);
  bCnt = 0;
  ButtonLoop(i)
    if (cflag[i]) button[i] = createButton(i, bCnt);
  Widget ca = (Widget) *controlArea;

  // special case for composites that allow scrolling
  if (XtClass(XtParent(ca)) == xmScrolledWindowWidgetClass)
    ca = XtParent(ca);

  // attach
  XtVaSetValues(ca,
		XmNtopAttachment,    XmATTACH_FORM,
		XmNleftAttachment,   XmATTACH_FORM,
		XmNrightAttachment,  XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_WIDGET,
		XmNbottomWidget,     (Widget) *separator,
		NULL);
  XtVaSetValues((Widget) *separator,
		XmNleftAttachment,   XmATTACH_FORM,
		XmNrightAttachment,  XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_WIDGET,
		XmNbottomWidget,     (Widget) *actionArea,
		NULL);
  XtVaSetValues((Widget) *actionArea,
		XmNleftAttachment,   XmATTACH_FORM,
		XmNrightAttachment,  XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
  controlArea->manage();
  separator->manage();
  actionArea->manage();
}

TextButton *
CustomDialog::createButton(int  buttonPos,
			   int &buttonCount)
{
  const char *label;

  if      (buttonPos == 0) label = Dialog::OK_LABEL;
  else if (buttonPos == 1) label = Dialog::APPLY_LABEL;
  else if (buttonPos == 2) label = Dialog::CANCEL_LABEL;
  else                     label = Dialog::HELP_LABEL;

  int lPos = TIGHTNESS * buttonCount++ + 1;
  int rPos = lPos + TIGHTNESS - 1;

  TextButton *button = new TextButton(label, actionArea);
 
  actionArea->attachPosition (*button, Form::LEFT, lPos);
  actionArea->attachForm     (*button, Form::TOP);
  actionArea->attachForm     (*button, Form::BOTTOM);
  actionArea->attachPosition (*button, Form::RIGHT, rPos);

  button->registerCallback(GraphicalObject::CB_ACTIVE,
			   customButtonCB,
			   this);
  button->manage();
  return button;
}

void 
CustomDialog::registerCallback(CallbackType    type,
			       CallbackHandler cb,
			       void *          data)
{
  StandardButton b;

  switch(type) {
  case CB_OK:
    b = OK;     break;
  case CB_APPLY:
    b = APPLY;  break;
  case CB_CANCEL:
    b = CANCEL; break;
  case CB_HELP:
    b = HELP;   break;
  default: // if it's not a button callback, pass it on
    GraphicalObject::registerCallback(type, cb, data);
    return;
  }
  activate[b] = cb;
  udata[b]    = data;
}

void 
CustomDialog::unregisterCallback(CallbackType    type)
{
  StandardButton b;

  switch(type) {
  case CB_OK:
    b = OK;     break;
  case CB_APPLY:
    b = APPLY;  break;
  case CB_CANCEL:
    b = CANCEL; break;
  case CB_HELP:
    b = HELP;   break;
  default: // if it's not a button callback, pass it on
    GraphicalObject::unregisterCallback(type);
    return;
  }
  activate[b] = NULL;
  udata[b]    = NULL;
}

static int SimpleReasonLookup[] = 
{ XmCR_OK, XmCR_APPLY, XmCR_CANCEL, XmCR_HELP };

static GraphicalObject::CallbackType CallbackTypeLookup[] = {
  GraphicalObject::CB_OK,
  GraphicalObject::CB_APPLY,
  GraphicalObject::CB_CANCEL,
  GraphicalObject::CB_HELP,
};

void 
customButtonCB(GraphicalObject              &button,
	       GraphicalObject::CallbackType,
	       void *                        udata)
{
  CustomDialog *dialog = (CustomDialog *)udata;
  int i;

  // find custom dialog button
  ButtonLoop(i) if (&button == dialog->button[i]) break;
  
  XmAnyCallbackStruct cbs;
  cbs.reason = SimpleReasonLookup[i];
  
  // explicitly invoke Dialog internal state callback
  stateCB(button, dialog, &cbs);

  if (dialog->activate[i])
    dialog->activate[i](*dialog,
			CallbackTypeLookup[i],
			dialog->udata[i]);
}


