//############################################################
// Dialog.C
// Kari Pulli
// 12/14/95
// A class for a Motif GUI library
//############################################################

#include "Dialog.h"
#include "CString.h"

const char *Dialog::FILTER_LABEL  = "Filter";
const char *Dialog::APPLY_LABEL   = "Apply";
const char *Dialog::CANCEL_LABEL  = "Cancel";
const char *Dialog::HELP_LABEL    = "Help";
const char *Dialog::NO_LABEL      = "No";
const char *Dialog::OK_LABEL      = "OK";
const char *Dialog::YES_LABEL     = "Yes";

Dialog::Dialog(CreateHook             cfn,
	       const CString         &name,
	       const GraphicalObject *parent)
: GraphicalObject()
{
  Widget wparent = parent ? (Widget) *parent : topLevel;

  posCalc    = FROM_PARENT;
  posType    = CENTER;
  dx = dy    = 0;
  dismissed  = cancelled = applied = dismissOnApply = FALSE;
  lastButton = UNKNOWN;

  XmString mtitle = XmStringCreateSimple((char *) name);
  Arg args[4];

  XtSetArg(args[0], XmNautoUnmanage,      False);
  XtSetArg(args[1], XmNdefaultPosition,   False);
  XtSetArg(args[2], XmNdialogTitle,       mtitle);
  XtSetArg(args[3], XmNdefaultButtonType, XmDIALOG_CANCEL_BUTTON);

  rtr.mrep = cfn(wparent, (char *) name, args, 4);

  XmStringFree(mtitle);

  XtAddCallback(XtParent(rtr.mrep),
		XmNpopupCallback,
		positionCB,
		this);
}

static unsigned char ModeLookUp[] = {
  XmDIALOG_FULL_APPLICATION_MODAL,
  XmDIALOG_MODELESS
  };

void
Dialog::pose(PoseType wait)
{
  XtVaSetValues(rtr.mrep,
		XmNdialogStyle, ModeLookUp[wait],
		NULL);
  dismissed = cancelled = applied = FALSE;
  lastButton = UNKNOWN;

  manage();  // put dialog on screen

  if (wait == WAIT) {
    while (!dismissed) GraphicalEnv::processEvent();
    unmanage();
  }
  XmUpdateDisplay(topLevel);
}

void
stateCB (Widget    ,
	 XtPointer clientd,
	 XtPointer calld)
{
  XmAnyCallbackStruct    *cbs = (XmAnyCallbackStruct *) calld;
  Dialog                 *dialog = (Dialog *) clientd;
  Dialog::StandardButton  button;

  switch(cbs->reason) {
  case XmCR_OK:
    button            = Dialog::OK;
    dialog->dismissed = TRUE;
    break;
  case XmCR_APPLY:
    button            = Dialog::APPLY;
    dialog->applied   = TRUE;
    break;
  case XmCR_CANCEL:
    button            = Dialog::CANCEL;
    dialog->dismissed = TRUE;
    dialog->cancelled = TRUE;
    break;
  case XmCR_HELP: 
    button            = Dialog::HELP;
    break;
  default:
    button            = Dialog::UNKNOWN;
    cerr << "Something wrong!" << __FILE__ << " " 
      << __LINE__ << endl;
    break;
  }
  dialog->lastButton = button;
}

void
Dialog::setPosition(PosCalc  calc,
		    PosType  pos,
		    int      x,
		    int      y)
{
  posCalc = calc;
  posType = pos;
  dx      = x;
  dy      = y;
}

void
positionCB(Widget , XtPointer clientd, XtPointer )
{
  Dialog    *dialog = (Dialog *) clientd;
  Dimension  shellw, shellh; // dialog shell dims

  XtVaGetValues(XtParent(dialog->rtr.mrep),
		XmNwidth,  &shellw,
		XmNheight, &shellh,
		NULL);

  Position  parentx, parenty;
  Dimension parentw, parenth;

  if (dialog->posCalc == Dialog::FROM_PARENT) {
    XtVaGetValues(XtParent(XtParent(dialog->rtr.mrep)),
		  XmNwidth,   &parentw,
		  XmNheight,  &parenth,
		  XmNx,       &parentx,
		  XmNy,       &parenty,
		  NULL);

    Widget grandParent =
      XtParent(XtParent(XtParent(dialog->rtr.mrep)));
    // translate grandparent relative coords to screen coords
    if (grandParent != (Widget) 0) {
      XtTranslateCoords(grandParent,
			parentx,  parenty,
			&parentx, &parenty);
    }
  } else {
    parentw = screenWidth();
    parenth = screenHeight();
    parentx = parenty = 0;
  }

  Position newx, newy; // new screen rel. coords for the dlg shell
  
  switch(dialog->posType) {
  case Dialog::EXPLICIT:
    newx = parentx + (Position) dialog->dx;
    newy = parenty + (Position) dialog->dy;
    break;

  case Dialog::TOP_LEFT:
    newx = parentx;
    newy = parenty;
    break;

  case Dialog::TOP_RIGHT:
    newx = parentx + (parentw - shellw);
    break;

  case Dialog::BOTTOM_LEFT:
    newy = parenth + (parenth - shellh);
    break;

  case Dialog::BOTTOM_RIGHT:
    newx = parentx + (parentw - shellw);
    newy = parenty + (parenth - shellh);
    break;

  case Dialog::CENTER:
    newx = parentx + (parentw - shellw) / 2;
    newy = parenty + (parenth - shellh) / 2;
    break;
  }

  XtVaSetValues(XtParent(dialog->rtr.mrep),
		XmNx, newx,
		XmNy, newy,
		NULL);

  // this should be called only the first time
  XtRemoveCallback(XtParent(dialog->rtr.mrep),
		   XmNpopupCallback,
		   positionCB,
		   dialog);
}

void
Dialog::setSensitivity(StandardButton button,
		       Boolean        on)
{
  Widget mbutton = findButton(button);

  if (mbutton) XtSetSensitive(mbutton, (Bool) on);
}

void
Dialog::setVisibility(StandardButton button,
		      Boolean        on)
{
  Widget mbutton = findButton(button);

  if (mbutton) 
    on ? XtManageChild(mbutton) : XtUnmanageChild(mbutton);
}

static char *LabelStringLookUp[] = {
  XmNokLabelString,
  XmNapplyLabelString,
  XmNcancelLabelString,
  XmNhelpLabelString
  };

void
Dialog::setButtonLabel(StandardButton button,
		       const CString &text)
{
  XmString label = XmStringCreateSimple((char *) text);
  XtVaSetValues(rtr.mrep,
		LabelStringLookUp[button], label,
		NULL);
  XmStringFree(label);
}
