//############################################################
// GraphicalObject.C
// Kari Pulli
// 12/10/95
// A base class for widgets for a Motif GUI library
//############################################################

#include "GraphicalObject.h"

GraphicalObject::GraphicalObject(void)
: GraphicalEnv()
{
  static Boolean classInit = FALSE;

  if (classInit == FALSE) {
    classConstruct();
    classInit = TRUE;
  }

  rtr.lrep = this;
  rtList.Append(&rtr);
  // NOTE: the derived classes have to set rtr.mrep!!
  dimsMonitored = FALSE;
  gobjWidth = gobjHeight = 0;
}

#include "Color.h"

RepList GraphicalObject::rtList;
Widget  GraphicalObject::topLevel = (Widget) NULL;

void
GraphicalObject::classConstruct(void)
{
  topLevel = XtVaAppCreateShell(appName,
				appClass,
				applicationShellWidgetClass,
				display,
				XmNvisual, Color::getVisual(),
				XmNcolormap, Color::getColorMap(),
				NULL);
}

GraphicalObject::~GraphicalObject(void)
{
  // remove rtr from rtList

  XtDestroyWidget(rtr.mrep);
  // make it happen now!
  XSync(display, FALSE);
  XmUpdateDisplay(topLevel);
}

static char *callbackTable[] = {
  XmNactivateCallback, XmNapplyCallback,
  XmNcancelCallback,   XmNhelpCallback,
  XmNokCallback,       XmNvalueChangedCallback
  };

void
internalCB(Widget    w,
	   XtPointer clientd,
	   XtPointer )
{
  GraphicalObject::RepRec *rtr = find(GraphicalObject::rtList, w);

  if (rtr == NULL) {
    cerr << "Could not find the widget!" << endl;
    return;
  }

  GraphicalObject::HandlerRec *hrp = (GraphicalObject::HandlerRec*) clientd;
 
  // call the user function
  hrp->ch(*rtr->lrep,
	  hrp->cbKind,
	  hrp->udata);
}

Boolean
GraphicalObject::registerCallback(CallbackType kind,
				  CallbackHandler ch,
				  void *udata)
{
  if (find(cbHandlers, kind)) {
    cerr << "We already have " << callbackTable[kind] << "!" 
      << endl << __FILE__ << __LINE__ << endl;
    return FALSE;
  }
  
  HandlerRec *hrp = new HandlerRec(kind, ch, udata);
  cbHandlers.Append(hrp);

  XtAddCallback(rtr.mrep,
		callbackTable[kind],
		internalCB,
		(XtPointer) hrp);
  return TRUE;
}

Boolean
GraphicalObject::unregisterCallback(CallbackType kind)
{
  HandlerRec *hrp = find(cbHandlers, kind);

  if (hrp == NULL) {
    cerr << "Could not find callback handler!" << endl;
    return FALSE;
  }

  XtRemoveCallback(rtr.mrep,
		   callbackTable[kind],
		   internalCB,
		   hrp->udata);
  // remove handler from the list
  delete hrp;

  return TRUE;
}

static EventMask eventMaskTable[] = {
  NoEventMask,
  KeyPressMask,
  ButtonPressMask,
  PointerMotionMask | ButtonMotionMask,
  EnterWindowMask,
  LeaveWindowMask
  };


Boolean
GraphicalObject::registerEventHandler(LEvent::Kind kind,
				      EventHandler eh,
				      void *udata)
{
  if (find(evHandlers, kind)) {
    cerr << "We already have this event handler!"
      << endl << __FILE__ << __LINE__ << endl;
    return FALSE;
  }
  
  HandlerRec *hrp = new HandlerRec(kind, eh, udata);
  evHandlers.Append(hrp);

  XtAddEventHandler(rtr.mrep,
		    eventMaskTable[kind],
		    False,
                    (XtEventHandler) internalEH,
		    (XtPointer) hrp);
  return TRUE;
}

void
internalEH(Widget    w,
	   XtPointer clientd,
	   XEvent *  event)
{
  GraphicalObject::RepRec *rtr = find(GraphicalObject::rtList, w);

  if (rtr == NULL) {
    cerr << "Could not find the widget!" << endl;
    return;
  }

  LEvent levent(*event);
  GraphicalObject::HandlerRec *hrp = (GraphicalObject::HandlerRec *) clientd;
 
  // check the event is valid
  if (levent.kind() != hrp->evKind) {
    cerr << "event error" << endl;
    return;
  }

  // call the user function
  hrp->eh(*rtr->lrep,
	  levent,
	  hrp->udata);
}

Boolean
GraphicalObject::unregisterEventHandler(LEvent::Kind kind)
{
  HandlerRec *hrp = find(evHandlers, kind);

  if (hrp == NULL) {
    cerr << "Could not find event handler!" << endl;
    return FALSE;
  }

  XtRemoveEventHandler(rtr.mrep,
		       eventMaskTable[kind],
		       True,
		       (XtEventHandler)internalEH,
		       (XtPointer) hrp);
  // remove handler from the list
  delete hrp;

  return TRUE;
}


int
GraphicalObject::x(void) const
{
  Position x;
  XtVaGetValues(rtr.mrep, XmNx, &x, NULL);
  return x;
}

int
GraphicalObject::y(void) const
{
  Position y;
  XtVaGetValues(rtr.mrep, XmNy, &y, NULL);
  return y;
}

void
GraphicalObject::setPosition(int x, int y)
{
  XtVaSetValues(rtr.mrep,
		XmNx, (Position) x,
		XmNy, (Position) y,
		NULL);
}

uint
GraphicalObject::getWidth(void) const
{
  if (dimsMonitored) return gobjWidth;
  Dimension w;
  XtVaGetValues(rtr.mrep, XmNwidth, &w, NULL);
  return (uint) w;
}

uint
GraphicalObject::getHeight(void) const
{
  if (dimsMonitored) return gobjHeight;
  Dimension h;
  XtVaGetValues(rtr.mrep, XmNheight, &h, NULL);
  return (uint) h;
}

void
GraphicalObject::setDimensions(uint w, uint h)
{
  if (w==0 || h==0) return;
  XtVaSetValues(rtr.mrep,
		XmNwidth,  (Dimension) w,
		XmNheight, (Dimension) h,
		NULL);
}

void
monitorCB(Widget    widget,
	  XtPointer clientd,
	  XtPointer calld)
{
  GraphicalObject *gobj = (GraphicalObject *)clientd;
  XmAnyCallbackStruct *cbs = (XmAnyCallbackStruct *) calld;

  if (cbs->reason == XmCR_EXPOSE ||
      cbs->reason == XmCR_RESIZE && cbs->event == NULL) {
    Dimension w=0, h=0;
    XtVaGetValues(widget,
		  XmNwidth,  &w,
		  XmNheight, &h,
		  NULL);
    gobj->gobjWidth  = (uint) w;
    gobj->gobjHeight = (uint) h;
  } else if (cbs->reason == XmCR_RESIZE) {
    gobj->gobjWidth  = (uint) cbs->event->xresizerequest.width;
    gobj->gobjHeight = (uint) cbs->event->xresizerequest.height;
  }
}

void
GraphicalObject::monitorDimensions(void)
{
  // init gobjWidth, gobjHeight when widget is mapped to screen
  XtAddCallback(rtr.mrep,
		XmNexposeCallback, monitorCB, this);
  // change gobjWidth, gobjHeight when resized
  XtAddCallback(rtr.mrep,
		XmNresizeCallback, monitorCB, this);
  dimsMonitored = TRUE;
}

GraphicalObject::HandlerRec *
find(HandlerList &hl, int k)
{
  if (!hl.GetCount()) return NULL;
  hl.ToHead();
  do {
    if ((int) hl.Get()->evKind == k)
      return hl.Get();
  } while (hl.ToNext());
  return NULL;
}

GraphicalObject::RepRec *
find(RepList &hl, const Widget &w)
{
  if (!hl.GetCount()) return NULL;
  hl.ToHead();
  do {
    if (hl.Get()->mrep == w)
      return hl.Get();
  } while (hl.ToNext());
  return NULL;
}

void
GraphicalObject::setCursor(GUI_Cursor c)
{
  Window win = XtWindow(rtr.mrep);
  if (c != DEFAULT) {
    Cursor curs = XCreateFontCursor(display, c);
    XDefineCursor(display, win, curs);
    XFreeCursor(display, curs);
  }
  else XUndefineCursor(display, win);
  XFlush(display);
}


