//############################################################
// GraphicalObject.h
// Kari Pulli
// 12/10/95
// A base class for widgets for a Motif GUI library
//############################################################

#ifndef _GRAPHICALOBJECT_H_
#define _GRAPHICALOBJECT_H_

extern "C" {
#include <Xm/Xm.h>
}

#include "GraphicalEnv.h"
#include "LEvent.h"
#include "slist.h"
#include "Color.h"

#define HandlerList  SListDynamic<GraphicalObject::HandlerRec *>
#define RepList      SListDynamic<GraphicalObject::RepRec *>

class GraphicalObject : public GraphicalEnv {
public:
  
  enum CallbackType {
    CB_ACTIVE, CB_APPLY, CB_CANCEL,
    CB_HELP,   CB_OK,    CB_VALUE_CHANGED, CB_TYPES
    };

  typedef void (* EventHandler)    (GraphicalObject &,
				    LEvent &,
				    void *);
  typedef void (* CallbackHandler) (GraphicalObject &,
				    CallbackType,
				    void *);

  struct HandlerRec {
    union {
      LEvent::Kind  evKind;
      CallbackType  cbKind;
    };
    union {
      EventHandler    eh;
      CallbackHandler ch;
    };
    void * udata; // user data

    HandlerRec(void) { eh = NULL; }
    HandlerRec(CallbackType    kind,
	       CallbackHandler handler,
	       void *          ud)
      { cbKind = kind; ch = handler; udata = ud; }
    HandlerRec(LEvent::Kind  kind,
	       EventHandler  handler,
	       void *        ud)
      { evKind = kind; eh = handler; udata = ud; }
    HandlerRec(const HandlerRec &hr)
      { *this = hr; }
    ~HandlerRec(void) {}

    const HandlerRec & operator =(const HandlerRec &hr)
      {
	udata   = hr.udata;
	evKind  = hr.evKind;
	eh      = hr.eh;
	return *this;
      }

    Boolean operator ==(const HandlerRec &hr) const
      { return evKind == hr.evKind; }
  };

  struct RepRec {
    Widget            mrep; // X representation
    GraphicalObject * lrep; // library representation

    RepRec(void) { lrep = NULL; }
    RepRec(const RepRec &rtr) { *this = rtr; }
    ~RepRec(void) {}

    const RepRec & operator =(const RepRec &rtr)
      { mrep = rtr.mrep; lrep = rtr.lrep; return *this; }

    Boolean operator ==(const RepRec &rtr) const
      { return lrep == rtr.lrep; }
  };

  static Widget  topLevel;
  static RepList rtList;
protected:
  RepRec rtr; // map to system representation

  HandlerList evHandlers;
  HandlerList cbHandlers;

private:
  
  friend void internalEH(Widget w,
			 XtPointer clientd,
			 XEvent *xevent);
  friend void internalCB(Widget w,
			 XtPointer clientd,
			 XtPointer calld);
  friend void monitorCB (Widget w,
			 XtPointer clientd,
			 XtPointer calld);
  Boolean dimsMonitored; // Are we monitoring

  static void classConstruct(void);

protected:
  
  GraphicalObject(void);

public:

  virtual ~GraphicalObject(void);

  Boolean registerEventHandler(LEvent::Kind  kind,
			       EventHandler  handler,
			       void *        ud = NULL);
  Boolean unregisterEventHandler(LEvent::Kind k);
  Boolean registerCallback(CallbackType     type,
			   CallbackHandler  cb,
			   void *           ud = NULL);
  Boolean unregisterCallback(CallbackType k);

  virtual void manage(void) const
    { XtManageChild(rtr.mrep); }
  virtual void unmanage(void) const
    { XtUnmanageChild(rtr.mrep); }

  virtual void realize(void) const
    { XtRealizeWidget(rtr.mrep); }
  virtual void unrealize(void) const
    { XtUnrealizeWidget(rtr.mrep); }
  virtual Boolean isRealized(void) const
    { return XtIsRealized(rtr.mrep); }
  static  void realizeAll(void)
    { XtRealizeWidget(topLevel); }

  void sensitize(void) const
    { XtSetSensitive(rtr.mrep, True); }
  void desensitize(void) const
    { XtSetSensitive(rtr.mrep, False); }
  Boolean isSensitive(void) const
    { return XtIsSensitive(rtr.mrep); }

  static void raise(void)
    { XRaiseWindow(display, XtWindow(topLevel)); }
  static void lower(void)
    { XLowerWindow(display, XtWindow(topLevel)); }
  static void iconify(void)
    { XtVaSetValues(topLevel, XmNiconic, True, NULL); }

  uint getHeight(void) const;
  uint getWidth (void) const;
  void setDimensions(uint w, uint h);
  uint gobjWidth;
  uint gobjHeight;
  void monitorDimensions(void);

  int x(void) const;
  int y(void) const;
  void setPosition(int x, int y);
    // relative to parent

  operator Widget(void) const
    { return rtr.mrep; }

  static Widget getTopLevel(void)
    { return topLevel; }

  Widget parent(void)
    { return XtParent(rtr.mrep); }

  Widget shell(void)
    { 
      for (Widget w = rtr.mrep; w&&!XtIsShell(w); w=XtParent(w)) ;
      return w;
    }
  
  enum GUI_Cursor { DEFAULT, WATCH = XC_watch };
  void setCursor(GUI_Cursor c);
    // change the cursor in the current window

private:

friend class HandlerList;
friend class RepList;
friend  GraphicalObject::HandlerRec *find(HandlerList &hl, int k);
friend  GraphicalObject::RepRec     *find(RepList &hl, const Widget &w);

};

#endif
