//############################################################
// PixelMap.C
// Kari Pulli
// 12/09/95
// The pixelmap class for a Motif GUI library
//############################################################

#include <iostream.h>
#include <assert.h>
#include "Color.h"
#include "PixelMap.h"
#include "Context.h"

#define SHOW(x) cout << #x << " = " << x << endl
#define TICK()  cout << "." << endl;

Boolean
PixelMap::resize(uint w, uint h)
{
  if (w == 0 || h == 0) return FALSE;

  if (width == w && height == h) return TRUE;

  if (pixmap) XFreePixmap(display, pixmap);
  width = w, height = h;
  pixmap = XCreatePixmap(display, root,
			 width, height, depth);
  return TRUE;
}

PixelMap::PixelMap(const PixelMap &other) 
: GraphicalEnv()
{
  if (!other.pixmap) {
    zero(); 
    return;
  }

  width  = other.width;
  height = other.height;

  pixmap = XCreatePixmap(display, root,
			 width, height, depth);
  XCopyArea(display,
	    other.pixmap,
	    pixmap,
	    DefaultGC(display, screen),
	    0, 0,
	    width, height,
	    0, 0);
}

PixelMap::~PixelMap(void)
{
  if (pixmap) XFreePixmap(display, pixmap);
}

#include <fstream.h>

static int
readColors(ifstream &inFile, CString colors[10])
{
  CString color;
  uint    start, finish;
  int     i = 0;

  // read colors until a blank line
  while (TRUE) {
    inFile >> color;
    if (color.isBlank()) {
      break;
    } else {
      start   = color.find('\"') + 1;
      finish  = color.rfind('\"');
      if (start > finish) return FALSE;
      if (i<10) {
	colors[i++] = color.subString(start, finish-start);
      }
    }
  }
  return i;
}

inline int ctoi(char c) { return int(c - '0'); }

static Boolean
readImage(ifstream  &inFile,
	  CString    colors[10],
	  int        nCol,
	  PixelMap  &pm)
{
  uint       width  = pm.getWidth(),
             height = pm.getHeight(),
             i;
  CString    textImage, string;

  // read in image specs
  for (i=0; i<height; i++) {
    inFile >> string;
    string.trim();
    if (string.length() != width) return FALSE;
    textImage += string;
  }
/*
  Context context(Color::defForeName(), Color::defBackName());
  uint j;
  int      colIndex, prevIndex = -1;
  Display *display   = getDisplay();

  SHOW(height); SHOW(width);
  for (i=0; i<height; i++) {
    for (j=0; j<width; j++) {
      colIndex = ctoi(textImage[i*width+j]);
      assert(colIndex>=0);
      if (colIndex >= nCol) colIndex = 0;
      if (colIndex != prevIndex)
	context.setForeground(colors[colIndex]);
      XDrawPoint(display, (Pixmap) pm, context, j, i);
      prevIndex = colIndex;
    }
  }
*/
  Display *display   = getDisplay();
  int      depth     = getDepth();
  int      screen    = getScreenNumber();
  int      colIndex, prevIndex = -1;
  ulong    cmapVal;
  uint     j;

  XImage *img = XCreateImage(display,
			     DefaultVisual(display, screen),
			     depth,
			     ZPixmap,
			     0, None,
			     width, height,
			     BitmapPad(display), 0);

  img->data = new char[height*img->bytes_per_line];

  for (i=0; i<height; i++) {
    for (j=0; j<width; j++) {
      colIndex = ctoi(textImage[i*width+j]);
      assert(colIndex>=0);
      if (colIndex >= nCol) colIndex = 0;
      if (colIndex != prevIndex)
	cmapVal = (ulong) Color(colors[colIndex]);
      XPutPixel(img, j, i, cmapVal);
      prevIndex = colIndex;
    }
  }

  XPutImage(display,
	    (Pixmap) pm,
	    DefaultGC(display, screen),
	    img,
	    0, 0,
	    0, 0,
	    width, height);

  delete[] img->data;
  XFree((char *)img);

  return TRUE;
}

void
PixelMap::clear(const CString &colName)
{
  GC        gc;
  XGCValues vals;
  Color     col(colName);

  vals.foreground  = (ulong) col;
  gc               = XCreateGC(display, pixmap,
			       GCForeground, &vals);
  XFillRectangle(display, pixmap, gc, 0, 0, width, height);
  XFreeGC(display, gc);
}

Boolean
PixelMap::read(const CString &fileName)
{
  ifstream inFile((const char *)fileName);
  if (!inFile) return FALSE;
  cout << "File open OK" << endl;

  uint w,h;
  inFile.seekg(0L, ios::beg);
  inFile >> w >> h;
  if (w==0 || h==0) return FALSE;
  inFile.get(); // eat rest of line

  CString colors[10];
  int     nCols;
  if (!(nCols = readColors(inFile, colors))) return FALSE;

  resize(w,h);

  if (!readImage(inFile, colors, nCols, *this)) return FALSE;

  return TRUE;
}
