//############################################################
// Trackball.cc
// Kari Pulli
// 03/06/97
//############################################################

#include <iostream.h>
#include "Trackball.h"

/*
 * (c) Copyright 1993, 1994, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 * Permission to use, copy, modify, and distribute this software for
 * any purpose and without fee is hereby granted, provided that the above
 * copyright notice appear in all copies and that both the copyright notice
 * and this permission notice appear in supporting documentation, and that
 * the name of Silicon Graphics, Inc. not be used in advertising
 * or publicity pertaining to distribution of the software without specific,
 * written prior permission.
 *
 * THE MATERIAL EMBODIED ON THIS SOFTWARE IS PROVIDED TO YOU "AS-IS"
 * AND WITHOUT WARRANTY OF ANY KIND, EXPRESS, IMPLIED OR OTHERWISE,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY OR
 * FITNESS FOR A PARTICULAR PURPOSE.  IN NO EVENT SHALL SILICON
 * GRAPHICS, INC.  BE LIABLE TO YOU OR ANYONE ELSE FOR ANY DIRECT,
 * SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY
 * KIND, OR ANY DAMAGES WHATSOEVER, INCLUDING WITHOUT LIMITATION,
 * LOSS OF PROFIT, LOSS OF USE, SAVINGS OR REVENUE, OR THE CLAIMS OF
 * THIRD PARTIES, WHETHER OR NOT SILICON GRAPHICS, INC.  HAS BEEN
 * ADVISED OF THE POSSIBILITY OF SUCH LOSS, HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE
 * POSSESSION, USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * US Government Users Restricted Rights
 * Use, duplication, or disclosure by the Government is subject to
 * restrictions set forth in FAR 52.227.19(c)(2) or subparagraph
 * (c)(1)(ii) of the Rights in Technical Data and Computer Software
 * clause at DFARS 252.227-7013 and/or in similar or successor
 * clauses in the FAR or the DOD or NASA FAR Supplement.
 * Unpublished-- rights reserved under the copyright laws of the
 * United States.  Contractor/manufacturer is Silicon Graphics,
 * Inc., 2011 N.  Shoreline Blvd., Mountain View, CA 94039-7311.
 *
 * OpenGL(TM) is a trademark of Silicon Graphics, Inc.
 */
/*
 * Trackball code:
 *
 * Implementation of a virtual trackball.
 * Implemented by Gavin Bell, lots of ideas from Thant Tessman and
 *   the August '88 issue of Siggraph's "Computer Graphics," pp. 121-129.
 *
 * Vector manip code:
 *
 * Original code from:
 * David M. Ciemiewicz, Mark Grossman, Henry Moreton, and Paul Haeberli
 *
 * Much mucking with by:
 * Gavin Bell
 * Kept maths, changed just about everything else:
 * Kari Pulli
 */
#include <math.h>

/*
   This size should really be based on the distance from the 
   center of rotation to the point on the object underneath the 
   mouse.  That point would then track the mouse as closely as 
   possible.  This is a simple example, though, so that is left 
   as an Exercise for the Programmer.
*/
#define TRACKBALLSIZE  (0.8)

#define SHOW(x) cout << #x " = " << x << endl

// Local function prototypes (not defined in trackball.h)
static float tb_project_to_sphere(float, float, float);
static void normalize_quat(float [4]);


//  Given an axis and angle, compute quaternion.
void
axis_to_quat(Pnt3 &a, float phi, float q[4])
{
  a.normalize();
  a *= sin(phi/2.0);
  q[0]=a[0]; q[1]=a[1]; q[2]=a[2];
  q[3] = cos(phi/2.0);
}

/*
   Ok, simulate a track-ball.  Project the points onto the virtual
   trackball, then figure out the axis of rotation, which is the 
   cross product of P1 P2 and O P1 (O is the center of the ball, 
   0,0,0)
   Note:  This is a deformed trackball-- is a trackball in the 
   center, but is deformed into a hyperbolic sheet of rotation 
   away from the center.  This particular function was chosen 
   after trying out several variations.

   It is assumed that the arguments to this routine are in the 
   range (-1.0 ... 1.0)
 */
void
trackball(float q[4], float p1x, float p1y, float p2x, float p2y)
{
  if (p1x == p2x && p1y == p2y) {
    // Zero rotation 
    q[0] = q[1] = q[2] = 0.0;
    q[3] = 1.0;
    return;
  }

  // First, figure out z-coordinates for projection of P1 and P2 
  // to deformed sphere
  Pnt3 p1(p1x,p1y,tb_project_to_sphere(TRACKBALLSIZE,p1x,p1y));
  Pnt3 p2(p2x,p2y,tb_project_to_sphere(TRACKBALLSIZE,p2x,p2y));
  
  // get axis of rotation
  Pnt3 a = cross(p2,p1);

  // Figure out how much to rotate around that axis.
  float t = dist(p1,p2) / (2.0*TRACKBALLSIZE);
  
  // Avoid problems with out-of-control values...
  if (t > 1.0) t = 1.0;
  if (t < -1.0) t = -1.0;
  float phi = 2.0 * asin(t); // how much to rotate about axis
  
  axis_to_quat(a,phi,q);
}

/*
   Project an x,y pair onto a sphere of radius r OR a hyperbolic 
   sheet if we are away from the center of the sphere.
 */
static float
tb_project_to_sphere(float r, float x, float y)
{
  float d, t, z;
  
  d = sqrt(x*x + y*y);
  if (d < r * 0.70710678118654752440) {    // Inside sphere
    z = sqrt(r*r - d*d);
  } else {                                 // On hyperbola
    t = r / 1.41421356237309504880;
    z = t*t / d;
  }
  return z;
}

/*
   Given two rotations, e1 and e2, expressed as quaternion 
   rotations, figure out the equivalent single rotation and stuff
   it into dest.
   This routine also normalizes the result every RENORMCOUNT 
   times it is called, to keep error from creeping in.

   NOTE: This routine is written so that q1 or q2 may be the same
   as dest (or each other).
*/

#define RENORMCOUNT 97

void
add_quats(float q1[4], float q2[4], float dest[4])
{
  static int count=0;

  Pnt3 t1(q1);
  t1 *= q2[3];
  
  Pnt3 t2(q2);
  t2 *= q1[3];

  Pnt3 t3 = cross(q2,q1);
  t3 += t1; t3 += t2;
  
  // watch out: q1 or q2 may be the same as dest!
  dest[3] = q1[3] * q2[3] - dot(q1,q2);
  dest[0] = t3[0];
  dest[1] = t3[1];
  dest[2] = t3[2];

  if (++count > RENORMCOUNT) {
    count = 0;
    normalize_quat(dest);
  }
}

/*
   Quaternions always obey:  a^2 + b^2 + c^2 + d^2 = 1.0
   If they don't add up to 1.0, dividing by their magnitude will
   renormalize them.
   
   Note: See the following for more information on quaternions:
   
   - Shoemake, K., Animating rotation with quaternion curves, 
     Computer Graphics 19, No 3 (Proc. SIGGRAPH'85), 245-254,1985.
     - Pletinckx, D., Quaternion calculus as a basic tool in 
       computer graphics, The Visual Computer 5, 2-13, 1989.
*/
static void
normalize_quat(float q[4])
{
  int i;
  float mag;
  
  mag = (q[0]*q[0] + q[1]*q[1] + q[2]*q[2] + q[3]*q[3]);
  for (i = 0; i < 4; i++) q[i] /= mag;
}

// Build a rotation matrix, given a quaternion rotation.
void
build_rotmatrix(float m[4][4], float q[4])
{
  m[0][0] = 1.0 - 2.0 * (q[1] * q[1] + q[2] * q[2]);
  m[0][1] = 2.0 * (q[0] * q[1] - q[2] * q[3]);
  m[0][2] = 2.0 * (q[2] * q[0] + q[1] * q[3]);
  m[0][3] = 0.0;

  m[1][0] = 2.0 * (q[0] * q[1] + q[2] * q[3]);
  m[1][1] = 1.0 - 2.0 * (q[2] * q[2] + q[0] * q[0]);
  m[1][2] = 2.0 * (q[1] * q[2] - q[0] * q[3]);
  m[1][3] = 0.0;

  m[2][0] = 2.0 * (q[2] * q[0] - q[1] * q[3]);
  m[2][1] = 2.0 * (q[1] * q[2] + q[0] * q[3]);
  m[2][2] = 1.0 - 2.0 * (q[1] * q[1] + q[0] * q[0]);
  m[2][3] = 0.0;

  m[3][0] = 0.0;
  m[3][1] = 0.0;
  m[3][2] = 0.0;
  m[3][3] = 1.0;
}

//############################################################
//
// Code written by Kari Pulli starts
//
//############################################################

Trackball::Trackball(void)
{
  trans = Pnt3();
  rotating = spinning = panning = zooming = 0;
  trackball(currQuat, 0.0, 0.0, 0.0, 0.0);
}

void
Trackball::setPerspXform(void)
{
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(field_of_view,
		 W/H, // aspect ratio
		 //		 1.0, // aspect ratio
		 znear, zfar);
}

void
Trackball::setSize(int w, int h)
{ 
  W=w; H=h; 
  glViewport(0, 0, w, h);
}

// c  is camera center, 
// o  is wherethe camera is looking at,
// up is the up vector of the camera
void
Trackball::setup(float _c[3], float _o[3], float up[3],
		 float r, float fov)
{
  o = Pnt3(_o);
  c = Pnt3(_c);

  radius = r;
  float d_oc = dist(o,c);
  zfar  = d_oc+radius;
  znear = zfar - 2*radius;
  if (znear < .01) znear = .01;
  field_of_view = fov;

  setPerspXform();

  trans = Pnt3();
  rotating = spinning = panning = zooming = 0;
  trackball(currQuat, 0.0, 0.0, 0.0, 0.0);
  // z is from the object to camera
  z = c - o;
  d = z.norm();
  z /= d;
  // u (x vector) is up X z
  u = cross(up,z);
  u.normalize();
  // v (y vector) is z X u
  v = cross(z,u);
}

void
Trackball::pressButton(int button, int up, int x, int y)
{
  beginx = x;
  beginy = (H-1)-y;
  if (button == 1 && up == 0) {
    if (panning) 
      zooming = 1;
    else {
      rotating = 1;
      spinning = 0;
    }
    panning = 0;
  }
  if (button == 1 && up == 1) {
    rotating = 0;
    zooming = 0;
  }
  if (button == 2 && up == 0) {
    if (rotating)
      zooming = 1;
    else 
      panning = 1;
    rotating  = 0;
  }
  if (button == 2 && up == 1) {
    panning = 0;
    zooming = 0;
  }
/*
  cout << "press button" << endl;
  SHOW(zooming);
  SHOW(panning);
  SHOW(rotating);
  SHOW(spinning);
  SHOW(beginx);
  SHOW(beginy);
*/
}


void
Trackball::move(int x, int y)
{
  y = (H-1)-y;
  if (rotating) {
    trackball(lastQuat,
	      2.0 * beginx / W - 1.0,
	      2.0 * beginy / H - 1.0,
	      2.0 * x / W - 1.0,
	      2.0 * y / H - 1.0);
    spinning = 1;
  }
  if (panning) {
    float factor = .8*(d-trans[2]);
    trans[0] += factor*(x-beginx)/W;
    trans[1] += factor*(y-beginy)/H;
  }
  if (zooming) {
    float tmp = .8*(d-trans[2])* (beginy-y)/H;
    trans[2] += tmp;
    zfar  -= tmp;
    znear = zfar - 2*radius;
    if (znear < .01) znear = .01;
    setPerspXform();
  }
  beginx = x;
  beginy = y;
/*
  cout << "move" << endl;
  SHOW(zooming);
  SHOW(panning);
  SHOW(rotating);
  SHOW(spinning);
  SHOW(beginx);
  SHOW(beginy);
*/
}

void
Trackball::getXform(float m[4][4], float t[3])
{
  build_rotmatrix(m, currQuat);
  for (int i=0; i<3; i++)
    t[i] = trans[0]*u[i] + trans[1]*v[i] + trans[2]*z[i];
}

void
Trackball::applyXform(void)
{
  spin();
  float m[4][4], t[3];
  getXform(m,t);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glTranslatef(-c[0], -c[1], -c[2]); // move the camera
  glTranslatef(t[0], t[1], t[2]);    // pan and zoom
  glTranslatef(o[0], o[1], o[2]);    // rotate around origin
  glMultMatrixf(&m[0][0]);
  glTranslatef(-o[0], -o[1], -o[2]);
}

void
Trackball::spin(void)
{
  if (spinning) {
    add_quats(lastQuat, currQuat, currQuat);
  }
}

Pnt3 
Trackball::dir(void)
{
  float m[4][4];
  build_rotmatrix(m, currQuat);
  return Pnt3(m[2]);
}
