//=============================================================
// Singly-Linked Lists      slist.h         v1.00
//      Defines templates for classes of singly-linked lists
//      Copyright 1995 Scott Robert Ladd
//=============================================================

#ifndef SLIST_H
#define SLIST_H

#include <assert.h>
#include <stdlib.h>
#include <iostream.h>
//#include <limits.h>
#include "Boolean.h"

//---------------------------
// dynamic singly-linked list
//---------------------------

template <class T>
class SListDynamic {
public:
  // constructors
  SListDynamic
    (
     Boolean circ = FALSE
     );
  
  // copy constructor
  SListDynamic
    (
     const SListDynamic<T> & slst,
     Boolean shallow = FALSE
     );

  // construct from array
  SListDynamic
    (
     const T * array,
     size_t no
     );

  // destructor
  ~SListDynamic();
  
  // assignment operator (deep copy)
  void operator =
    (
     const SListDynamic<T> & slst
     );
  
  // conversion to array operator
  operator T * ();
  
  // append new items
  void Append(const T & item)
    {
      Node * n = new Node;
      assert(n);
      
      n->Data = item;
      n->Next = NULL;
      
      if (Tail == NULL) {
	Head = n;
	Tail = n;
	Work = n;
      } else {
	Tail->Next = n;
	Tail = n;
      }
      ++Count;
    }


  void Append(const T * array, size_t no)
    {
      if ((array == NULL) || (no == 0))
	return;
      const T * aptr = array;
      for (size_t n = 0; n < no; ++n) {
	Append(*aptr);
	++aptr;
      }
    }

  
  void Append(const SListDynamic<T> & slst)
    {
      const Node * n = slst.Head;
      while (n != NULL) {
	Append(n->Data);
	n = n->Next;
      }
    }

  // remove all items from the list
  void Erase()
    {
      if (!Shallow) {
	Work = Head;
	while (Work != NULL) {
	  Head = Work->Next;
	  delete Work;
	  Work = Head;
	}
      }
      SetNull();
      Shallow = FALSE;
    }

  // get current item
  T Get() { assert(Count && Work); return Work->Data; }
  
  // TRUE if current item is last item
  Boolean AtTail();
  
  // set current item to head of list
  void ToHead();
  
  // move to next item in list
  Boolean ToNext()
    {
      if (!Count || !Work) return FALSE;
      if (Work == Tail) {
	if (Circular)
	  Work = Head;
	else
	  return FALSE;
      } else
	Work = Work->Next;
      return TRUE;
    }


  // interrogate list
  size_t  GetCount();
  Boolean IsShallow();
  Boolean IsCircular();
  
protected:
  // type defining a node in the list
  struct Node
    {
      T Data;
      Node * Next;
    };

  size_t Count; // # of items in list
  Node * Head;  // first item
  Node * Tail;  // last item
  Node * Work;  // current item

  Boolean Shallow;  // Is this a shallow copy?
  Boolean Circular; // is this a circular list?

  // internal utility functions
  void DeepCopy
    (
     const SListDynamic<T> & slst
     );

  void ShallowCopy
    (
     const SListDynamic<T> & slst
     );

  void SetNull();
};

// implement utility functions
template <class T>
void SListDynamic<T>::DeepCopy
(
 const SListDynamic<T> & slst
 )
{
  if (slst.Count == 0)
    return;

  Node * n = slst.Head;

  do  {
    Append(n->Data);
    n = n->Next;
  }
  while (n != NULL);

  Work = Head;
  Shallow  = FALSE;
  Circular = slst.Circular;
}

template <class T>
inline void SListDynamic<T>::ShallowCopy
(
 const SListDynamic<T> & slst
 )
{
  Head  = slst.Head;
  Tail  = slst.Tail;
  Work  = slst.Work;
  Count = slst.Count;
  Shallow  = TRUE;
  Circular = slst.Circular;
}

template <class T>
inline void SListDynamic<T>::SetNull()
{
  Head  = NULL;
  Tail  = NULL;
  Work  = NULL;
  Count = 0;
}

// constructors
template <class T>
inline SListDynamic<T>::SListDynamic
(
 Boolean circ
 )
{
  SetNull();
  Shallow  = FALSE;
  Circular = circ;
}

// copy constructor
template <class T>
SListDynamic<T>::SListDynamic
(
 const SListDynamic<T> & slst,
 Boolean shallow
 )
{
  if (shallow)
    ShallowCopy(slst);
  else
    {
      SetNull();
      DeepCopy(slst);
    }
}

// construct from array
template <class T>
SListDynamic<T>::SListDynamic
(
 const T * array,
 size_t no
 )
{
  SetNull();

  Circular = FALSE;
  Shallow  = FALSE;

  if ((array == NULL) || (no == 0))
    return;

  const T * aptr = array;

  for (size_t n = 0; n < no; ++n)
    {
      Append(*aptr);
      ++aptr;
    }
}

// destructor
template <class T>
inline SListDynamic<T>::~SListDynamic()
{
  Erase();
}

// assignment operator (deep copy)
template <class T>
void SListDynamic<T>::operator =
(
 const SListDynamic<T> & slst
 )
{
  Erase();
  DeepCopy(slst);
}

// conversion to array operator
template <class T>
SListDynamic<T>::operator T * ()
{
  if (Count == 0)
    return NULL;

  T * result = new T [Count];
  assert(result);

  Node * temp = Head;
  T    * aptr = result;

  while (temp != NULL)
    {
      *aptr = temp->Data;
      ++aptr;
      temp = temp->Next;
    }

  // note: caller must delete this array!
  return result;
}

// TRUE if current item is last item
template <class T>
inline Boolean SListDynamic<T>::AtTail()
{
  if (Work == Tail)
    return TRUE;
  else
    return FALSE;
}

// set current item to head of list
template <class T>
inline void SListDynamic<T>::ToHead()
{
  Work = Head;
}

// interrogate list
template <class T>
inline size_t SListDynamic<T>::GetCount()
{
  return Count;
}

template <class T>
inline Boolean SListDynamic<T>::IsShallow()
{
  return Shallow;
}

template <class T>
inline Boolean SListDynamic<T>::IsCircular()
{
  return Circular;
}

//--------------------------
// static singly-linked list
//--------------------------

template <class T, size_t Limit>
class SList
{
public:
  // constructors
  SList
    (
     Boolean circ = FALSE
     );

  // copy constructor
  SList
    (
     const SList<T,Limit> & slst
     );

  // construct from array
  SList
    (
     const T * array,
     size_t no
     );

  // destructor
  ~SList();

  // assignment operator (deep copy)
  void operator =
    (
     const SList<T,Limit> & slst
     );

  // conversion to array operator
  operator T * ();

  // append new items
  void Append
    (
     const T & item
     );

  void Append
    (
     const T * array,
     size_t no
     );

  void Append
    (
     const SList<T,Limit> & slst
     );

  // remove all items from the list
  void Erase();

  // get current item
  T Get();

  // TRUE if current item is last item
  Boolean AtTail();

  // set current item to head of list
  void ToHead();

  // move to next item in list
  Boolean ToNext();

  // interrogate list
  size_t  GetCount();
  size_t  GetLimit();
  Boolean IsCircular();

protected:
  size_t Count;      // # of items in list
  size_t Work;       // currently-selected element
  T Data[Limit];     // elements in list

  Boolean Circular; // is this a circular list?

  // internal utility functions
  void DeepCopy
    (
     const SList<T,Limit> & slst
     );
};

// implement utility functions
template <class T, size_t Limit>
void SList<T,Limit>::DeepCopy
(
 const SList<T,Limit> & slst
 )
{
  Count = slst.Count;
  Work  = slst.Work;

  for (size_t n = 0; n < Count; ++n)
    Data[n] = slst.Data[n];

  Circular = slst.Circular;
}

// remove all items from the list
template <class T, size_t Limit>
inline void SList<T,Limit>::Erase()
{
  Count = 0;
  Work  = 0;
}

// constructors
template <class T, size_t Limit>
inline SList<T,Limit>::SList
(
 Boolean circ
 )
{
  Count    = 0;
  Work     = 0;
  Circular = circ;
}

// copy constructor
template <class T, size_t Limit>
inline SList<T,Limit>::SList
(
 const SList<T,Limit> & slst
 )
{
  DeepCopy(slst);
}

// construct from array
template <class T, size_t Limit>
SList<T,Limit>::SList
(
 const T * array,
 size_t no
 )
{
  if ((array == NULL) || (no == 0))
    return;

  Count = 0;
  Work  = 0;
  Circular = FALSE;

  const T * aptr = array;

  for (size_t n = 0; n < no; ++n)
    {
      Append(*aptr);
      ++aptr;
    }
}

// destructor
template <class T, size_t Limit>
inline SList<T,Limit>::~SList()
{
  Erase();
}

// assignment operator (deep copy)
template <class T, size_t Limit>
inline void SList<T,Limit>::operator =
(
 const SList<T,Limit> & slst
 )
{
  Erase();
  DeepCopy(slst);
}

// conversion to array operator
template <class T, size_t Limit>
SList<T,Limit>::operator T * ()
{
  if (Count == 0)
    return NULL;

  T * result = new T [Count];
  assert(result);

  for (size_t n = 0; n < Count; ++n)
    result[n] = Data[n];

  // note: caller must delete this array!
  return result;
}

// append new item
template <class T, size_t Limit>
void SList<T,Limit>::Append
(
 const T & item
 )
{
  assert(Count != Limit);

  Data[Count] = item;

  ++Count;
}

// append an array of new items
template <class T, size_t Limit>
void SList<T,Limit>::Append
(
 const T * array,
 size_t no
 )
{
  if ((array == NULL) || (no == 0))
    return;

  const T * aptr = array;

  for (size_t n = 0; n < no; ++n)
    {
      Append(*aptr);
      ++aptr;
    }
}

template <class T, size_t Limit>
void SList<T,Limit>::Append
(
 const SList<T,Limit> & slst
 )
{
  for (size_t n = 0; n < slst.Count; ++n)
    Append(slst.Data[n]);
}

// get current item
template <class T, size_t Limit>
T SList<T,Limit>::Get()
{
  assert(Count && Work!=Count);

  return Data[Work];
}

// TRUE if current item is last item
template <class T, size_t Limit>
inline Boolean SList<T,Limit>::AtTail()
{
  if (Work == Count)
    return TRUE;
  else
    return FALSE;
}

// set current item to head of list
template <class T, size_t Limit>
inline void SList<T,Limit>::ToHead()
{
  Work = 0;
}

// move to next item in list
template <class T, size_t Limit>
Boolean SList<T,Limit>::ToNext()
{
  if (!Count || Work == Count) return FALSE;

  ++Work;

  if (Work == Count)
    {
      if (Circular)
	Work = 0;
      else
	return FALSE;
    }

  return TRUE;
}

// interrogate list
template <class T, size_t Limit>
inline size_t SList<T,Limit>::GetCount()
{
  return Count;
}

template <class T, size_t Limit>
inline size_t SList<T,Limit>::GetLimit()
{
  return Limit;
}

template <class T, size_t Limit>
inline Boolean SList<T,Limit>::IsCircular()
{
  return Circular;
}

#endif
