package csci4534.scheduler;

import java.util.Iterator;
import java.util.StringTokenizer;
import java.util.Vector;

/**
 * A command-line CPU scheduling simulator.
 *
 * @author Toli Lerios
 **/

public class MainSimulator
{

    // PUBLIC INTERFACE.

    /**
     * Main program.
     *
     * @param args The command-line arguments: execute without
     * arguments for usage instructions.
     *
     * @throws Exception Thrown iff an error occurs.
     **/

    public static void main(String[] args)
        throws Exception
    {

        // Error-checking.

        if (args.length<2) {
            System.err.println
                ("Usage: <scheduler> (<priority>:<arrival>:<burst>)+");
            return;
        }

        // Parse fixed command-line arguments.

        Scheduler scheduler=(Scheduler)Class.forName(args[0]).newInstance();
        System.err.println("Scheduler: "+scheduler.getClass().getName());

        // Load process information.

        int name=1;
        Vector processes=new Vector();
        for (int i=1;i<args.length;i++,name++) {
            StringTokenizer tokenizer=new StringTokenizer(args[i],":");
            int priority=Integer.parseInt(tokenizer.nextToken());
            int arrivalTime=Integer.parseInt(tokenizer.nextToken());
            int remainingTime=Integer.parseInt(tokenizer.nextToken());
            Process process=
                new Process("P"+name,priority,arrivalTime,remainingTime);
            processes.add(process);
            System.out.println("Loaded "+process);
        }

        // Run simulation.

        int time=0;                        // Time is in arbitrary ticks.
        int waitingTime=0;                 // Total time spent waiting.
        int processCount=processes.size(); // The total number of processes.
        Process runningProcess=null;       // The process currently on the CPU.
        while (true) {

            // If a process was running, kick it off the CPU.

            if (runningProcess!=null) {
                runningProcess.stopRunning(time);
                System.out.println("@"+time+" stopped "+runningProcess);

                // If the removed process has finished its execution,
                // get rid of it altogether after updating our
                // statistics.

                if (runningProcess.getRemainingTime()==0) {
                    processes.remove(runningProcess);
                    System.out.println("@"+time+" finished "+runningProcess);
                    waitingTime+=runningProcess.getWaitingTime();
                }

                // No process on the CPU.

                runningProcess=null;
            }

            // If no processes want the CPU, the simulation is over.

            if (processes.size()==0) {
                break;
            }

            // Find all the processes that want the CPU. Exclude
            // processes whose arrival time is in the future but keep
            // track of the earliest future process because...

            int minFutureArrivalTime=-1;
            Vector active=new Vector();
            for (Iterator p=processes.iterator();p.hasNext();) {
                Process process=(Process)p.next();
                if (process.getArrivalTime()<=time) {
                    active.add(process);
                } else if ((minFutureArrivalTime==-1) ||
                           (process.getArrivalTime()<minFutureArrivalTime)) {
                    minFutureArrivalTime=process.getArrivalTime();
                }
            }

            // ... if all processes are in the future, jump forward in
            // time and go back to search again for those interested
            // in the CPU.

            if (active.size()==0) {
                time=minFutureArrivalTime;
                continue;
            }

            // Invoke scheduler to choose a process to run on the CPU.

            Process[] processesA=new Process[active.size()];
            active.toArray(processesA);
            Decision decision=scheduler.choose(processesA);

            // Assign CPU to the process.

            runningProcess=decision.getProcess();
            runningProcess.startRunning(time);
            System.out.println("@"+time+" started "+runningProcess);

            // Jump forward to the point in time when the CPU should
            // become available again.

            time+=decision.getTimeout();
        }

        // Compute average waiting time.

        System.out.println
            ("Average waiting time: "+((float)waitingTime)/processCount);
    }
}
