package csci4534.controller;

/**
 * A memory controller. It may include both operations that usually
 * take place in hardware (such as splitting a logical address into
 * its parts), as well as operating system tasks (such as handling
 * segment de/allocation, and implementing virtual memory).
 *
 * <P>
 *
 * {@link
 * #initialize(ControllerConfiguration,Memory,Disk,DMAController)} is
 * called before any other method, and provides the controller with
 * its operating environment. If that method fails, no other method is
 * called. Otherwise, the controller is consider ready to service
 * processes.
 *
 * <P>
 *
 * In a real operating system, a process is created by the kernel, and
 * the memory controller subsystem keeps track of memory management
 * information in fields within the Process Control Block (PCB). To
 * simplify the simulation, instances of this controller just create a
 * {@link Process} instance every time {@link #newProcess()} is
 * called. This instance contains exactly the same controller
 * information that is normally placed into the PCB of a pre-existing
 * process.
 *
 * <P>
 *
 * Once a new process is thus created, it can allocate memory in zero
 * or more segments via multiple calls to {@link
 * #malloc(Process,boolean,boolean,int)}. If a call is successful, the
 * process receives a segment number, which it can use to read and
 * write bytes within this segment by
 *
 * <UL>
 * <LI> composing logical addresses that refer to that segment, and
 * <LI> invoking {@link #read(Process,int)} or {@link
 * #write(Process,int,byte)} with such logical addresses.
 * </UL>
 *
 * The process can use a segment until it decides to deallocate it via
 * {@link #free(Process,int)}. Note that there is no support for
 * shared memory. Also, the contents of a newly allocated segment are
 * not zeroed out.
 *
 * <P>
 *
 * Eventually, a process terminates, and the controller is notified of
 * its termination via {@link #processEnded(Process)}, at which point
 * it automatically releases all memory the process had allocated
 * (even if the process neglected to deallocate it via explicit calls
 * to {@link #free(Process,int)}.
 *
 * <P>
 *
 * Controllers throw instances of {@link ControllerException} when
 * they encounter problems. In all cases, the exception message
 * explains the nature of the failure.
 *
 * @author Toli Lerios
 **/

public interface Controller
{

    // PUBLIC INTERFACE.

    /**
     * Initializes the controller.
     *
     * @param configuration The controller's configuration.
     * @param memory The physical memory.
     * @param disk The disk, which may optionally be used for virtual
     * memory.
     * @param controller The DMA controller, which may optionally be
     * used for virtual memory.
     **/

    public void initialize(ControllerConfiguration configuration,
                           Memory memory,
                           Disk disk,
                           DMAController controller);

    /**
     * Creates a new process. All tables the process will use, as well
     * as their entries, are allocated before this method returns with
     * one exception: if the controller uses hierarchical page tables,
     * then no storage in allocated for the inner page tables. That
     * exception aside, unused entries are still allocated, but are
     * simply marked as invalid to mirror the typical approach taken
     * by the hardware.
     **/

    public Process newProcess();

    /**
     * Allocates a new segment for the given process, and returns its
     * number <EM>s</EM>. If the controller uses hierarchical page
     * tables, then this method also allocates all inner page tables
     * necessary to store all the frame numbers of the segment's
     * pages. For every inner page table, unused entries are still
     * allocated, but are simply marked as invalid to mirror the
     * typical approach taken by the hardware.
     *
     * @param process The process.
     * @param readable True iff data can be read from the segment.
     * @param writeable True iff data can be written into the segment.
     * @param length The segment's length, in bytes.
     *
     * @return The segment number <EM>s</EM>.
     *
     * @throws ControllerException Thrown iff there is a controller
     * failure, such as running out of memory.
     **/

    public int malloc(Process process,
                      boolean readable,
                      boolean writeable,
                      int length)
        throws ControllerException;

    /**
     * Returns the memory contents at the given logical address of the
     * given process.
     *
     * @param process The process.
     * @param logicalAddress The logical address.
     *
     * @return The contents.
     *
     * @throws ControllerException Thrown iff there is a controller
     * failure, such as an attempt to read from a write-only segment.
     **/

    public byte read(Process process,
                     int logicalAddress)
        throws ControllerException;

    /**
     * Sets the memory contents at the given logical address of the
     * given process to the given value.
     *
     * @param process The process.
     * @param logicalAddress The logical address.
     * @param value The value.
     *
     * @throws ControllerException Thrown iff there is a controller
     * failure, such as an attempt to write onto a read-only segment.
     **/

    public void write(Process process,
                      int logicalAddress,
                      byte value)
        throws ControllerException;

    /**
     * Deallocates the given segment of the given process. If the
     * controller uses hierarchical page tables, then this method also
     * deallocates all inner page tables the segment used.
     *
     * @param process The process.
     * @param s The segment number.
     *
     * @throws ControllerException Thrown iff there is a controller
     * failure, such as being given an invalid segment number.
     **/

    public void free(Process process,
                     int s)
        throws ControllerException;

    /**
     * Informs the controller that the given process has terminated.
     *
     * @param process The process.
     **/

    public void processEnded(Process process);
}
