package csci4534.controller;

import java.util.StringTokenizer;

/**
 * A command-line memory controller simulator.
 *
 * @author Toli Lerios
 **/

public class MainSimulator
{

    // PUBLIC INTERFACE.

    /**
     * Main program.
     *
     * @param args The command-line arguments: execute without
     * arguments for usage instructions.
     *
     * @throws Exception Thrown iff an error occurs.
     **/

    public static void main(String[] args)
        throws Exception
    {

        // Error-checking.

        if (args.length<4) {
            System.err.println
                ("Usage: <controller> <bit counts> <memory frame count> "+
                 "<disk block count>");
            System.err.println
                ("       (n|(x:<pid>)|(a:<pid>:<length>:r?w?)|"+
                 "(d:<pid>:<s>)|");
            System.err.println
                ("       (r:<pid>:<address>)|(w:<pid>:<address>:<value>))+");
            return;
        }

        // Parse fixed command-line arguments.

        Controller controller=(Controller)Class.forName(args[0]).newInstance();
        System.err.println("Controller: "+controller.getClass().getName());
        StringTokenizer tokenizer=new StringTokenizer(args[1],":");
        int sBits=Integer.parseInt(tokenizer.nextToken());
        int offsetBits=Integer.parseInt(tokenizer.nextToken());
        int p1Bits=Integer.parseInt(tokenizer.nextToken());
        int p2Bits=Integer.parseInt(tokenizer.nextToken());
        int dBits=Integer.parseInt(tokenizer.nextToken());
        int frameCount=Integer.parseInt(args[2]);
        int blockCount=Integer.parseInt(args[3]);

        // Initialize controller configuration, disks, memory,
        // controller, and process table.

        ControllerConfiguration configuration=new ControllerConfiguration
            (sBits,offsetBits,p1Bits,p2Bits,dBits);
        int frameSize=Utilities.pow2(dBits);
        Disk disk=new Disk(blockCount,frameSize);
        Memory memory=new Memory(frameCount*frameSize);
        controller.initialize(configuration,memory,disk,new DMAController());
        ProcessTable pt=new ProcessTable();

        // Run simulation.

        for (int i=4;i<args.length;i++) {
            tokenizer=new StringTokenizer(args[i],":");
            String operation=tokenizer.nextToken();
            try {
                if (operation.equals("n")) {
                    System.out.print("Creating process: ");
                    int pid=pt.put(controller.newProcess());
                    System.out.println(pid);
                } else if (operation.equals("a")) {
                    int pid=Integer.parseInt(tokenizer.nextToken());
                    int len=Integer.parseInt(tokenizer.nextToken(),16);
                    String access=tokenizer.nextToken();
                    boolean readable=(access.indexOf('r')!=-1);
                    boolean writeable=(access.indexOf('w')!=-1);
                    System.out.print("Allocating length "+
                                     Integer.toHexString(len)+" access ");
                    if (readable) {
                        System.out.print("r");
                    }
                    if (writeable) {
                        System.out.print("w");
                    }
                    System.out.print(" for process "+pid+": ");
                    int s=controller.malloc
                        (pt.get(pid),readable,writeable,len);
                    System.out.println(s);
                } else if (operation.equals("r")) {
                    int pid=Integer.parseInt(tokenizer.nextToken());
                    int logicalAddress=
                        Integer.parseInt(tokenizer.nextToken(),16);
                    System.out.print("r@"+pid+":"+
                                     Integer.toHexString(logicalAddress)+"=");
                    byte value=controller.read(pt.get(pid),logicalAddress);
                    System.out.println(value);
                    pt.checkRead(pid,logicalAddress,value);
                } else if (operation.equals("w")) {
                    int pid=Integer.parseInt(tokenizer.nextToken());
                    int logicalAddress=
                        Integer.parseInt(tokenizer.nextToken(),16);
                    byte value=Byte.parseByte(tokenizer.nextToken());
                    System.out.print("w@"+pid+":"+
                                     Integer.toHexString(logicalAddress)+"=");
                    controller.write(pt.get(pid),logicalAddress,value);
                    System.out.println(value);
                    pt.applyWrite(pid,logicalAddress,value);
                } else if (operation.equals("d")) {
                    int pid=Integer.parseInt(tokenizer.nextToken());
                    int s=Integer.parseInt(tokenizer.nextToken());
                    System.out.print
                        ("Deallocating "+s+" for process "+pid+": ");
                    controller.free(pt.get(pid),s);
                    System.out.println("done");
                } else if (operation.equals("x")) {
                    int pid=Integer.parseInt(tokenizer.nextToken());
                    System.out.print("Terminating process "+pid+": ");
                    controller.processEnded(pt.get(pid));
                    System.out.println("done");
                    pt.remove(pid);
                } else {
                    System.err.println("Skipping invalid operation: "+args[i]);
                }
            } catch (ControllerException ex) {
                System.out.println(ex);
            }
        }

        // Display disk and memory state.

        System.out.println("Memory: "+memory);
        System.out.println("Disk: "+disk);
    }
}
