package csci4534.controller;

// ADD CODE HERE IF APPROPRIATE.

import java.util.LinkedList;

/**
 * A memory controller with segmentation and paging but no virtual
 * memory; all pages are stored in physical memory only. Available
 * (free) memory frames are maintained on a linked list. The page
 * tables are stored in a two-level hierarchical fashion, with a
 * top-level page directory containing (pointers to) one or more inner
 * page tables, which in turn contain the frame numbers of pages.
 *
 * <P>
 *
 * Memory allocation and decallocation is fairly simplistic. We always
 * use the first available segment descriptor to allocate a new
 * segment. But the page directory entry we choose is the one
 * following the last valid entry (if there's none, allocation fails);
 * if we chose instead the first invalid entry, then the segment size
 * would be limited by the size of the gap between that entry and the
 * next valid one. Such holes in the page directory may occur if, for
 * example, we allocate three segments in order and then deallocate
 * the second one. In effect, the page directory becomes
 * fragmented. This simple controller does not attempt to reuse these
 * holes (unless it so happens that they eventually merge together
 * with the last hole). Of course, this undesireable fragmentation
 * only affects the page directory and has no impact on physical
 * memory.
 *
 * <P>
 *
 * A subtlety of the above algorithm is that a new segment always uses
 * a brand new inner page table, rather than take up the unused
 * entries in another inner page table. This theoretically enables a
 * process to extend those segments, though such functionality is not
 * implemented here.
 *
 * <P>
 *
 * A benefit of the above de/allocation algorithm is that all the
 * valid entries in every page table start at the table's beginning
 * and they are contiguous. This means that whenever we want to
 * traverse all valid entries of a page table, we can stop as soon as
 * we encounter an invalid one. However, the page directory table does
 * not have this property.
 *
 * <P>
 *
 * This controller does address translation via instances of {@link
 * AddressTranslation}.
 *
 * @author Toli Lerios
 **/

public class PhysicalController
    implements Controller
{

    // INSTANCE DATA.

    private ControllerConfiguration mConfiguration;
    private Memory mMemory;
    private LinkedList mFreeFrames=new LinkedList();

    // ADD CODE HERE IF APPROPRIATE.


    // PRIVATE HELPERS.

    /**
     * Returns the number of a free frame. It assumes one is
     * available. The returned frame is removed from the free frame
     * list.
     *
     * @return The frame number.
     **/

    private int getFreeFrameUnsafe()
    {
        return ((Integer)(mFreeFrames.removeFirst())).intValue();
    }

    /**
     * Adds a frame to the free frame list.
     *
     * @param frame The frame number.
     **/

    private void addFreeFrame(int frame)
    {
        mFreeFrames.addFirst(new Integer(frame));
    }

    // ADD CODE HERE IF APPROPRIATE.


    // SUBCLASS INTERFACE.

    /**
     * Returns the controller's configuration.
     *
     * @return The configuration.
     **/

    protected ControllerConfiguration getConfiguration()
    {
        return mConfiguration;
    }

    /**
     * Returns the physical memory.
     *
     * @return The memory.
     **/

    protected Memory getMemory()
    {
        return mMemory;
    }

    /**
     * Returns a free frame for use by the given process.
     *
     * @throws ControllerException Thrown iff no free frames are
     * available.
     **/

    protected int getFreeFrame()
        throws ControllerException
    {
        if (mFreeFrames.size()==0) {
            throw new ControllerException("No free frames available");
        }
        return getFreeFrameUnsafe();
    }

    /**
     * Ensures that there is adequate memory space to store the given
     * number of pages. This implementation checks physical memory
     * availability only. Subclasses may override this method to
     * check disk availability instead.
     *
     * @param pageCount The number of pages.
     *
     * @throws ControllerException Thrown iff there isn't adequate
     * space.
     **/

    protected void ensureAdequateSpace(int pageCount)
        throws ControllerException
    {
        if (mFreeFrames.size()<pageCount) {
            throw new ControllerException("Out of physical memory");
        }
    }

    /**
     * Allocates a single page table entry. This implementation also
     * allocates a memory frame to the new page. Subclasses may
     * override this method to allocate disk blocks for pages,
     * instead. This method is called <EM>n</EM> times in a row only
     * if a prior call to {@link #ensureAdequateSpace(int)} succeeded
     * with <EM>n</EM> as argument; hence implementations can safely
     * assume space is available.
     *
     * @return The entry.
     **/

    protected PageTableEntry mallocEntry()
    {
        return new PageTableEntry
            (true,false,false,true,getFreeFrameUnsafe(),false,0);
    }

    /**
     * Frees all resources associated with the given page table
     * entry. This implementation frees only the memory frame
     * containing the page, if any. Subclasses may override this
     * method to deallocate disk blocks, but should also invoke this
     * implementation.
     *
     * @param pte The entry.
     **/

    protected void free(PageTableEntry pte)
    {
        if (pte.isInMemory()) {
            addFreeFrame(pte.getFrame());
        }
    }

    /**
     * Notifies the subclass that a memory read operation is about to
     * be performed on behalf of the given process at the given
     * address. In part, the subclass is expected to ensure that the
     * page is in memory. The subclass may assume that the access is a
     * valid one (e.g. the segment has read access).
     *
     * @param process The process.
     * @param address The translated address.
     *
     * @throws ControllerException Thrown iff there is a controller
     * failure in which case there is no followup attempt to access
     * physical memory, and the overall operation fails.
     **/

    protected void read(Process process,
                        AddressTranslation address)
        throws ControllerException {}

    /**
     * Notifies the subclass that a memory write operation is about to
     * be performed on behalf of the given process at the given
     * address. In part, the subclass is expected to ensure that the
     * page is in memory. The subclass may assume that the access is a
     * valid one (e.g. the segment has write access).
     *
     * @param process The process.
     * @param address The translated address.
     *
     * @throws ControllerException Thrown iff there is a controller
     * failure, in which case there is no followup attempt to access
     * physical memory, and the overall operation fails.
     **/

    protected void write(Process process,
                         AddressTranslation address)
        throws ControllerException {}

    // ADD CODE HERE IF APPROPRIATE.


    // Controller.

    public void initialize(ControllerConfiguration configuration,
                           Memory memory,
                           Disk disk,
                           DMAController controller)
    {
        mConfiguration=configuration;
        mMemory=memory;
        // Initially, all of the physical memory contains free frames.
        for (int i=getConfiguration().getFrameCount
                 (getMemory().getCapacity())-1;i>=0;i--) {
            addFreeFrame(i);
        }
    }

    public Process newProcess()
    {
        // Create segment descriptor table: all entries are invalid.
        DescriptorTable dt=new DescriptorTable
            (getConfiguration().getDescriptorTableCapacity());
        for (int i=0;i<dt.getCapacity();i++) {
            dt.setEntry(i,new SegmentDescriptor(false,false,false,0,0));
        }
        // Create page directory table: all entries are invalid. Page
        // tables are created during segment allocation.
        PageDirectory pd=new PageDirectory
            (getConfiguration().getPageDirectoryCapacity());
        for (int i=0;i<pd.getCapacity();i++) {
            pd.setEntry(i,new PageDirectoryEntry(false,null));
        }
        // Create process: while the circular queue pointer is not
        // used by this implementation, it is set to the first entry
        // of the first page table of the first segment.
        return new Process(dt,pd,0,0);
    }

    public int malloc(Process process,
                      boolean readable,
                      boolean writeable,
                      int length)
        throws ControllerException
    {
        // Error checking.
        if (length>getConfiguration().getMaxSegmentLength()) {
            throw new ControllerException("Segment too long");
        }
        // Translate the segment length into a page count.
        int pageCount=getConfiguration().getFrameCount(length);
        // Error checking.
        ensureAdequateSpace(pageCount);
        // Find the first available segment descriptor, if any.
        DescriptorTable dt=process.getDescriptorTable();
        SegmentDescriptor sd=null;
        int sdIndex=0;
        for (;sdIndex<dt.getCapacity();sdIndex++) {
            sd=dt.getEntry(sdIndex);
            if (!sd.isValid()) {
                break;
            }
        }
        if (sdIndex==dt.getCapacity()) {
            throw new ControllerException("No segment available");
        }
        // Find the first available page directory entry that has no
        // valid entries after it: start from the end of the table and
        // go backwards until we find the first valid entry.
        PageDirectory pd=process.getPageDirectory();
        int pdeIndex=pd.getCapacity()-1;
        for (;pdeIndex>=0;pdeIndex--){
            PageDirectoryEntry pde=pd.getEntry(pdeIndex);
            if (pde.isValid()) {
                break;
            }
        }
        // The one after it (if any) must have been invalid.
        pdeIndex++;
        // Ensure we have enough space left in the page directory (and
        // the page tables we'll create) to accomodate all segment
        // pages.
        if ((pd.getCapacity()-pdeIndex)*
            getConfiguration().getPageTableCapacity()<pageCount) {
            throw new ControllerException("No page available");
        }
        // Update segment descriptor.
        sd.setValid(true);
        sd.setReadable(readable);
        sd.setWriteable(writeable);
        sd.setLimit(length);
        // Translate the index of the first available page directory
        // entry into the segment descriptor's base address.
        sd.setBase(getConfiguration().getLinearAddress(pdeIndex,0,0));
        // Allocate page tables and create entries, as long as there
        // are still pages left to allocate.
        while (pageCount>0) {
            PageDirectoryEntry pde=pd.getEntry(pdeIndex);
            pde.setValid(true);
            // Allocate page table.
            PageTable pt=new PageTable
                (getConfiguration().getPageTableCapacity());
            pde.setTable(pt);
            // Validate all page table entries while there are still
            // pages left to allocate.
            for (int j=0;j<pt.getCapacity();j++) {
                PageTableEntry pte;
                if (pageCount>0) {
                    pte=mallocEntry();
                    pageCount--;
                } else {
                    pte=new PageTableEntry(false,false,false,false,0,false,0);
                }
                pt.setEntry(j,pte);
            }
            pdeIndex++;
        }
        return sdIndex;
    }

    public byte read(Process process,
                     int logicalAddress)
        throws ControllerException
    {

        // ADD CODE HERE IF APPROPRIATE.

        // REMOVE THE LINE BELOW.
        return 0;
    }

    public void write(Process process,
                      int logicalAddress,
                      byte value)
        throws ControllerException
    {

        // ADD CODE HERE IF APPROPRIATE.

    }

    public void free(Process process,
                     int s)
        throws ControllerException
    {
        // Error checking.
        SegmentDescriptor sd=process.getDescriptorTable().getEntry(s);
        if (!sd.isValid()) {
            throw new ControllerException("Segment not allocated");
        }
        // Use the segment descriptor base address without an offset
        // to find the first page directory entry of the segment.
        int pdeIndex=getConfiguration().getP1(sd.getBase());
        // Translate the segment length into a page count.
        int pageCount=getConfiguration().getFrameCount(sd.getLimit());
        // Iterate over page directory entries used by the segment, as
        // long as there are still pages left to free.
        PageDirectory pd=process.getPageDirectory();
        while (pageCount>0) {
            PageDirectoryEntry pde=pd.getEntry(pdeIndex);
            PageTable pt=pde.getTable();
            // Iterate over all page table entries while there are
            // still pages left to free.
            for (int j=0;((j<pt.getCapacity()) && (pageCount>0));j++) {
                free(pt.getEntry(j));
                pageCount--;
            }
            pde.setValid(false);
            pde.setTable(null);
            pdeIndex++;
        }
        // Update segment descriptor.
        sd.setValid(false);
    }

    public void processEnded(Process process)
    {
        // Iterate over all process pages and free them.
        PageDirectory pd=process.getPageDirectory();
        // For each valid page directory entry.
        for (int i=0;i<pd.getCapacity();i++) {
            PageDirectoryEntry pde=pd.getEntry(i);
            if (!pde.isValid()) {
                continue;
            }
            PageTable pt=pde.getTable();
            // For each valid page table entry.
            for (int j=0;j<pt.getCapacity();j++) {
                PageTableEntry pte=pt.getEntry(j);
                // Bail out at first invalid entry.
                if (!pte.isValid()) {
                    break;
                }
                free(pte);                
            }
        }
    }
}
