package csci4534.controller;

import java.util.Hashtable;

/**
 * A table in which a controller simulator keeps track of simulated
 * processes.
 *
 * @author Toli Lerios
 **/

public class ProcessTable
{

    // INSTANCE DATA.

    private Hashtable mProcesses=new Hashtable();
    private int mNextProcess=0;


    // PRIVATE HELPERS.

    /**
     * The information record the simulator maintains for each
     * process.
     **/

    private static class ProcessInfo
    {

        // INSTANCE DATA.

        private Process mProcess;
        private Hashtable mVerifier=new Hashtable();


        // CONSTRUCTORS.

        /**
         * Creates a new process information record for the given
         * process.
         *
         * @param process The process.
         **/

        ProcessInfo(Process process)
        {
            mProcess=process;
        }


        // PUBLIC INTERFACE.

        /**
         * Returns the process in the record.
         *
         * @return The process.
         **/

        Process getProcess()
        {
            return mProcess;
        }

        /**
         * Checks the correctness of a read operation on behalf of the
         * record's process at the given logical address. It prints a
         * message onto standard output if the given actual data read
         * doesn't match the expected data.
         *
         * @param logicalAddress The logical address.
         * @param actual The actual data read.
         **/

        void checkRead(int logicalAddress,
                       byte actual)
        {
            Byte expectedB=(Byte)(mVerifier.get(new Integer(logicalAddress)));
            if (expectedB==null) {
                return;
            }
            byte expected=expectedB.byteValue();
            if (expected!=actual) {
                System.out.println
                    ("Read mismatch: expected "+expected+" but got "+actual);
            }
        }

        /**
         * Updates the record to reflect a write operation on behalf
         * of the record's process at the given logical address.
         *
         * @param logicalAddress The logical address.
         * @param actual The actual data written.
         **/

        void applyWrite(int logicalAddress,
                        byte actual)
        {
            mVerifier.put(new Integer(logicalAddress),new Byte(actual));
        }
    }

    /**
     * Returns the information record for the given process.
     *
     * @param pid The process ID.
     *
     * @return The process information.
     **/

    private ProcessInfo getProcessInfo(int pid)
    {
        return (ProcessInfo)(mProcesses.get(new Integer(pid)));
    }


    // PUBLIC INTERFACE.

    /**
     * Creates a new record of simulator information for the given
     * process, and assigns it a process ID.
     *
     * @param process The process.
     *
     * @return The ID assigned to the process.
     **/

    public int put(Process process)
    {
        mProcesses.put(new Integer(mNextProcess),new ProcessInfo(process));
        return mNextProcess++;
    }

    /**
     * Returns the process with the given ID.
     *
     * @param pid The process ID.
     *
     * @return The process.
     **/

    public Process get(int pid)
    {
        return getProcessInfo(pid).getProcess();
    }

    /**
     * Eliminates all simulator information for the process with the
     * given ID.
     *
     * @param pid The process ID.
     **/

    public void remove(int pid)
    {
        mProcesses.remove(new Integer(pid));
    }

    /**
     * Checks the correctness of a read operation on behalf of the
     * process with the given ID at the given logical address. It
     * prints a message onto standard output if the given actual data
     * read doesn't match the expected data.
     *
     * @param pid The process ID.
     * @param logicalAddress The logical address.
     * @param actual The actual data read.
     **/

    public void checkRead(int pid,
                          int logicalAddress,
                          byte actual)
    {
        getProcessInfo(pid).checkRead(logicalAddress,actual);
    }

    /**
     * Updates the information record of the process with the given ID
     * to reflect a write operation on behalf of that process at the
     * given logical address.
     *
     * @param pid The process ID.
     * @param logicalAddress The logical address.
     * @param actual The actual data written.
     **/

    public void applyWrite(int pid,
                           int logicalAddress,
                           byte actual)
    {
        getProcessInfo(pid).applyWrite(logicalAddress,actual);
    }
}
