package csci4534.controller;

import java.util.LinkedList;
import java.util.ListIterator;

/**
 * A simple memory controller. Neither virtual memory nor paging are
 * used. A process may allocate zero or one segment but no more. The
 * process segment is stored in a contiguous section of
 * memory. Available memory sections (holes) are maintained by the
 * controller, and assigned on a first-fit basis. The list of holes is
 * always kept sorted on the hole's beginning address to simplify
 * adjacent hole concatenation; but compaction is not supported. Given
 * this simple model, the mapping of logical to physical addresses is
 * a simple matter of a limit register (segment limit) and a
 * relocation register (segment base) computation; in effect, in terms
 * of the Intel 80386 architecture under OS/2, we are
 *
 * <UL>
 *
 * <LI> always using segment number 0,
 * <LI> treat the logical address as a segment offset,
 * <LI> and use the linear address as a physical address.
 *
 * </UL>
 *
 * @author Toli Lerios
 **/

public class SimpleController
    implements Controller
{

    // INSTANCE DATA.

    private Memory mMemory;
    private LinkedList mHoles=new LinkedList();


    // PRIVATE HELPERS.

    /**
     * A memory hole, containing unallocated memory. The beginning
     * address is, of course, a physical address.
     **/

    private static class Hole
    {

        // INSTANCE DATA.

        private int mBase;
        private int mLength;


        // CONSTRUCTORS.

        /**
         * Creates a new hole.
         *
         * @param base The hole's beginning address.
         * @param length The hole's length.
         **/

        Hole(int base,
             int length)
        {
            mBase=base;
            mLength=length;
        }


        // PUBLIC INTERFACE.

        /**
         * Returns the hole's beginning address.
         *
         * @return The address.
         **/

        int getBase()
        {
            return mBase;
        }

        /**
         * Sets the hole's beginning address to the given address.
         *
         * @param base The address.
         **/

        void setBase(int base)
        {
            mBase=base;
        }

        /**
         * Returns the hole's length.
         *
         * @return The length.
         **/

        int getLength()
        {
            return mLength;
        }

        /**
         * Sets the hole's length to the given length.
         *
         * @param length The length.
         **/

        void setLength(int length)
        {
            mLength=length;
        }
    }


    // Controller.

    public void initialize(ControllerConfiguration configuration,
                           Memory memory,
                           Disk disk,
                           DMAController controller)
    {
        mMemory=memory;
        // Initially, all of the physical memory is one large hole.
        mHoles.add(new Hole(0,mMemory.getCapacity()));
    }

    public Process newProcess()
    {
        // The segment descriptor table has a single entry, for the
        // single segment a process may allocate.
        DescriptorTable dt=new DescriptorTable(1);
        // The entry is initially invalid because the segment is not
        // automatically allocated.
        dt.setEntry(0,new SegmentDescriptor(false,false,false,0,0));
        // No page directory is used.
        return new Process(dt,null,0,0);
    }

    public int malloc(Process process,
                      boolean readable,
                      boolean writeable,
                      int length)
        throws ControllerException
    {
        // Error checking.
        SegmentDescriptor sd=process.getDescriptorTable().getEntry(0);
        if (sd.isValid()) {
            throw new ControllerException("Process can only own segment 0");
        }
        // Find a hole large enough for the requested segment length
        // (first-fit).
        Hole result=null;
        ListIterator resultPosition;
        for (resultPosition=mHoles.listIterator();resultPosition.hasNext();) {
            Hole hole=(Hole)resultPosition.next();
            if (hole.getLength()>=length) {
                result=hole;
                break;
            }
        }
        if (result==null) {
            throw new ControllerException("No hole available");
        }
        // Update segment descriptor.
        sd.setValid(true);
        sd.setReadable(readable);
        sd.setWriteable(writeable);
        sd.setLimit(length);
        sd.setBase(result.getBase());
        // Update hole list.
        if (result.getLength()>length) {
            // If the hole was bigger than the segment, keep it but
            // shrink it.
            result.setBase(result.getBase()+length);
            result.setLength(result.getLength()-length);
        } else {
            // If the hole was exactly equal to the segment, remove
            // it.
            resultPosition.remove();
        }
        // Segment number 0 is the only one that is ever allocated.
        return 0;
    }

    public byte read(Process process,
                     int logicalAddress)
        throws ControllerException
    {
        // Error checking.
        SegmentDescriptor sd=process.getDescriptorTable().getEntry(0);
        if (!sd.isValid()) {
            throw new ControllerException("Invalid segment descriptor");
        }
        if (!sd.isReadable()) {
            throw new ControllerException("Read disallowed");
        }
        if (logicalAddress>=sd.getLimit()) {
            throw new ControllerException("Address exceeds segment limit");
        }
        // Read memory.
        return mMemory.read(logicalAddress+sd.getBase());
    }

    public void write(Process process,
                      int logicalAddress,
                      byte value)
        throws ControllerException
    {
        // Error checking.
        SegmentDescriptor sd=process.getDescriptorTable().getEntry(0);
        if (!sd.isValid()) {
            throw new ControllerException("Invalid segment descriptor");
        }
        if (!sd.isWriteable()) {
            throw new ControllerException("Write disallowed");
        }
        if (logicalAddress>=sd.getLimit()) {
            throw new ControllerException("Address exceeds segment limit");
        }
        // Write memory.
        mMemory.write(logicalAddress+sd.getBase(),value);
    }

    public void free(Process process,
                     int s)
        throws ControllerException
    {
        // Error checking.
        if (s!=0) {
            throw new ControllerException("Process can only own segment 0");
        }
        SegmentDescriptor sd=process.getDescriptorTable().getEntry(0);
        if (!sd.isValid()) {
            throw new ControllerException("Segment not allocated");
        }
        // Update segment descriptor.
        sd.setValid(false);
        // Create a new hole to represent the space just deallocated.
        Hole newHole=new Hole(sd.getBase(),sd.getLimit());
        // Add the hole to the sorted list, and keep track of the two
        // holes (if any) before and after it.
        Hole before=null;
        Hole after=null;
        for (ListIterator i=mHoles.listIterator();i.hasNext();) {
            Hole next=(Hole)i.next();
            if (next.getBase()>newHole.getBase()) {
                i.previous();
                i.add(newHole);
                after=next;
                break;
            }
            before=next;
        }
        if (after==null) {
            mHoles.addLast(newHole);
        }            
        // If possible, merge the new hole and the one before it.
        if ((before!=null) &&
            (before.getBase()+before.getLength()==newHole.getBase())) {
            before.setLength(before.getLength()+newHole.getLength());
            // If we do merge, the new hole gets replaced by its
            // predecessor,
            mHoles.remove(newHole);
            newHole=before;
        }
        // If possible, merge the new hole (or its predecessor if we
        // already merged the new hole with its predecessor) and the
        // one after it.
        if ((after!=null) &&
            (newHole.getBase()+newHole.getLength()==after.getBase())) {
            newHole.setLength(newHole.getLength()+after.getLength());
            mHoles.remove(after);
        }
    }

    public void processEnded(Process process)
    {
        // If the process never allocated its segment, there is
        // nothing to do.
        SegmentDescriptor sd=process.getDescriptorTable().getEntry(0);
        if (!sd.isValid()) {
            return;
        }
        // Otherwise, deallocate the process segment.
        try {
            free(process,0);
        } catch (ControllerException ex) {
            // Cannot occur since we already checked that the segment
            // was indeed allocated.
        }
    }
}
