package csci4534.allocator;

/**
 * A resource allocator. It simulates the portion of a real operating
 * system that manages resources on behalf of processes. Accordingly,
 * implementations must be extremely robust, and respond gracefully to
 * invalid input. The only simplifying assumptions an allocator can
 * make is that it will never run out of memory, its methods will
 * always be called by a single Java thread, and that no input will
 * ever cause an arithmetic overflow. However, inputs can be
 * <CODE>null</CODE> or violate other semantic constraints.
 * <EM>Some</EM> of those constraints are outlined below, but this
 * listing is not exhaustive.
 *
 * <P>
 *
 * The allocator is first used to create one or more processes. In a
 * real operating system, a process is created by the kernel, and the
 * resource allocator subsystem keeps track of resource allocation
 * information inside internal parallel data structures. To simplify
 * the simulation, instances of this allocator just create a {@link
 * Process} instance every time {@link #createProcess()} is
 * called. Additional processes can be created at any point during the
 * allocator's lifetime.
 *
 * <P>
 *
 * Similarly to processes, a resource is created by invoking {@link
 * #createResource(int)} and specifying the number of total resource
 * instances. This total number never changes over the lifetime of a
 * resource, though some instances become allocated and thus
 * unavailable to other processes. Resources can be created at any
 * point during the allocator's lifetime. Each resource (and all its
 * resource instances combined) is represented by a single object
 * implementing {@link Resource}.
 *
 * <P>
 *
 * Processes make allocation requests via {@link
 * #allocate(Request)}. The request may be granted, enqueued, or it
 * might fail. The return value distinguishes the first two cases,
 * while an exception identifies the latter. The allocation algorithm
 * must meet the following requirements (in addition to general
 * robustness):
 *
 * <UL>
 *
 * <LI> The allocator maintains an internal queue of pending
 * requests. Requests that cannot be immediately granted are placed in
 * this queue, whose status is always available via {@link
 * #getQueue()}. A request may fail if placing the request in the
 * queue would cause a deadlock.
 *
 * <P>
 *
 * <LI> A granted request is <EM>wholly</EM> granted. That is, if a
 * request asks for instances of two different resources, but only one
 * resource has the requisite number of instances available, then the
 * allocator can never grant this request in a partial manner.
 *
 * <P>
 *
 * <LI> Typically, a request for resource instances that are all
 * available is immediately granted. However, for the allocator to
 * guarantee fairness, sometimes such requests are enqueued
 * instead. Fairness is the opposite of starvation: a starving process
 * is one that has a pending request which is never granted even
 * though the resource instances it needs are regularly released by
 * other processes.
 *
 * </UL>
 *
 * Resources are released by a process willingly via {@link
 * #release(Request)}. A request to release is similar to an
 * allocation request, except that, naturally, a process should
 * already be allocated the resource instances it's trying to
 * release. If the resources are released successfully, the allocator
 * may dequeue and grant pending requests from the queue, depending on
 * its fairness policy. The (possibly empty) collection of granted
 * requests is returned to the caller.
 *
 * <P>
 *
 * Processes are deleted/terminated via {@link
 * #deleteProcess(Process)}; this is best understood as a command
 * issued by another process to have the former one terminated; it is
 * not a self-termination request like <CODE>exit()</CODE>. As a
 * result, it can be issued even when a process has a pending
 * request. By constrast, processes with pending requests cannot
 * allocate or release resources because such processes are not in a
 * running state.
 *
 * <P>
 *
 * Resources are deleted via {@link #deleteResource(Resource)}.
 *
 * @author Toli Lerios
 **/

public interface Allocator
{

    // PUBLIC INTERFACE.

    /**
     * Creates an allocator process.
     *
     * @return The process.
     **/

    public Process createProcess();

    /**
     * Creates an allocator resource.
     *
     * @param instances The total number of resource instances.
     *
     * @return The resource.
     *
     * @throws AllocatorException Thrown iff the argument is invalid.
     **/

    public Resource createResource(int instances)
        throws AllocatorException;

    /**
     * Handles a request to allocate one or more instances of one or
     * more resources to a process.
     *
     * @param request The request.
     *
     * @return True iff the request is granted; false iff it is
     * enqueued.
     *
     * @throws AllocatorException Thrown iff the argument is invalid,
     * or deadlock would occur if the request was enqueued.
     **/

    public boolean allocate(Request request)
        throws AllocatorException;

    /**
     * Handles a request to release one or more instances of one or
     * more resources that a process has in its
     * allocation/possession.
     *
     * @param request The request.
     *
     * @return The collection of pending requests that were granted
     * due to the (temporary) availability of the released resource
     * instances. If this collection is empty, the result is an empty
     * collection, not <CODE>null</CODE>.
     *
     * @throws AllocatorException Thrown iff the argument is invalid.
     **/

    public Request[] release(Request request)
        throws AllocatorException;

    /**
     * Deletes the given resource. A resource that is either in active
     * use, i.e. allocated to a process, or pending use, i.e. the
     * allocator has a pending request for that resource, cannot be
     * released.
     *
     * @param resource The resource.
     *
     * @throws AllocatorException Thrown iff the argument is invalid.
     **/

    public void deleteResource(Resource resource)
        throws AllocatorException;

    /**
     * Deletes/terminates the given process. Pending requests of
     * terminating processes are automatically discarded from the
     * queue. Resource instances allocated to the process are also
     * implicitly released.
     *
     * @param process The process.
     *
     * @return The collection of pending requests that were granted
     * due to the (temporary) availability of the released resource
     * instances. If this collection is empty, the result is an empty
     * collection, not <CODE>null</CODE>.
     *
     * @throws AllocatorException Thrown iff the argument is invalid.
     **/

    public Request[] deleteProcess(Process process)
        throws AllocatorException;

    /**
     * Returns the current queue of pending requests. The requests are
     * listed in the returned collection in no particular order.
     *
     * @return The requests. If the queue is empty, the result is an
     * empty collection, not <CODE>null</CODE>.
     **/

    public Request[] getQueue();
}
