package csci4534.allocator;

import java.util.Enumeration;
import java.util.Hashtable;

/**
 * A generic table associating integer IDs with arbitrary objects. The
 * IDs can be static or dynamic:
 *
 * <UL>
 *
 * <LI> Static IDs are assigned once to each object, and never change
 * again during the table's lifetime.
 *
 * <LI> Dynamic IDs change whenever elements are removed from the
 * table so that all remaining objects always have sequential IDs in
 * the range [0,{@link #getSize()}-1].
 *
 * </UL>
 *
 * @author Toli Lerios
 **/

abstract public class IdTable
{

    // INSTANCE DATA.

    private boolean mDynamic;
    private Hashtable mForwardMap=new Hashtable(); // ID to object.
    private Hashtable mReverseMap=new Hashtable(); // Object to ID.
    private int mNextID; // The ID of the next object to be added.


    // PRIVATE HELPERS.

    /**
     * Reassigns dynamic IDs to all objects in the table, if
     * appropriate.
     **/

    private void reassignIDs()
    {
        // Static IDs are never changed.
        if (!mDynamic) {
            return;
        }
        // Reset the ID for the next object to be added.
        mNextID=0;
        // If there are no objects in the table, no reassignment is
        // necessary and we are done.
        if (getSize()==0) {
            return;
        }
        // Otherwise, rebuild both the forward and reverse maps, while
        // also updating the ID for the next object to be added.
        Hashtable forwardMap=new Hashtable(mForwardMap.size());
        Hashtable reverseMap=new Hashtable(mReverseMap.size());
        for (Enumeration e=mReverseMap.keys();e.hasMoreElements();) {
            Object o=e.nextElement();
            Integer nextID=new Integer(mNextID++);
            forwardMap.put(nextID,o);
            reverseMap.put(o,nextID);
        }
        mForwardMap=forwardMap;
        mReverseMap=reverseMap;
    }


    // SUBCLASS INTERFACE.

    /**
     * Adds the given object to the table, and assigns it an object
     * ID. Subclasses are expected to use this method to implement
     * another one with identical functionality but with strong
     * typing.
     *
     * @param o The object.
     *
     * @return The ID assigned to the object.
     **/

    protected int putUntyped(Object o)
    {
        Integer nextID=new Integer(mNextID);
        mForwardMap.put(nextID,o);
        mReverseMap.put(o,nextID);
        return mNextID++;
    }

    /**
     * Returns the object with the given ID. Subclasses are expected
     * to use this method to implement another one with identical
     * functionality but with strong typing.
     *
     * @param id The object ID.
     *
     * @return The object.
     **/

    protected Object getUntyped(int id)
    {
        return mForwardMap.get(new Integer(id));
    }

    /**
     * Returns the ID of the given object. Subclasses are expected to
     * use this method to implement another one with identical
     * functionality but with strong typing.
     *
     * @param o The object.
     *
     * @return The object ID.
     **/

    protected int getUntyped(Object o)
    {
        return ((Integer)(mReverseMap.get(o))).intValue();
    }

    /**
     * Checks whether the table contains the given object. Subclasses
     * are expected to use this method to implement another one with
     * identical functionality but with strong typing.
     *
     * @param o The object.
     *
     * @return True iff the table contains the given object.
     **/

    protected boolean containsUntyped(Object o)
    {
        return mReverseMap.containsKey(o);
    }

    /**
     * Removes the given object from the table. Subclasses are
     * expected to use this method to implement another one with
     * identical functionality but with strong typing.
     *
     * @param o The object.
     **/

    protected void removeUntyped(Object o)
    {
        Integer id=(Integer)mReverseMap.remove(o);
        // If it wasn't in the table anyway, we're done.
        if (id==null) {
            return;
        }
        // Otherwise, update the forward map as well.
        mForwardMap.remove(id);
        // Possibly reassign dynamic IDs to fill-in the gap just
        // created.
        reassignIDs();
    }


    // CONSTRUCTORS.

    /**
     * Creates a new table.
     *
     * @param dynamic True iff object IDs should be dynamic.
     **/

    public IdTable(boolean dynamic)
    {
        mDynamic=dynamic;
    }


    // PUBLIC INTERFACE.

    /**
     * Removes the object with the given ID from the table.
     *
     * @param id The object ID.
     **/

    public void remove(int id)
    {
        Object o=mForwardMap.remove(new Integer(id));
        // If it wasn't in the table anyway, we're done.
        if (o==null) {
            return;
        }
        // Otherwise, update the reverse map as well.
        mReverseMap.remove(o);
        // Possibly reassign dynamic IDs to fill-in the gap just
        // created.
        reassignIDs();
    }

    /**
     * Returns the number of objects in the table.
     *
     * @return The number of objects.
     **/

    public int getSize()
    {
        return mForwardMap.size();
    }

    /**
     * Returns all IDs assigned to objects in the table.
     *
     * @return The IDs.
     **/

    public int[] getAllIDs()
    {
        int[] result=new int[getSize()];
        int i=0;
        for (Enumeration e=mForwardMap.keys();e.hasMoreElements();) {
            result[i++]=((Integer)(e.nextElement())).intValue();
        }
        return result;
    }
}
