package csci4534.allocator;

import java.util.Arrays;
import java.util.LinkedList;
import java.util.StringTokenizer;

/**
 * A command-line resource allocator.
 *
 * @author Toli Lerios
 **/

public class MainSimulator
{

    // CLASS DATA.

    private static ResourceTable sRT=new ResourceTable(false);
    private static ProcessTable sPT=new ProcessTable(false);


    // PRIVATE HELPERS.

    /**
     * Returns the request represented in input string form within the
     * tokens of the given tokenizer. The input string form starts
     * with the numeric process ID token, followed by one or more
     * pairs of tokens. Each pair represents a resource count: the
     * first part of each pair is a numeric resource ID, and the
     * second is the number of resource instances.
     *
     * @param tokenizer The tokenizer.
     *
     * @return The request.
     **/

    private static Request parseRequest(StringTokenizer tokenizer)
    {
        // Determine process.
        Process process=sPT.get(Integer.parseInt(tokenizer.nextToken()));
        // Determine resource counts.
        LinkedList rcs=new LinkedList();
        do {
            // Determine resource.
            Resource resource=sRT.get(Integer.parseInt(tokenizer.nextToken()));
            // Determine instance count.
            int instances=Integer.parseInt(tokenizer.nextToken());
            rcs.addLast(new ResourceCount(resource,instances));
        } while (tokenizer.hasMoreTokens());
        // Compose request.
        return new Request(process,(ResourceCount[])rcs.toArray
                           (new ResourceCount[rcs.size()]));
    }

    /**
     * Returns the output string form of the given resource counts.
     *
     * @param resourceCounts The resource counts.
     *
     * @return The output string form.
     **/

    private static String toString(ResourceCount[] resourceCounts)
    {
        StringBuffer buffer=new StringBuffer();
        for (int i=0;i<resourceCounts.length;i++) {
            buffer.append('(');
            buffer.append(sRT.get(resourceCounts[i].getResource()));
            buffer.append(',');
            buffer.append(resourceCounts[i].getInstances());
            buffer.append(')');
        }
        return buffer.toString();
    }

    /**
     * Returns the output string form of the given request.
     *
     * @param request The request.
     *
     * @return The output string form.
     **/

    private static String toString(Request request)
    {
        StringBuffer buffer=new StringBuffer();
        buffer.append('[');
        buffer.append(sPT.get(request.getProcess()));
        buffer.append(';');
        buffer.append(toString(request.getResourceCounts()));
        buffer.append(']');
        return buffer.toString();
    }


    // PUBLIC INTERFACE.

    /**
     * Main program.
     *
     * @param args The command-line arguments: execute without
     * arguments for usage instructions.
     *
     * @throws Exception Thrown iff an error occurs.
     **/

    public static void main(String[] args)
        throws Exception
    {

        // Error-checking.

        if (args.length<1) {
            System.err.println
                ("Usage: <allocator> ((np)|(nr:<instances>)|"+
                 "(a:<pid>(:<rid>:<instances>)+)|");
            System.err.println
                ("       (r:<pid>(:<rid>:<instances>)+)|"+
                 "(xr:<rid>)|(xp:<pid>))");
            return;
        }

        // Parse fixed command-line arguments.

        Allocator allocator=(Allocator)Class.forName(args[0]).newInstance();
        System.err.println("Allocator: "+allocator.getClass().getName());

        // Run simulation.

        for (int i=1;i<args.length;i++) {
            StringTokenizer tokenizer=new StringTokenizer(args[i],":");
            String operation=tokenizer.nextToken();
            try {
                if (operation.equals("np")) {
                    System.out.print("Creating process: ");
                    System.out.println(sPT.put(allocator.createProcess()));
                } else if (operation.equals("nr")) {
                    int instances=Integer.parseInt(tokenizer.nextToken());
                    System.out.print("Creating resource with "+instances+
                                     " instances: ");
                    System.out.println
                        (sRT.put(allocator.createResource(instances)));
                } else if (operation.equals("a")) {
                    System.out.print("Allocation request ");
                    Request request=parseRequest(tokenizer);
                    System.out.print(toString(request)+": ");
                    System.out.println
                        (allocator.allocate(request)?"granted":"enqueued");
                } else if (operation.equals("r")) {
                    System.out.print("Release ");
                    Request request=parseRequest(tokenizer);
                    System.out.print(toString(request)+": ");
                    Request[] requests=allocator.release(request);
                    System.out.println("done");
                    for (int j=0;j<requests.length;j++) {
                        System.out.println
                            (" Dequeued: "+toString(requests[j]));
                    }
                } else if (operation.equals("xp")) {
                    int pid=Integer.parseInt(tokenizer.nextToken());
                    System.out.print("Deleting process "+pid+": ");
                    Request[] requests=allocator.deleteProcess(sPT.get(pid));
                    System.out.println("done");
                    sPT.remove(pid);
                    for (int j=0;j<requests.length;j++) {
                        System.out.println
                            (" Dequeued: "+toString(requests[j]));
                    }
                } else if (operation.equals("xr")) {
                    int rid=Integer.parseInt(tokenizer.nextToken());
                    System.out.print("Deleting resource "+rid+": ");
                    allocator.deleteResource(sRT.get(rid));
                    System.out.println("done");
                    sRT.remove(rid);
                } else {
                    System.err.println("Skipping invalid operation: "+args[i]);
                }
            } catch (AllocatorException ex) {
                System.out.println(ex);
            }
        }

        // Display status.

        int[] rids=sRT.getAllIDs();
        Arrays.sort(rids);
        System.out.println("Resources:");
        for (int i=0;i<rids.length;i++) {
            System.out.println(" ID: "+rids[i]+"; Available: "+
                               sRT.get(rids[i]).getAvailable());
        }
        int[] pids=sPT.getAllIDs();
        Arrays.sort(pids);
        System.out.println("Processes:");
        for (int i=0;i<pids.length;i++) {
            System.out.println(" ID: "+pids[i]+"; Allocations: "+
                               toString(sPT.get(pids[i]).getAllocations()));
        }
        System.out.println("Allocator request queue:");
        Request[] requests=allocator.getQueue();
        for (int i=0;i<requests.length;i++) {
            System.out.println(" "+toString(requests[i]));
        }
    }
}
